<?php


namespace Mobile\Measure\Entity\Variable;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Measure\Entity\Variable\Base\Variable;
use Mobile\Project\Entity\DataEntryProject;
use Shared\Authentication\Entity\IdentifiedEntity;

/**
 * Class Material
 * @ORM\Entity()
 * @ORM\Table(name="material", schema="adonis")
 */
class Material extends IdentifiedEntity
{
    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $physicalDevice;

    /**
     * @var string | null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $manufacturer;

    /**
     * @var string | null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $type;

    /**
     * @var Driver
     *
     * @ORM\OneToOne(targetEntity="Mobile\Measure\Entity\Variable\Driver", inversedBy="material", cascade={"persist","remove", "detach"})
     */
    private $driver;

    /**
     * @var Collection<UniqueVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\UniqueVariable", mappedBy="material", cascade={"persist","remove", "detach"})
     */
    private $uniqueVariables;

    /**
     * @var Collection<GeneratorVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\GeneratorVariable", mappedBy="material", cascade={"persist","remove", "detach"})
     */
    private $generatorVariables;

    /**
     * @var DataEntryProject
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Project\Entity\DataEntryProject", inversedBy="materials")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    private $project;

    public function __construct()
    {
        $this->generatorVariables = new ArrayCollection();
        $this->uniqueVariables = new ArrayCollection();
    }

    /**
     * @param Variable $variable
     */
    public function addVariable(Variable $variable): void
    {
        if ($variable instanceof UniqueVariable) {
            if (!$this->uniqueVariables->contains($variable)) {
                $variable->setMaterial($this);
                $this->uniqueVariables->add($variable);
            }
        } else {
            if (!$this->generatorVariables->contains($variable)) {
                $variable->setMaterial($this);
                $this->generatorVariables->add($variable);
            }
        }
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Material
     */
    public function setName(string $name): Material
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getPhysicalDevice(): string
    {
        return $this->physicalDevice;
    }

    /**
     * @param string $physicalDevice
     * @return Material
     */
    public function setPhysicalDevice(string $physicalDevice): Material
    {
        $this->physicalDevice = $physicalDevice;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getManufacturer(): ?string
    {
        return $this->manufacturer;
    }

    /**
     * @param string|null $manufacturer
     * @return Material
     */
    public function setManufacturer(?string $manufacturer): Material
    {
        $this->manufacturer = $manufacturer;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getType(): ?string
    {
        return $this->type;
    }

    /**
     * @param string|null $type
     * @return Material
     */
    public function setType(?string $type): Material
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return Driver
     */
    public function getDriver(): Driver
    {
        return $this->driver;
    }

    /**
     * @param Driver $driver
     * @return Material
     */
    public function setDriver(Driver $driver): Material
    {
        $this->driver = $driver;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getUniqueVariables(): Collection
    {
        return $this->uniqueVariables;
    }

    /**
     * @param Collection $uniqueVariables
     * @return Material
     */
    public function setUniqueVariables(Collection $uniqueVariables): Material
    {
        $this->uniqueVariables = $uniqueVariables;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getGeneratorVariables(): Collection
    {
        return $this->generatorVariables;
    }

    /**
     * @param Collection $generatorVariables
     * @return Material
     */
    public function setGeneratorVariables(Collection $generatorVariables): Material
    {
        $this->generatorVariables = $generatorVariables;
        return $this;
    }

    /**
     * @return DataEntryProject
     */
    public function getProject(): DataEntryProject
    {
        return $this->project;
    }

    /**
     * @param DataEntryProject $project
     * @return Material
     */
    public function setProject(DataEntryProject $project): Material
    {
        $this->project = $project;
        return $this;
    }

}
