<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2020
 */

declare(strict_types=1);

namespace Shared\TransferSync\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Annotation\ApiSubresource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\BooleanFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\ExistsFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Mobile\Project\Entity\DataEntryProject;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\User;
use Shared\TransferSync\Controller\CancelTransferToMobileAction;
use Shared\TransferSync\Controller\TransferToWebappAction;
use Shared\TransferSync\Dto\SessionAndChangeSelectionInput;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Entity\Project;

/**
 * Class StatusDataEntry.
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={}
 *     },
 *     itemOperations={
 *         "get"={"normalization_context"={"groups"={"status_project_webapp_return_data", "id_read"}},},
 *         "patch"={
 *             "security"="object.getUser() == user",
 *             "denormalization_context"={"groups"={"status_project_patch"}},
 *         },
 *         "delete"={
 *             "security"="object.getUser() == user || (object.getWebappProject() !== null && object.getWebappProject().getOwner() == user)",
 *             "controller"=CancelTransferToMobileAction::class,
 *         },
 *         "patch_return_data"={
 *             "method"="PATCH",
 *             "path"="/status_data_entries/{id}/return-data",
 *             "controller"=TransferToWebappAction::class,
 *             "input"=SessionAndChangeSelectionInput::class,
 *             "validate"=false,
 *         },
 *     }
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={
 *     "webappProject": "exact",
 *     "webappProject.platform.site": "exact",
 *     "user": "exact",
 *     "syncable": "exact",
 *     "response.improvised": "exact"
 * })
 * @ApiFilter(GroupFilter::class, arguments={
 *     "whitelist"={
 *         "status_project_mobile_view",
 *         "status_project_webapp_view",
 *         "status_project_webapp_return_data",
 *         "id_read",
 *     }
 * })
 * @ApiFilter(BooleanFilter::class, properties={
 *     "request.benchmark"
 * })
 * @ApiFilter(ExistsFilter::class, properties={
 *     "response"
 * })
 *
 * @Gedmo\SoftDeleteable()
 * @ORM\Table(name="rel_user_project_status", schema="shared")
 * @ORM\Entity(repositoryClass="Shared\TransferSync\Repository\StatusDataEntryRepository")
 */
class StatusDataEntry extends IdentifiedEntity
{
    use SoftDeleteableEntity;
    public const STATUS_SYNCHRONIZED = 'synchronized';
    public const STATUS_NEW = 'new';
    public const STATUS_WRITE_PENDING = 'writing';
    public const STATUS_DONE = 'done';
    /**
     * @var User
     * @Groups({"status_project_webapp_view", "status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\User", inversedBy="projects")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    protected User $user;

    /**
     * @var string
     * @Groups({"status_project_webapp_view", "status_project_patch", "status_project_mobile_view"})
     * @ORM\Column(name="status", type="string", nullable=false)
     */
    protected string $status;

    /**
     * @var bool
     * @Groups({"status_project_webapp_view", "status_project_patch"})
     * @ORM\Column(name="syncable", type="boolean", nullable=false, options={"default"=false})
     */
    protected bool $syncable = false;

    /**
     * @var DataEntryProject | null
     * @Groups({"status_project_webapp_view", "status_project_mobile_view"})
     * @ORM\ManyToOne(targetEntity="Mobile\Project\Entity\DataEntryProject", cascade={"persist"})
     * @ORM\JoinColumn(nullable=true)
     */
    private ?DataEntryProject $request;

    /**
     * @var DataEntryProject | null
     * @ApiSubresource()
     * @Groups({"status_project_webapp_return_data", "user_linked_file_job_read"})
     * @ORM\OneToOne(targetEntity="Mobile\Project\Entity\DataEntryProject", cascade={"persist", "remove", "detach"})
     * @ORM\JoinColumn(nullable=true)
     */
    private ?DataEntryProject $response;

    /**
     * @var Project|null
     * @Groups({"status_project_webapp_view", "status_project_webapp_return_data", "status_project_mobile_view"})
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Project", inversedBy="statusDataEntries")
     */
    private ?Project $webappProject;

    /**
     * @var Collection<IndividualStructureChange>
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\OneToMany(targetEntity="Shared\TransferSync\Entity\IndividualStructureChange",mappedBy="statusDataEntry", cascade={"persist", "remove", "detach"}, orphanRemoval=true)
     */
    private $changes;

    /**
     * StatusDataEntry constructor.
     * @param User $user
     * @param string $status
     */
    public function __construct(User $user, string $status)
    {
        $this->user = $user;
        $this->status = $status;
        $this->changes = new ArrayCollection();
        $this->request = null;
        $this->response = null;
        $this->webappProject = null;
    }

    /**
     * @return User
     */
    public function getUser(): User
    {
        return $this->user;
    }

    /**
     * @param User $user
     * @return StatusDataEntry
     */
    public function setUser(User $user): StatusDataEntry
    {
        $this->user = $user;
        return $this;
    }

    /**
     * @return string
     */
    public function getStatus(): string
    {
        return $this->status;
    }

    /**
     * @param string $status
     * @return StatusDataEntry
     */
    public function setStatus(string $status): StatusDataEntry
    {
        $this->status = $status;
        return $this;
    }

    /**
     * @return bool
     */
    public function isSyncable(): bool
    {
        return $this->syncable;
    }

    /**
     * @param bool $syncable
     * @return StatusDataEntry
     */
    public function setSyncable(bool $syncable): StatusDataEntry
    {
        $this->syncable = $syncable;
        return $this;
    }

    /**
     * @return DataEntryProject|null
     */
    public function getRequest(): ?DataEntryProject
    {
        return $this->request;
    }

    /**
     * @param DataEntryProject|null $request
     * @return StatusDataEntry
     */
    public function setRequest(?DataEntryProject $request): StatusDataEntry
    {
        $this->request = $request;
        return $this;
    }

    /**
     * @return DataEntryProject|null
     */
    public function getResponse(): ?DataEntryProject
    {
        return $this->response;
    }

    /**
     * @param DataEntryProject|null $response
     * @return StatusDataEntry
     */
    public function setResponse(?DataEntryProject $response): StatusDataEntry
    {
        $this->response = $response;
        return $this;
    }

    /**
     * @return Project|null
     */
    public function getWebappProject(): ?Project
    {
        return $this->webappProject;
    }

    /**
     * @param Project|null $webappProject
     * @return StatusDataEntry
     */
    public function setWebappProject(?Project $webappProject): StatusDataEntry
    {
        $this->webappProject = $webappProject;
        return $this;
    }

    /**
     * @return Collection<IndividualStructureChange>
     */
    public function getChanges()
    {
        return $this->changes;
    }

    /**
     * @param IndividualStructureChange $change
     * @return StatusDataEntry
     */
    public function addChange(IndividualStructureChange $change): StatusDataEntry
    {
        if (!$this->changes->contains($change)) {
            $this->changes->add($change);
            $change->setStatusDataEntry($this);
        }
        return $this;
    }

    /**
     * @param IndividualStructureChange $change
     * @return StatusDataEntry
     */
    public function removeChange(IndividualStructureChange $change): StatusDataEntry
    {
        if ($this->changes->contains($change)) {
            $this->changes->removeElement($change);
        }
        return $this;
    }

    /**
     * @return string|null
     * @Groups({"status_project_webapp_view"})
     */
    public function getMobileProjectName()
    {
        return isset($this->response) ? $this->response->getName() : null;
    }
}
