<?php

namespace Webapp\Core\ApiOperation;

use ApiPlatform\Core\Api\IriConverterInterface;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Security\Core\Security;
use Webapp\Core\Entity\AbstractVariable;
use Webapp\Core\Entity\Annotation;
use Webapp\Core\Entity\FieldGeneration;
use Webapp\Core\Entity\FieldMeasure;
use Webapp\Core\Entity\Fusion\DataEntryFusion;
use Webapp\Core\Entity\GeneratorVariable;
use Webapp\Core\Entity\Measure;
use Webapp\Core\Entity\ProjectData;
use Webapp\Core\Entity\SemiAutomaticVariable;
use Webapp\Core\Entity\Session;
use Webapp\Core\Entity\SimpleVariable;

/**
 * Class FusionDataEntriesOperation.
 */
final class FusionDataEntriesOperation
{
    private IriConverterInterface $iriConverter;

    private EntityManagerInterface $entityManager;

    private Security $security;

    public function __construct(IriConverterInterface $iriConverter, EntityManagerInterface $entityManager, Security $security)
    {
        $this->iriConverter = $iriConverter;
        $this->entityManager = $entityManager;
        $this->security = $security;
    }

    /**
     * @param Request $request
     * @param DataEntryFusion $data
     * @return mixed
     */
    public function __invoke(Request $request, $data)
    {
        if (
            count($data->getOrderedDefaultProjectDatasIris()) === 0 ||
            count($data->getSelectedVariableIris()) === 0
        ) {
            return new BadRequestHttpException();
        }
        $res = (new ProjectData())
            ->setFusion(true)
            ->setName($data->getOrderedDefaultProjectDatasIris()[0]->getProject()->getName())
            ->setUser($this->security->getUser());
        $data->getOrderedDefaultProjectDatasIris()[0]->getProject()->addProjectData($res);
        $iriMap = [];

        $variableNamesMap = [];
        foreach ($data->getSelectedVariableIris() as $selectedVariableIri) {
            /** @var AbstractVariable $variable */
            $variable = $this->iriConverter->getItemFromIri($selectedVariableIri);
            if (isset($variableNamesMap[$variable->getName()])) {
                $variableNamesMap[$variable->getName()][] = $variable;
                $iriMap[$selectedVariableIri] = $iriMap[$this->iriConverter->getIriFromItem($variableNamesMap[$variable->getName()][0])];
            } else {
                $variableNamesMap[$variable->getName()] = [$variable];
                if ($variable instanceof SimpleVariable) {
                    $resVariable = $this->duplicateSimpleVariable($variable);
                } elseif ($variable instanceof SemiAutomaticVariable) {
                    $resVariable = $this->duplicateSemiAutomaticVariable($variable);
                } elseif ($variable instanceof GeneratorVariable) {
                    $resVariable = $this->duplicateGeneratorVariable($variable, $iriMap);
                }

                $iriMap[$selectedVariableIri] = $resVariable;
                $res->addVariable($resVariable);
            }

        }

        $conflictingVariableIris = [];
        foreach ($variableNamesMap as $variables) {
            if (count($variables) > 1) {
                foreach ($variables as $variable) {
                    $conflictingVariableIris[] = $this->iriConverter->getIriFromItem($variable);
                }
            }
        }

        $orderedIriForObjectMap = [];
        foreach ($data->getSpecificOrderForItem() as $object) {
            $orderedIriForObjectMap[$object["objectIri"]] = array_map(function ($item) {
                return $this->iriConverter->getItemFromIri($item);
            }, $object["orderedProjectDatasIris"]);
        }

        // check conflicts
        $conflicts = [];
        // map variableName + TargetIri => prio fieldMeasure
        /** @var array<FieldMeasure> $prioFieldMeasure */
        $prioFieldMeasure = [];
        foreach ($data->getOrderedDefaultProjectDatasIris() as $projectData) {
            foreach ($projectData->getSessions() as $session) {
                foreach ($session->getFieldMeasures() as $fieldMeasure) {
                    $variableName = $fieldMeasure->getVariable()->getName();
                    $targetIri = $this->iriConverter->getIriFromItem($fieldMeasure->getTarget());
                    if (in_array($this->iriConverter->getIriFromItem($fieldMeasure->getVariable()), $conflictingVariableIris)) {
                        if ($fieldMeasure->getMeasures()[0]->getState() === null || $fieldMeasure->getMeasures()[0]->getState()->getTitle() !== "Donnée Manquante") {
                            if (!isset($prioFieldMeasure[$variableName . $targetIri])) {
                                $prioFieldMeasure[$variableName . $targetIri] = $fieldMeasure;
                            } else {
                                if (
                                    (
                                        $prioFieldMeasure[$variableName . $targetIri]->getMeasures()[0]->getState() !== null ||
                                        $fieldMeasure->getMeasures()[0]->getState() !== null
                                    ) &&
                                    (
                                        $prioFieldMeasure[$variableName . $targetIri]->getMeasures()[0]->getState() === null ||
                                        $fieldMeasure->getMeasures()[0]->getState() === null ||
                                        $prioFieldMeasure[$variableName . $targetIri]->getMeasures()[0]->getState()->getCode() !== $fieldMeasure->getMeasures()[0]->getState()->getCode()
                                    )
                                ) {
                                    if (!isset($conflicts[$variableName . $targetIri])) {
                                        $conflicts[$variableName . $targetIri] = [$prioFieldMeasure[$variableName . $targetIri]];
                                    }
                                    $conflicts[$variableName . $targetIri][] = $fieldMeasure;
                                }
                            }
                        }
                    }
                }
            }
        }

        if (array_reduce($conflicts, function ($acc, $item) use ($data) {
            return $acc && count(array_uintersect($item, $data->getMergePriority(), function ($a, $b) {
                    return strcmp($this->iriConverter->getIriFromItem($a), $this->iriConverter->getIriFromItem($b));
                })) > 0;
        }, true)) {
            //target + variable name -> field measure
            $conflictingFormFieldMap = [];
            foreach ($data->getOrderedDefaultProjectDatasIris() as $projectData) {
                foreach ($projectData->getSessions() as $session) {
                    $res->addSession($this->duplicateSession($session, $iriMap, $conflictingVariableIris, $data->getOrderedDefaultProjectDatasIris(), $orderedIriForObjectMap, $conflicts, $data->getMergePriority(), $conflictingFormFieldMap));
                }
            }
            $this->entityManager->persist($res);
            $this->entityManager->flush();
        } else {
            $data->setConflicts(array_values($conflicts));
            $res = null;
        }

        return $data->setId(0)->setResult($res);
    }

    private function duplicateSession(Session $session, array &$iriMap, array $conflictingVariableIris, array $orderedDataEntriesIri, array $specificOrderForItemMap, array $conflicts, array $mergePriorities, array &$conflictingFormFieldMap): Session
    {
        $res = (new Session())
            ->setUser($session->getUser())
            ->setEndedAt($session->getEndedAt())
            ->setStartedAt($session->getStartedAt());
        foreach ($session->getFieldMeasures() as $fieldMeasure) {
            $variableIri = $this->iriConverter->getIriFromItem($fieldMeasure->getVariable());
            if (in_array($variableIri, array_keys($iriMap))) {
                if (!in_array($variableIri, $conflictingVariableIris) || in_array($fieldMeasure, $mergePriorities)) {
                    $res->addFieldMeasure($this->duplicateFieldMeasure($fieldMeasure, $iriMap));
                } else if (
                    array_reduce($conflicts, function ($acc, $item) use ($fieldMeasure) {
                        return $acc && !in_array($fieldMeasure, $item);
                    }, true) &&
                    ($fieldMeasure->getMeasures()[0]->getState() === null || $fieldMeasure->getMeasures()[0]->getState()->getTitle() !== "Donnée Manquante")
                ) {
                    /** @var ProjectData[] $order */
                    $order = $specificOrderForItemMap[$this->iriConverter->getIriFromItem($fieldMeasure->getTarget())] ?? $orderedDataEntriesIri;
                    foreach ($order as $projectData) {
                        if (count(array_filter(
                                $projectData->getVariables(),
                                function (AbstractVariable $item) use ($conflictingVariableIris, $fieldMeasure) {
                                    return $fieldMeasure->getVariable()->getName() === $item->getName() &&
                                        in_array($this->iriConverter->getIriFromItem($item), $conflictingVariableIris);
                                })) > 0) {
                            $key = $this->iriConverter->getIriFromItem($fieldMeasure->getTarget()) . $fieldMeasure->getVariable()->getName();
                            if (
                                $projectData === $session->getProjectData() ||
                                !isset($conflictingFormFieldMap[$key])
                            ) {
                                $conflictingFormFieldMap[$key] = $fieldMeasure;
                                $res->addFieldMeasure($this->duplicateFieldMeasure($fieldMeasure, $iriMap));
                            } else {
                                break;
                            }
                        }
                    }
                }
            }
        }
        foreach ($session->getAnnotations() as $annotation) {
            if (!$annotation->getTarget() instanceof Measure || isset($iriMap[$this->iriConverter->getIriFromItem($annotation->getTarget())])) {
                $res->addAnnotation($this->duplicateAnnotation($annotation, $iriMap));
            }
        }
        return $res;
    }

    private function duplicateAnnotation(Annotation $annotation, array $iriMap): Annotation
    {
        return (new Annotation())
            ->setValue($annotation->getValue())
            ->setTimestamp($annotation->getTimestamp())
            ->setType($annotation->getType())
            ->setTarget($annotation->getTarget() instanceof Measure ?
                $iriMap[$this->iriConverter->getIriFromItem($annotation->getTarget())] :
                $annotation->getTarget())
            ->setKeywords($annotation->getKeywords())
            ->setCategories($annotation->getKeywords())
            ->setName($annotation->getName())
            ->setImage($annotation->getImage());
    }

    private function duplicateSimpleVariable(SimpleVariable $variable): SimpleVariable
    {
        return (new SimpleVariable("", "", 0, "", false, ""))
            ->setType($variable->getType())
            ->setName($variable->getName())
            ->setCreated($variable->getCreated())
            ->setIdentifier($variable->getIdentifier())
            ->setOrder($variable->getOrder())
            ->setComment($variable->getComment())
            ->setDefaultTrueValue($variable->getDefaultTrueValue())
            ->setFormat($variable->getFormat())
            ->setFormatLength($variable->getFormatLength())
            ->setLastModified($variable->getLastModified())
            ->setMandatory($variable->isMandatory())
            ->setPathLevel($variable->getPathLevel())
            ->setRepetitions($variable->getRepetitions())
            ->setShortName($variable->getShortName())
            ->setUnit($variable->getUnit());
    }

    private function duplicateSemiAutomaticVariable(SemiAutomaticVariable $variable): SemiAutomaticVariable
    {
        return (new SemiAutomaticVariable("", "", 0, "", false, "", 0, 0))
            ->setType($variable->getType())
            ->setName($variable->getName())
            ->setCreated($variable->getCreated())
            ->setIdentifier($variable->getIdentifier())
            ->setStart($variable->getStart())
            ->setEnd($variable->getEnd())
            ->setDevice($variable->getDevice())
            ->setOrder($variable->getOrder())
            ->setComment($variable->getComment())
            ->setDefaultTrueValue($variable->getDefaultTrueValue())
            ->setFormat($variable->getFormat())
            ->setFormatLength($variable->getFormatLength())
            ->setLastModified($variable->getLastModified())
            ->setMandatory($variable->isMandatory())
            ->setPathLevel($variable->getPathLevel())
            ->setRepetitions($variable->getRepetitions())
            ->setShortName($variable->getShortName())
            ->setUnit($variable->getUnit());
    }

    private function duplicateGeneratorVariable(GeneratorVariable $variable, array &$iriMap): GeneratorVariable
    {
        $res = (new GeneratorVariable("", "", 0, "", false, "", false, false, false))
            ->setName($variable->getName())
            ->setCreated($variable->getCreated())
            ->setIdentifier($variable->getIdentifier())
            ->setNumeralIncrement($variable->isNumeralIncrement())
            ->setDirectCounting($variable->isDirectCounting())
            ->setGeneratedPrefix($variable->getGeneratedPrefix())
            ->setPathWayHorizontal($variable->isPathWayHorizontal())
            ->setOrder($variable->getOrder())
            ->setComment($variable->getComment())
            ->setDefaultTrueValue($variable->getDefaultTrueValue())
            ->setFormat($variable->getFormat())
            ->setFormatLength($variable->getFormatLength())
            ->setLastModified($variable->getLastModified())
            ->setMandatory($variable->isMandatory())
            ->setPathLevel($variable->getPathLevel())
            ->setRepetitions($variable->getRepetitions())
            ->setShortName($variable->getShortName())
            ->setUnit($variable->getUnit());
        foreach ($variable->getGeneratedGeneratorVariables() as $generatedGeneratorVariable) {
            $resVariable = $this->duplicateGeneratorVariable($generatedGeneratorVariable, $iriMap);
            $iriMap[$this->iriConverter->getIriFromItem($generatedGeneratorVariable)] = $resVariable;
            $res->addGeneratedGeneratorVariable($resVariable);
        }
        foreach ($variable->getGeneratedSimpleVariables() as $generatedSimpleVariable) {
            $resVariable = $this->duplicateSimpleVariable($generatedSimpleVariable);
            $iriMap[$this->iriConverter->getIriFromItem($generatedSimpleVariable)] = $resVariable;
            $res->addGeneratedSimpleVariable($resVariable);
        }
        return $res;
    }

    private function duplicateFieldMeasure(FieldMeasure $fieldMeasure, array &$iriMap): FieldMeasure
    {
        $res = (new FieldMeasure())
            ->setTarget($fieldMeasure->getTarget())
            ->setVariable($iriMap[$this->iriConverter->getIriFromItem($fieldMeasure->getVariable())]);
        foreach ($fieldMeasure->getFieldGenerations() as $fieldGeneration) {
            $res->addFieldGeneration($this->duplicateFieldGeneration($fieldGeneration, $iriMap));
        }
        foreach ($fieldMeasure->getMeasures() as $measure) {
            $resMeasure = $this->duplicateMeasure($measure);
            $iriMap[$this->iriConverter->getIriFromItem($measure)] = $resMeasure;
            $res->addMeasure($resMeasure);
        }
        return $res;
    }

    private function duplicateFieldGeneration(FieldGeneration $fieldGeneration, array &$iriMap): FieldGeneration
    {
        $res = (new FieldGeneration())
            ->setIndex($fieldGeneration->getIndex())
            ->setNumeralIncrement($fieldGeneration->getIndex())
            ->setPrefix($fieldGeneration->getPrefix());
        foreach ($fieldGeneration->getChildren() as $child) {
            $res->addChild($this->duplicateFieldMeasure($child, $iriMap));
        }
        return $res;
    }

    private function duplicateMeasure(Measure $measure): Measure
    {
        return (new Measure())
            ->setState($measure->getState())
            ->setTimestamp($measure->getTimestamp())
            ->setValue($measure->getValue())
            ->setRepetition($measure->getRepetition());
    }

}
