<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Annotation\ApiSubresource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Annotation\GraphicallyDeletable;
use Webapp\Core\Traits\GraphicallyDeletableEntity;

/**
 * Class UnitPlot
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *          "get"={}
 *     },
 *     itemOperations={
 *          "get"={},
 *          "patch"={"denormalization_context"={"groups"={"declare_dead", "edit"}}}
 *     }
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={"block": "exact", "subBlock": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"design_explorer_view", "platform_full_view", "parent_view"}})
 *
 * @Gedmo\SoftDeleteable()
 * @GraphicallyDeletable()
 *
 * @ORM\Entity()
 * @ORM\Table(name="surfacic_unit_plot", schema="webapp")
 */
class SurfacicUnitPlot extends IdentifiedEntity implements BusinessObject
{
    use GraphicallyDeletableEntity;

    use SoftDeleteableEntity;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "change_report", "fusion_result"})
     */
    private string $number;

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "change_report"})
     */
    private int $x;

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "change_report"})
     */
    private int $y;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "declare_dead"})
     */
    private bool $dead;

    /**
     * @var DateTime $appeared
     *
     * @Gedmo\Timestampable(on="create")
     * @ORM\Column(type="datetime")
     * @Groups({"platform_full_view", "webapp_data_view"})
     */
    private DateTime $appeared;

    /**
     * @var DateTime|null $disappeared
     *
     * @Gedmo\Timestampable(on="change", field="dead", value=true)
     * @ORM\Column(type="datetime", nullable=true)
     * @Groups({"platform_full_view", "webapp_data_view"})
     */
    private ?DateTime $disappeared;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view"})
     */
    private ?string $identifier;

    /**
     * @var float|null
     *
     * @ORM\Column(type="float", nullable=true)
     * @Groups({"design_explorer_view", "platform_full_view"})
     */
    private ?float $latitude;

    /**
     * @var float|null
     *
     * @ORM\Column(type="float", nullable=true)
     * @Groups({"design_explorer_view", "platform_full_view"})
     */
    private ?float $longitude;

    /**
     * @var float|null
     *
     * @ORM\Column(type="float", nullable=true)
     * @Groups({"design_explorer_view", "platform_full_view"})
     */
    private ?float $height;

    /**
     * @var Treatment
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Treatment")
     * @ORM\JoinColumn(nullable=false)
     * @Groups({"design_explorer_view", "webapp_data_view", "platform_full_view", "change_report", "edit"})
     */
    private Treatment $treatment;

    /**
     * @var ?Block
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Block", inversedBy="surfacicUnitPlots")
     * @Groups({"webapp_data_view", "parent_view", "change_report"})
     */
    private $block;

    /**
     * @var ?SubBlock
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SubBlock", inversedBy="surfacicUnitPlots")
     * @Groups({"webapp_data_view", "parent_view", "change_report"})
     */
    private $subBlock;

    /**
     * @var Collection|array<Note>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Note", mappedBy="surfacicUnitPlotTarget", cascade={"persist", "remove"}, orphanRemoval=true)
     * @Groups({"note_view"})
     * @ApiSubresource()
     */
    private $notes;

    /**
     * @var int|null
     * @ORM\Column(type="integer", nullable=true)
     * @Groups({"platform_full_view"})
     */
    private ?int $color;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"platform_full_view", "edit"})
     */
    private ?string $comment;

    public function __construct()
    {
        $this->dead = false;
        $this->notes = new ArrayCollection();
        $this->disappeared = null;
        $this->identifier = null;
        $this->latitude = null;
        $this->longitude = null;
        $this->height = null;
        $this->color = null;
        $this->comment = null;
    }

    /**
     * @Groups({"platform_full_view"})
     * @return mixed
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getNumber(): string
    {
        return $this->number;
    }

    /**
     * @param string $number
     * @return SurfacicUnitPlot
     */
    public function setNumber(string $number): SurfacicUnitPlot
    {
        $this->number = $number;
        return $this;
    }

    /**
     * @return int
     */
    public function getX(): int
    {
        return $this->x;
    }

    /**
     * @param int $x
     * @return SurfacicUnitPlot
     */
    public function setX(int $x): SurfacicUnitPlot
    {
        $this->x = $x;
        return $this;
    }

    /**
     * @return int
     */
    public function getY(): int
    {
        return $this->y;
    }

    /**
     * @param int $y
     * @return SurfacicUnitPlot
     */
    public function setY(int $y): SurfacicUnitPlot
    {
        $this->y = $y;
        return $this;
    }

    /**
     * @return bool
     */
    public function isDead(): bool
    {
        return $this->dead;
    }

    /**
     * @param bool $dead
     * @return SurfacicUnitPlot
     */
    public function setDead(bool $dead): SurfacicUnitPlot
    {
        $this->dead = $dead;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getAppeared(): DateTime
    {
        return $this->appeared;
    }

    /**
     * @param DateTime $appeared
     * @return SurfacicUnitPlot
     */
    public function setAppeared(DateTime $appeared): SurfacicUnitPlot
    {
        $this->appeared = $appeared;
        return $this;
    }

    /**
     * @return DateTime|null
     */
    public function getDisappeared(): ?DateTime
    {
        return $this->disappeared;
    }

    /**
     * @param DateTime|null $disappeared
     * @return SurfacicUnitPlot
     */
    public function setDisappeared(?DateTime $disappeared): SurfacicUnitPlot
    {
        $this->disappeared = $disappeared;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getIdentifier(): ?string
    {
        return $this->identifier;
    }

    /**
     * @param string|null $identifier
     * @return SurfacicUnitPlot
     */
    public function setIdentifier(?string $identifier): SurfacicUnitPlot
    {
        $this->identifier = $identifier;
        return $this;
    }

    /**
     * @return float|null
     */
    public function getLatitude(): ?float
    {
        return $this->latitude;
    }

    /**
     * @param float|null $latitude
     * @return SurfacicUnitPlot
     */
    public function setLatitude(?float $latitude): SurfacicUnitPlot
    {
        $this->latitude = $latitude;
        return $this;
    }

    /**
     * @return float|null
     */
    public function getLongitude(): ?float
    {
        return $this->longitude;
    }

    /**
     * @param float|null $longitude
     * @return SurfacicUnitPlot
     */
    public function setLongitude(?float $longitude): SurfacicUnitPlot
    {
        $this->longitude = $longitude;
        return $this;
    }

    /**
     * @return float|null
     */
    public function getHeight(): ?float
    {
        return $this->height;
    }

    /**
     * @param float|null $height
     * @return SurfacicUnitPlot
     */
    public function setHeight(?float $height): SurfacicUnitPlot
    {
        $this->height = $height;
        return $this;
    }

    /**
     * @return Treatment
     */
    public function getTreatment(): Treatment
    {
        return $this->treatment;
    }

    /**
     * @param Treatment $treatment
     * @return SurfacicUnitPlot
     */
    public function setTreatment(Treatment $treatment): SurfacicUnitPlot
    {
        $this->treatment = $treatment;
        return $this;
    }

    /**
     * @return Block|null
     */
    public function getBlock(): ?Block
    {
        return $this->block;
    }

    /**
     * @param Block|null $block
     * @return SurfacicUnitPlot
     */
    public function setBlock(?Block $block): SurfacicUnitPlot
    {
        $this->block = $block;
        return $this;
    }

    /**
     * @return SubBlock|null
     */
    public function getSubBlock(): ?SubBlock
    {
        return $this->subBlock;
    }

    /**
     * @param SubBlock|null $subBlock
     * @return SurfacicUnitPlot
     */
    public function setSubBlock(?SubBlock $subBlock): SurfacicUnitPlot
    {
        $this->subBlock = $subBlock;
        return $this;
    }

    /**
     * @return Collection|Note[]
     */
    public function getNotes()
    {
        return $this->notes;
    }

    /**
     * @param Collection|Note[] $notes
     * @return SurfacicUnitPlot
     */
    public function setNotes($notes)
    {
        $this->notes = $notes;
        return $this;
    }

    /**
     * @param Note $note
     * @return SurfacicUnitPlot
     */
    public function addNote($note)
    {
        if (!$this->notes->contains($note)) {
            $this->notes->add($note);
            $note->setTarget($this);
        }
        return $this;
    }

    /**
     * @param Note $note
     * @return SurfacicUnitPlot
     */
    public function removeNote($note)
    {
        if ($this->notes->contains($note)) {
            $this->notes->removeElement($note);
            $note->setTarget(null);
        }
        return $this;
    }

    /**
     * @return int|null
     */
    public function getColor(): ?int
    {
        return $this->color;
    }

    /**
     * @param int|null $color
     * @return SurfacicUnitPlot
     */
    public function setColor(?int $color): SurfacicUnitPlot
    {
        $this->color = $color;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return SurfacicUnitPlot
     */
    public function setComment(?string $comment): SurfacicUnitPlot
    {
        $this->comment = $comment;
        return $this;
    }

    public function setDeletedAt(DateTime $deletedAt = null)
    {
        $this->deletedAt = $deletedAt;
        if($deletedAt === null){
            foreach ($this->getNotes() as $child){
                $child->setDeletedAt($deletedAt);
            }
        }

        return $this;
    }

    function parent(): ?BusinessObject
    {
        return $this->block ?? $this->subBlock;
    }

    function children(): array
    {
        return [];
    }


}
