<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2020
 */

declare(strict_types=1);

namespace Mobile\Device\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Project\Entity\ProjectObject;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class BusinessObject.
 *
 * @ORM\Entity()
 * @ORM\Table("business_object", schema="adonis")
 * @ORM\InheritanceType(value="JOINED")
 * @ORM\DiscriminatorColumn("discr", type="string")
 * @ORM\DiscriminatorMap({
 *     "Device" = "Mobile\Device\Entity\Device",
 *     "Block" = "Mobile\Device\Entity\Block",
 *     "SubBlock" = "Mobile\Device\Entity\SubBlock",
 *     "UnitParcel" = "Mobile\Device\Entity\UnitParcel",
 *     "Individual" = "Mobile\Device\Entity\Individual"
 * })
 */
abstract class BusinessObject extends ProjectObject
{
    /**
     * @var string
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\Column(type="string", nullable=false)
     */
    private $name;

    /**
     * @var Collection<EntryNote>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Device\Entity\EntryNote", mappedBy="businessObject", cascade={"persist", "remove", "detach"})
     */
    private $notes;

    /**
     * @var Anomaly | null
     *
     * @ORM\OneToOne(targetEntity="Mobile\Device\Entity\Anomaly", inversedBy="businessObject", cascade={"persist", "remove", "detach"})
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    private $anomaly;

    /**
     * BusinessObject constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->notes = new ArrayCollection();
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return BusinessObject
     */
    public function setName(string $name): BusinessObject
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getNotes(): Collection
    {
        return $this->notes;
    }

    /**
     * @param EntryNote $note
     */
    public function addNote(EntryNote $note): void
    {
        if (!$this->notes->contains($note)) {
            $note->setBusinessObject($this);
            $this->notes->add($note);
        }
    }

    /**
     * @param EntryNote $note
     */
    public function removeNote(EntryNote $note): void
    {
        if ($this->notes->contains($note)) {
            $note->setBusinessObject(null);
            $this->notes->remove($note);
        }
    }

    /**
     * @return Anomaly|null
     */
    public function getAnomaly(): ?Anomaly
    {
        return $this->anomaly;
    }

    /**
     * @param Anomaly|null $anomaly
     * @return BusinessObject
     */
    public function setAnomaly(?Anomaly $anomaly): BusinessObject
    {
        $this->anomaly = $anomaly;
        return $this;
    }

    public abstract function parent(): ?BusinessObject;

}
