<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2020
 */

declare(strict_types=1);

namespace Mobile\Measure\Entity;

use Datetime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Project\Entity\DataEntryProject;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class DataEntry.
 *
 * @ORM\Entity(repositoryClass="Mobile\Measure\Repository\DataEntryRepository")
 * @ORM\Table(name="data_entry", schema="adonis")
 */
class DataEntry extends IdentifiedEntity
{
    const STATUS_NEW = 'new';
    const STATUS_WIP = 'wip';
    const STATUS_DONE = 'done';

    /**
     * @var DataEntryProject
     *
     * @ORM\OneToOne(targetEntity="Mobile\Project\Entity\DataEntryProject", inversedBy="dataEntry")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    private $project;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $status;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     */
    private $startedAt;

    /**
     * @var DateTime|null
     *
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $endedAt;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $currentPosition;

    /**
     * @var string[]|null
     *
     * @ORM\Column(type="json", nullable=true)
     */
    private $workpath;

    /**
     * @var Collection<Session>
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Session", mappedBy="dataEntry", cascade={"persist", "remove", "detach"})
     */
    private $sessions;

    /**
     * DataEntry constructor.
     */
    public function __construct()
    {
        $this->startedAt = date_create();
        $this->status = static::STATUS_NEW;
        $this->sessions = new ArrayCollection();
    }

    /**
     * @return DataEntryProject
     */
    public function getProject(): DataEntryProject
    {
        return $this->project;
    }

    /**
     * @param DataEntryProject $project
     * @return DataEntry
     */
    public function setProject(DataEntryProject $project): DataEntry
    {
        $this->project = $project;
        return $this;
    }

    /**
     * @return string
     */
    public function getStatus(): string
    {
        return $this->status;
    }

    /**
     * @param string $status
     * @return DataEntry
     */
    public function setStatus(string $status): DataEntry
    {
        $this->status = $status;
        return $this;
    }

    /**
     * @return Datetime
     */
    public function getStartedAt(): Datetime
    {
        return $this->startedAt;
    }

    /**
     * @param Datetime $startedAt
     * @return DataEntry
     */
    public function setStartedAt(Datetime $startedAt): DataEntry
    {
        $this->startedAt = $startedAt;
        return $this;
    }

    /**
     * @return Datetime|null
     */
    public function getEndedAt(): ?Datetime
    {
        return $this->endedAt;
    }

    /**
     * @param Datetime|null $endedAt
     * @return DataEntry
     */
    public function setEndedAt(?Datetime $endedAt): DataEntry
    {
        $this->endedAt = $endedAt;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getCurrentPosition(): ?string
    {
        return $this->currentPosition;
    }

    /**
     * @param string|null $currentPosition
     * @return DataEntry
     */
    public function setCurrentPosition(?string $currentPosition): DataEntry
    {
        $this->currentPosition = $currentPosition;
        return $this;
    }

    /**
     * @return string[]|null
     */
    public function getWorkpath(): ?array
    {
        return $this->workpath;
    }

    /**
     * @param string[]|null $workpath
     * @return DataEntry
     */
    public function setWorkpath(?array $workpath): DataEntry
    {
        $this->workpath = $workpath;
        return $this;
    }

    /**
     * @return Collection<Session>
     */
    public function getSessions(): Collection
    {
        return $this->sessions;
    }

    /**
     * @param Session $session
     */
    public function addSession(Session $session): void
    {
        if (!$this->sessions->contains($session)) {
            $this->sessions->add($session);
            $session->setDataEntry($this);
        }
    }

    /**
     * @param Session $session
     */
    public function removeSession(Session $session): void
    {
        if ($this->sessions->contains($session)) {
            $this->sessions->removeElement($session);
        }
    }
}
