<?php

namespace Mobile\Project\Entity;

use ApiPlatform\Core\Annotation\ApiProperty;
use ApiPlatform\Core\Annotation\ApiResource;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Device\Entity\Block;
use Mobile\Device\Entity\Device;
use Mobile\Device\Entity\Individual;
use Mobile\Device\Entity\SubBlock;
use Mobile\Device\Entity\UnitParcel;
use Mobile\Measure\Entity\Annotation;
use Mobile\Measure\Entity\DataEntry;
use Mobile\Measure\Entity\Variable\Base\Variable;
use Mobile\Measure\Entity\Variable\GeneratorVariable;
use Mobile\Measure\Entity\Variable\Material;
use Mobile\Measure\Entity\Variable\RequiredAnnotation;
use Mobile\Measure\Entity\Variable\StateCode;
use Mobile\Measure\Entity\Variable\UniqueVariable;
use Mobile\Project\Dto\DataEntryProjectInput;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\FileManagement\Entity\RequestFile;
use Webapp\FileManagement\Entity\ResponseFile;

/**
 * Class DataEntryProject.
 *
 * @ApiResource(
 *     collectionOperations={
 *         "post"={
 *              "security"="is_granted('ROLE_USER')",
 *              "input"=DataEntryProjectInput::class,
 *              "output"=false,
 *          }
 *     },
 *     itemOperations={
 *         "get"={
 *             "security"="is_granted('ROLE_USER')",
 *             "output"=DataEntryProjectInput::class,
 *          }
 *     }
 * )
 *
 * @ORM\Table(name="project", schema="adonis")
 * @ORM\Entity(repositoryClass="Mobile\Project\Repository\DataEntryProjectRepository")
 */
class DataEntryProject extends IdentifiedEntity
{
    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    protected $name;

    /**
     * @var DesktopUser
     *
     * @ORM\OneToOne(targetEntity="Mobile\Project\Entity\DesktopUser", cascade={"persist", "remove", "detach"})
     */
    protected $creator;

    /**
     * @var Collection<DesktopUser>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Project\Entity\DesktopUser", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $desktopUsers;

    /**
     * @var Platform
     *
     * @ORM\OneToOne(targetEntity="Mobile\Project\Entity\Platform", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $platform;

    /**
     * @var Collection<StateCode>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\StateCode",mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $stateCodes;

    /**
     * @var Collection<UniqueVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\UniqueVariable", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $uniqueVariables;

    /**
     * @var Collection<GeneratorVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\GeneratorVariable", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $generatorVariables;

    /**
     * @var GraphicalStructure
     *
     * @ORM\OneToOne(targetEntity="Mobile\Project\Entity\GraphicalStructure", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $graphicalStructure;

    /**
     * @var Collection<NatureZHE>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Project\Entity\NatureZHE", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $naturesZHE;

    /**
     * @var DataEntry | null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\OneToOne(targetEntity="Mobile\Measure\Entity\DataEntry", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $dataEntry;

    /**
     * @var RequestFile | null
     *
     * @ORM\OneToOne(targetEntity="Webapp\FileManagement\Entity\RequestFile", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $originFile;

    /**
     * @var ResponseFile | null
     *
     * @Groups({"user_linked_file_job_read"})
     * @ORM\OneToOne(targetEntity="Webapp\FileManagement\Entity\ResponseFile", mappedBy="project", cascade={"persist", "remove", "detach"})
     */
    protected $responseFile;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime",nullable=false)
     */
    protected $creationDate;

    /**
     * @var Collection<Material>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\Material", mappedBy="project", cascade={"persist","remove", "detach"})
     */
    protected $materials;

    /**
     * @var Collection<Workpath>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Project\Entity\Workpath", mappedBy="dataEntryProject", cascade={"persist", "remove", "detach"})
     */
    private $workpaths;

    /**
     * @var Collection<UniqueVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\UniqueVariable", mappedBy="connectedDataEntryProject", cascade={"persist", "remove", "detach"})
     */
    private $connectedUniqueVariables;

    /**
     * @var Collection<GeneratorVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\GeneratorVariable", mappedBy="connectedDataEntryProject", cascade={"persist", "remove", "detach"})
     */
    private $connectedGeneratorVariables;

    /**
     * @var Collection<RequiredAnnotation>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\RequiredAnnotation",mappedBy="dataEntryProject", cascade={"persist", "remove", "detach"})
     */
    private $requiredAnnotations;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", nullable=false, options={"default"=false})
     */
    private $improvised;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", nullable=false, options={"default"=false})
     */
    private $benchmark;

    /**
     * DataEntryProject constructor.
     */
    public function __construct()
    {
        $this->desktopUsers = new ArrayCollection();
        $this->stateCodes = new ArrayCollection();
        $this->uniqueVariables = new ArrayCollection();
        $this->generatorVariables = new ArrayCollection();
        $this->naturesZHE = new ArrayCollection();
        $this->materials = new ArrayCollection();
        $this->workpaths = new ArrayCollection();
        $this->connectedUniqueVariables = new ArrayCollection();
        $this->connectedGeneratorVariables = new ArrayCollection();
        $this->requiredAnnotations = new ArrayCollection();
        $this->benchmark = false;
    }

    /**
     * @ApiProperty(readable=false, writable=false)
     * @return array<Annotation>
     */
    public function getAnnotations()
    {
        $res = [];
        $parcelCallback = function ($unitPlotList) use (&$res) {
            /** @var UnitParcel $unitPlot */
            foreach ($unitPlotList as $unitPlot) {
                $res = array_merge($res, $unitPlot->getAnnotations()->toArray());
                /** @var Individual $individual */
                foreach ($unitPlot->getIndividuals() as $individual) {
                    $res = array_merge($res, $individual->getAnnotations()->toArray());
                }
            }
        };
        /** @var Device $device */
        foreach ($this->getPlatform()->getDevices() as $device) {
            $res = array_merge($res, $device->getAnnotations()->toArray());
            /** @var Block $block */
            foreach ($device->getBlocks() as $block) {
                $res = array_merge($res, $block->getAnnotations()->toArray());
                /** @var SubBlock $subBlock */
                foreach ($block->getSubBlocks() as $subBlock) {
                    $res = array_merge($res, $subBlock->getAnnotations()->toArray());
                    $parcelCallback($subBlock->getUnitParcels());
                }
                $parcelCallback($block->getUnitParcels());
            }
        }
        return $res;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return DataEntryProject
     */
    public function setName(string $name): DataEntryProject
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return DesktopUser|null
     */
    public function getCreator(): ?DesktopUser
    {
        return $this->creator;
    }

    /**
     * @param DesktopUser $creator
     * @return DataEntryProject
     */
    public function setCreator(DesktopUser $creator): DataEntryProject
    {
        $this->creator = $creator;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getDesktopUsers(): Collection
    {
        return $this->desktopUsers;
    }

    /**
     * @param DesktopUser $desktopUser
     */
    public function addDesktopUser(DesktopUser $desktopUser): void
    {
        if (!$this->desktopUsers->contains($desktopUser)) {
            $desktopUser->setProject($this);
            $this->desktopUsers->add($desktopUser);
        }
    }

    /**
     * @param DesktopUser $desktopUser
     */
    public function removeDesktopUser(DesktopUser $desktopUser): void
    {
        if ($this->desktopUsers->contains($desktopUser)) {
            $desktopUser->setProject(null);
            $this->desktopUsers->removeElement($desktopUser);
        }
    }

    /**
     * @return Platform
     */
    public function getPlatform(): Platform
    {
        return $this->platform;
    }

    /**
     * @param Platform $platform
     * @return DataEntryProject
     */
    public function setPlatform(Platform $platform): DataEntryProject
    {
        $this->platform = $platform;
        $platform->setProject($this);
        return $this;
    }

    /**
     * @return Collection
     */
    public function getStateCodes(): Collection
    {
        return $this->stateCodes;
    }

    /**
     * @param StateCode $stateCode
     */
    public function addStateCode(StateCode $stateCode): void
    {
        if (!$this->stateCodes->contains($stateCode)) {
            $stateCode->setProject($this);
            $this->stateCodes->add($stateCode);
        }
    }

    /**
     * @param StateCode $stateCode
     */
    public function removeStateCode(StateCode $stateCode): void
    {
        if ($this->stateCodes->contains($stateCode)) {
            $this->stateCodes->removeElement($stateCode);
        }
    }

    /**
     * @return Collection<UniqueVariable>
     */
    public function getUniqueVariables(): Collection
    {
        return $this->uniqueVariables;
    }

    /**
     * @param UniqueVariable $variable
     */
    public function removeUniqueVariable(UniqueVariable $variable): void
    {
        if ($this->uniqueVariables->contains($variable)) {
            $this->uniqueVariables->removeElement($variable);
        }
    }

    /**
     * @return Collection<GeneratorVariable>
     */
    public function getGeneratorVariables(): Collection
    {
        return $this->generatorVariables;
    }

    /**
     * @param GeneratorVariable $variable
     */
    public function removeGeneratorVariable(GeneratorVariable $variable): void
    {
        if ($this->generatorVariables->contains($variable)) {
            $this->generatorVariables->removeElement($variable);
        }
    }

    /**
     * @param Variable $variable
     */
    public function addVariable(Variable $variable)
    {
        if ($variable instanceof UniqueVariable) {
            $this->addUniqueVariable($variable);
        } else if ($variable instanceof GeneratorVariable) {
            $this->addGeneratorVariable($variable);
        }
    }

    /**
     * @param UniqueVariable $variable
     */
    public function addUniqueVariable(UniqueVariable $variable): void
    {
        if (!$this->uniqueVariables->contains($variable)) {
            $variable->setProject($this);
            $this->uniqueVariables->add($variable);
        }
    }

    /**
     * @param GeneratorVariable $variable
     */
    public function addGeneratorVariable(GeneratorVariable $variable): void
    {
        if (!$this->generatorVariables->contains($variable)) {
            $variable->setProject($this);
            $this->generatorVariables->add($variable);
        }
    }

    /**
     * @return GraphicalStructure
     */
    public function getGraphicalStructure(): GraphicalStructure
    {
        return $this->graphicalStructure;
    }

    /**
     * @param GraphicalStructure $graphicalStructure
     * @return DataEntryProject
     */
    public function setGraphicalStructure(GraphicalStructure $graphicalStructure): DataEntryProject
    {
        $graphicalStructure->setProject($this);
        $this->graphicalStructure = $graphicalStructure;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getNaturesZHE(): Collection
    {
        return $this->naturesZHE;
    }

    /**
     * @param NatureZHE $natureZhe
     */
    public function addNatureZHE(NatureZHE $natureZhe): void
    {
        if (!$this->naturesZHE->contains($natureZhe)) {
            $natureZhe->setProject($this);
            $this->naturesZHE->add($natureZhe);
        }
    }

    /**
     * @param NatureZHE $natureZhe
     */
    public function removeNatureZHE(NatureZHE $natureZhe): void
    {
        if ($this->naturesZHE->contains($natureZhe)) {
            $natureZhe->setProject(null);
            $this->naturesZHE->remove($natureZhe);
        }
    }

    /**
     * @return DataEntry | null
     */
    public function getDataEntry(): ?DataEntry
    {
        return $this->dataEntry;
    }

    /**
     * @param DataEntry | null $dataEntry
     * @return DataEntryProject
     */
    public function setDataEntry(?DataEntry $dataEntry): DataEntryProject
    {
        $this->dataEntry = $dataEntry;
        $dataEntry->setProject($this);
        return $this;
    }

    /**
     * @return RequestFile | null
     */
    public function getOriginFile(): ?RequestFile
    {
        return $this->originFile;
    }

    /**
     * @param RequestFile | null $originFile
     * @return DataEntryProject
     */
    public function setOriginFile(?RequestFile $originFile): DataEntryProject
    {
        $this->originFile = $originFile;
        if (!is_null($originFile)) {
            $originFile->setProject($this);
        }
        return $this;
    }

    /**
     * @return ResponseFile | null
     */
    public function getResponseFile(): ?ResponseFile
    {
        return $this->responseFile;
    }

    /**
     * @param ResponseFile | null $responseFile
     * @return DataEntryProject
     */
    public function setResponseFile(?ResponseFile $responseFile): DataEntryProject
    {
        $this->responseFile = $responseFile;
        if (!is_null($responseFile)) {
            $responseFile->setProject($this);
        }
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getCreationDate(): DateTime
    {
        return $this->creationDate;
    }

    /**
     * @param DateTime $creationDate
     * @return DataEntryProject
     */
    public function setCreationDate(DateTime $creationDate): DataEntryProject
    {
        $this->creationDate = $creationDate;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getMaterials(): Collection
    {
        return $this->materials;
    }

    /**
     * @param Collection $materials
     * @return DataEntryProject
     */
    public function setMaterials(Collection $materials): DataEntryProject
    {
        $this->materials = $materials;
        return $this;
    }

    /**
     * @param Material $material
     */
    public function addMaterial(Material $material): void
    {
        if (!$this->materials->contains($material)) {
            $material->setProject($this);
            $this->materials->add($material);
        }
    }

    /**
     * @param Material $material
     */
    public function removeMaterial(Material $material): void
    {
        if ($this->materials->contains($material)) {
            $material->setProject(null);
            $this->materials->remove($material);
        }
    }

    /**
     * @return Collection<Workpath>
     */
    public function getWorkpaths(): Collection
    {
        return $this->workpaths;
    }

    /**
     * @param Workpath $workpath
     */
    public function addWorkpath(Workpath $workpath): void
    {
        if (!$this->workpaths->contains($workpath)) {
            $workpath->setDataEntryProject($this);
            $this->workpaths->add($workpath);
        }
    }

    /**
     * @param Workpath $workpath
     */
    public function removeWorkpath(Workpath $workpath): void
    {
        if ($this->workpaths->contains($workpath)) {
            $this->workpaths->removeElement($workpath);
        }
    }

    /**
     * @return Collection
     */
    public function getConnectedUniqueVariables(): Collection
    {
        return $this->connectedUniqueVariables;
    }

    /**
     * @param UniqueVariable $variable
     */
    public function removeConnectedUniqueVariable(UniqueVariable $variable): void
    {
        if ($this->connectedUniqueVariables->contains($variable)) {
            $variable->setConnectedDataEntryProject(null);
            $this->connectedUniqueVariables->remove($variable);
        }
    }

    /**
     * @return Collection
     */
    public function getConnectedGeneratorVariables(): Collection
    {
        return $this->connectedGeneratorVariables;
    }

    /**
     * @param GeneratorVariable $variable
     */
    public function removeConnectedGeneratorVariable(GeneratorVariable $variable): void
    {
        if ($this->connectedGeneratorVariables->contains($variable)) {
            $variable->setConnectedDataEntryProject(null);
            $this->connectedGeneratorVariables->remove($variable);
        }
    }

    /**
     * @param Variable $variable
     */
    public function addConnectedVariable(Variable $variable)
    {
        if ($variable instanceof UniqueVariable) {
            $this->addConnectedUniqueVariable($variable);
        } else if ($variable instanceof GeneratorVariable) {
            $this->addConnectedGeneratorVariable($variable);
        }
    }

    /**
     * @param UniqueVariable $variable
     */
    public function addConnectedUniqueVariable(UniqueVariable $variable): void
    {
        if (!$this->connectedUniqueVariables->contains($variable)) {
            $variable->setConnectedDataEntryProject($this);
            $this->connectedUniqueVariables->add($variable);
        }
    }

    /**
     * @param GeneratorVariable $variable
     */
    public function addConnectedGeneratorVariable(GeneratorVariable $variable): void
    {
        if (!$this->connectedGeneratorVariables->contains($variable)) {
            $variable->setConnectedDataEntryProject($this);
            $this->connectedGeneratorVariables->add($variable);
        }
    }

    /**
     * @return Collection
     */
    public function getRequiredAnnotations(): Collection
    {
        return $this->requiredAnnotations;
    }

    /**
     * @param RequiredAnnotation $requiredAnnotation
     */
    public function addRequiredAnnotation(RequiredAnnotation $requiredAnnotation): void
    {
        if (!$this->requiredAnnotations->contains($requiredAnnotation)) {
            $requiredAnnotation->setDataEntryProject($this);
            $this->requiredAnnotations->add($requiredAnnotation);
        }
    }

    /**
     * @param RequiredAnnotation $requiredAnnotation
     */
    public function removeRequiredAnnotation(RequiredAnnotation $requiredAnnotation): void
    {
        if ($this->requiredAnnotations->contains($requiredAnnotation)) {
            $requiredAnnotation->setDataEntryProject(null);
            $this->requiredAnnotations->remove($requiredAnnotation);
        }
    }

    /**
     * @return bool
     */
    public function isImprovised(): bool
    {
        return $this->improvised;
    }

    /**
     * @param bool $improvised
     * @return DataEntryProject
     */
    public function setImprovised(bool $improvised): DataEntryProject
    {
        $this->improvised = $improvised;
        return $this;
    }

    /**
     * @return bool
     */
    public function isBenchmark(): bool
    {
        return $this->benchmark;
    }

    /**
     * @param bool $benchmark
     * @return DataEntryProject
     */
    public function setBenchmark(bool $benchmark): DataEntryProject
    {
        $this->benchmark = $benchmark;
        return $this;
    }

}
