<?php

namespace Webapp\Core\Entity;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Enumeration\Annotation\EnumType;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Enumeration\PathLevelEnum;
use Webapp\Core\Enumeration\VariableFormatEnum;

/**
 * Class AbstractVariable
 * @package Webapp\Core\Entity
 *
 * @ORM\MappedSuperclass()
 *
 * @psalm-import-type PathLevelEnumId from PathLevelEnum
 * @psalm-import-type VariableFormatEnumId from VariableFormatEnum
 */
abstract class AbstractVariable extends IdentifiedEntity
{
    protected string $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post", "webapp_data_view", "data_entry_synthesis", "project_synthesis", "variable_synthesis", "webapp_data_fusion"})
     */
    protected string $shortName;

    /**
     * @var integer
     *
     * @ORM\Column(type="integer", nullable=false)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post", "webapp_data_view", "data_entry_synthesis", "project_synthesis", "variable_synthesis"})
     */
    protected int $repetitions;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post", "webapp_data_view", "data_entry_synthesis", "project_synthesis", "variable_synthesis"})
     */
    protected ?string $unit;

    /**
     * @var string
     * @psalm-var PathLevelEnumId|''
     *
     * @ORM\Column(type="string", nullable=false)
     * @EnumType(class="Webapp\Core\Enumeration\PathLevelEnum")
     * @Groups({"generator_variable_post", "semi_automatic_variable", "webapp_data_view", "simple_variable_get", "simple_variable_post", "connected_variables", "data_entry_synthesis", "project_synthesis", "variable_synthesis", "webapp_data_path", "webapp_data_fusion"})
     */
    protected string $pathLevel = '';

    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post", "variable_synthesis"})
     */
    protected ?string $comment;

    /**
     * @var integer|null
     *
     * @ORM\Column(type="integer", name="print_order", nullable=true)
     * @Groups({"project_explorer_view", "generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post", "project_synthesis"})
     */
    protected ?int $order;

    /**
     * @var string|null
     * @psalm-format VariableFormatEnumId|numeric-string|null
     *
     * @ORM\Column(type="string", nullable=true)
     * @EnumType(class="Webapp\Core\Enumeration\VariableFormatEnum", nullable=true)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post", "webapp_data_view", "data_view_item", "graphical_measure_view"})
     */
    protected ?string $format;

    /**
     * @var int|null
     *
     * @ORM\Column(type="integer", nullable=true)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post"})
     */
    protected ?int $formatLength;

    /**
     * @var bool|null
     *
     * @ORM\Column(type="boolean", nullable=true)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post"})
     */
    protected ?bool $defaultTrueValue;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post", "webapp_data_view"})
     */
    protected ?string $identifier;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", nullable=false)
     * @Groups({"generator_variable_post", "semi_automatic_variable", "simple_variable_get", "simple_variable_post"})
     */
    protected bool $mandatory;

    /**
     * @var Collection<array-key, Test>
     */
    protected $tests;

    /**
     * @var DateTime
     * @Groups({"simple_variable_get"})
     * @Gedmo\Timestampable(on="create")
     * @ORM\Column(type="datetime")
     */
    protected DateTime $created;

    /**
     * @var DateTime
     * @Groups({"simple_variable_get"})
     * @Gedmo\Timestampable()
     * @ORM\Column(type="datetime")
     */
    protected DateTime $lastModified;

    /**
     * @var Collection<array-key, VariableConnection>
     */
    protected $connectedVariables;

    public function __construct()
    {
        $this->unit = null;
        $this->comment = null;
        $this->format = null;
        $this->formatLength = null;
        $this->defaultTrueValue = null;
        $this->identifier = null;
        $this->order = null;
        $this->connectedVariables = new ArrayCollection();
        $this->tests = new ArrayCollection();
    }

    /**
     * @return Project|null
     */
    public abstract function getProject(): ?Project;

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return static
     */
    public function setName(string $name): self
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getShortName(): string
    {
        return $this->shortName;
    }

    /**
     * @param string $shortName
     * @return static
     */
    public function setShortName(string $shortName): self
    {
        $this->shortName = $shortName;
        return $this;
    }

    /**
     * @return int
     */
    public function getRepetitions(): int
    {
        return $this->repetitions;
    }

    /**
     * @param int $repetitions
     * @return static
     */
    public function setRepetitions(int $repetitions): self
    {
        $this->repetitions = $repetitions;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getUnit(): ?string
    {
        return $this->unit;
    }

    /**
     * @param string|null $unit
     * @return static
     */
    public function setUnit(?string $unit): self
    {
        $this->unit = $unit;
        return $this;
    }

    /**
     * @return string
     * @psalm-return PathLevelEnumId|''
     */
    public function getPathLevel(): string
    {
        return $this->pathLevel;
    }

    /**
     * @param string $pathLevel
     * @psalm-param PathLevelEnumId|'' $pathLevel
     * @return static
     */
    public function setPathLevel(string $pathLevel): self
    {
        $this->pathLevel = $pathLevel;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return static
     */
    public function setComment(?string $comment): self
    {
        $this->comment = $comment;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getOrder(): ?int
    {
        return $this->order;
    }

    /**
     * @param int|null $order
     * @return static
     */
    public function setOrder(?int $order): self
    {
        $this->order = $order;
        return $this;
    }

    /**
     * @return string|null
     * @psalm-return VariableFormatEnumId|numeric-string|null
     */
    public function getFormat(): ?string
    {
        return $this->format;
    }

    /**
     * @param string|null $format
     * @psalm-param VariableFormatEnumId|numeric-string|null $format
     * @return static
     */
    public function setFormat(?string $format): self
    {
        $this->format = $format;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getFormatLength(): ?int
    {
        return $this->formatLength;
    }

    /**
     * @param int|null $formatLength
     * @return static
     */
    public function setFormatLength(?int $formatLength): self
    {
        $this->formatLength = $formatLength;
        return $this;
    }

    /**
     * @return bool|null
     */
    public function getDefaultTrueValue(): ?bool
    {
        return $this->defaultTrueValue;
    }

    /**
     * @param bool|null $defaultTrueValue
     * @return static
     */
    public function setDefaultTrueValue(?bool $defaultTrueValue): self
    {
        $this->defaultTrueValue = $defaultTrueValue;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getIdentifier(): ?string
    {
        return $this->identifier;
    }

    /**
     * @param string|null $identifier
     * @return static
     */
    public function setIdentifier(?string $identifier): self
    {
        $this->identifier = $identifier;
        return $this;
    }

    /**
     * @return bool
     */
    public function isMandatory(): bool
    {
        return $this->mandatory;
    }

    /**
     * @param bool $mandatory
     * @return static
     */
    public function setMandatory(bool $mandatory): self
    {
        $this->mandatory = $mandatory;
        return $this;
    }

    /**
     * @return Collection<array-key, Test>
     */
    public function getTests()
    {
        return $this->tests;
    }

    /**
     * @param Collection<array-key, Test> $tests
     * @return static
     */
    public function setTests($tests)
    {
        $this->tests = $tests;
        return $this;
    }

    /**
     * @param Test $test
     * @return static
     */
    public function addTest($test): self
    {
        if (!$this->tests->contains($test)) {
            $this->tests->add($test);
            $test->setVariable($this);
        }
        return $this;
    }

    /**
     * @param Test $test
     * @return static
     */
    public function removeTest($test): self
    {
        if ($this->tests->contains($test)) {
            $this->tests->removeElement($test);
            $test->setVariable(null);
        }
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getCreated(): DateTime
    {
        return $this->created;
    }

    /**
     * @param DateTime $created
     * @return static
     */
    public function setCreated(DateTime $created): self
    {
        $this->created = $created;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getLastModified(): DateTime
    {
        return $this->lastModified;
    }

    /**
     * @param DateTime $lastModified
     * @return static
     */
    public function setLastModified(DateTime $lastModified): self
    {
        $this->lastModified = $lastModified;
        return $this;
    }

    /**
     * @return Collection<array-key, VariableConnection>
     */
    public function getConnectedVariables()
    {
        return $this->connectedVariables;
    }

    /**
     * @param Collection<array-key, VariableConnection> $connectedVariables
     * @return static
     */
    public function setConnectedVariables($connectedVariables): self
    {
        $this->connectedVariables = $connectedVariables;
        return $this;
    }


    /**
     * @param VariableConnection $variableConnection
     * @return static
     */
    public function addVariableConnection($variableConnection): self
    {
        if (!$this->connectedVariables->contains($variableConnection)) {
            $this->connectedVariables->add($variableConnection);
            $variableConnection->setProjectVariable($this);
        }
        return $this;
    }

    /**
     * @param VariableConnection $variableConnection
     * @return static
     */
    public function removeVariableConnection($variableConnection): self
    {
        if ($this->connectedVariables->contains($variableConnection)) {
            $this->connectedVariables->removeElement($variableConnection);
            $variableConnection->setProjectVariable(null);
        }
        return $this;
    }

}
