<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class Algorithm
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER')"}
 *     }
 * )
 *
 *
 * @ORM\Entity
 * @ORM\Table(name="algorithm", schema="webapp")
 */
class Algorithm extends IdentifiedEntity
{

    public const SPECIAL_ALGORITHM_SPLIT_PLOT = "SplitPlot";

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     * @Groups({"platform_full_view", "platform_synthesis", "protocol_synthesis", "protocol_full_view"})
     */
    private string $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     * @Groups({"platform_full_view", "protocol_full_view"})
     */
    private string $scriptName;

    /**
     * @var AlgorithmParameter[] | Collection
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\AlgorithmParameter", mappedBy="algorithm", cascade={"persist", "remove"})
     * @Groups({"platform_full_view", "protocol_full_view"})
     */
    private $algorithmParameters;

    /**
     * @var AlgorithmCondition[] | Collection
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\AlgorithmCondition", mappedBy="algorithm", cascade={"persist", "remove"})
     * @Groups({"platform_full_view", "protocol_full_view"})
     */
    private $algorithmConditions;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean")
     * @Groups({"platform_full_view", "protocol_full_view"})
     */
    private bool $withSubBlock;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean")
     * @Groups({"platform_full_view", "protocol_full_view"})
     */
    private bool $withRepartition;

    public function __construct()
    {
        $this->algorithmParameters = new ArrayCollection();
        $this->algorithmConditions = new ArrayCollection();
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Algorithm
     */
    public function setName(string $name): Algorithm
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getScriptName(): string
    {
        return $this->scriptName;
    }

    /**
     * @param string $scriptName
     * @return Algorithm
     */
    public function setScriptName(string $scriptName): Algorithm
    {
        $this->scriptName = $scriptName;
        return $this;
    }

    /**
     * @return Collection|AlgorithmParameter[]
     */
    public function getAlgorithmParameters()
    {
        return $this->algorithmParameters;
    }

    /**
     * @param AlgorithmParameter $algorithmParameter
     * @return Algorithm
     */
    public function addAlgorithmParameter($algorithmParameter)
    {
        if (!$this->algorithmParameters->contains($algorithmParameter)) {
            $this->algorithmParameters->add($algorithmParameter);
            $algorithmParameter->setAlgorithm($this);
        }
        return $this;
    }

    /**
     * @param AlgorithmParameter $algorithmParameter
     * @return Algorithm
     */
    public function removeAlgorithmParameter($algorithmParameter)
    {
        if ($this->algorithmParameters->contains($algorithmParameter)) {
            $this->algorithmParameters->removeElement($algorithmParameter);
            $algorithmParameter->setAlgorithm(null);
        }
        return $this;
    }

    /**
     * @param Collection|AlgorithmParameter[] $algorithmParameters
     * @return Algorithm
     */
    public function setAlgorithmParameters($algorithmParameters)
    {
        $this->algorithmParameters = $algorithmParameters;
        return $this;
    }

    /**
     * @param AlgorithmCondition $algorithmCondition
     * @return Algorithm
     */
    public function addAlgorithmCondition($algorithmCondition)
    {
        if (!$this->algorithmConditions->contains($algorithmCondition)) {
            $this->algorithmConditions->add($algorithmCondition);
            $algorithmCondition->setAlgorithm($this);
        }
        return $this;
    }

    /**
     * @param AlgorithmCondition $algorithmCondition
     * @return Algorithm
     */
    public function removeAlgorithmCondition($algorithmCondition)
    {
        if ($this->algorithmConditions->contains($algorithmCondition)) {
            $this->algorithmConditions->removeElement($algorithmCondition);
            $algorithmCondition->setAlgorithm(null);
        }
        return $this;
    }

    /**
     * @return Collection|AlgorithmCondition[]
     */
    public function getAlgorithmConditions()
    {
        return $this->algorithmConditions;
    }

    /**
     * @param Collection|AlgorithmCondition[] $algorithmConditions
     * @return Algorithm
     */
    public function setAlgorithmConditions($algorithmConditions)
    {
        $this->algorithmConditions = $algorithmConditions;
        return $this;
    }

    /**
     * @return bool
     */
    public function isWithSubBlock(): bool
    {
        return $this->withSubBlock;
    }

    /**
     * @param bool $withSubBlock
     * @return Algorithm
     */
    public function setWithSubBlock(bool $withSubBlock): Algorithm
    {
        $this->withSubBlock = $withSubBlock;
        return $this;
    }

    /**
     * @return bool
     */
    public function isWithRepartition(): bool
    {
        return $this->withRepartition;
    }

    /**
     * @param bool $withRepartition
     * @return Algorithm
     */
    public function setWithRepartition(bool $withRepartition): Algorithm
    {
        $this->withRepartition = $withRepartition;
        return $this;
    }


}
