<?php

namespace Webapp\FileManagement\Service;


use Doctrine\ORM\EntityManagerInterface;
use DOMDocument;
use DOMElement;
use DOMException;
use Mobile\Measure\Entity\Test\CombinationTest;
use Mobile\Measure\Entity\Test\GrowthTest;
use Mobile\Measure\Entity\Test\PreconditionedCalculation;
use Mobile\Measure\Entity\Test\RangeTest;
use Mobile\Measure\Entity\Variable\Base\Variable;
use Mobile\Measure\Entity\Variable\UniqueVariable;
use Mobile\Project\Entity\DataEntryProject;
use Mobile\Project\Entity\DesktopUser;
use Mobile\Project\Entity\NatureZHE;
use ReflectionException;
use Shared\FileManagement\Entity\UserLinkedJob;
use Shared\TransferSync\Entity\StatusDataEntry;
use Shared\TransferSync\Worker\IndividualStructureCheckHelper;
use Vich\UploaderBundle\Storage\StorageInterface;
use Webapp\FileManagement\Entity\ResponseFile;
use Webapp\FileManagement\Exception\WritingException;
use ZipArchive;

/**
 * Class FileWriterWorker.
 */
class FileWriterService extends AbstractWriterService
{

    private WriterHelper $writerHelper;
    private IndividualStructureCheckHelper $checkWorker;

    private string $desktopVersion;

    public function __construct(EntityManagerInterface         $entityManager,
                                DirectoryNamer                 $directoryNameManager,
                                StorageInterface               $storage,
                                WriterHelper                   $writerHelper,
                                IndividualStructureCheckHelper $checkWorker,
                                string                         $tmpDir,
                                string                         $desktopVersion)
    {
        parent::__construct($entityManager, $directoryNameManager, $storage, $tmpDir);
        $this->desktopVersion = $desktopVersion;
        $this->checkWorker = $checkWorker;
        $this->writerHelper = $writerHelper;
    }

    /**
     * Main function of the writer
     * @param int $responseFileId
     * @throws ReflectionException
     * @throws WritingException|DOMException
     */
    public function createFile(int $responseFileId)
    {
        $responseFile = $this->entityManager->getRepository(ResponseFile::class)
            ->find($responseFileId);
        /** @var StatusDataEntry $statusDataEntry */
        $statusDataEntry = $this->entityManager->getRepository(StatusDataEntry::class)
            ->findOneByResponseProject($responseFile->getProject());
        $projectResult = $statusDataEntry->getResponse();
        $this->writerHelper->init();
        $responseFile->setStatus(UserLinkedJob::STATUS_RUNNING);
        $this->entityManager->flush();

        $projectName = $projectResult->getName();
        $fileName = sprintf("Transfert-%s-%s.zip",
            iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $projectName),
            ($responseFile->getUploadDate())->format(self::dateFormat));
        $responseFile->setFilePath($fileName);

        $dom = $this->constructReturnElement($projectResult);

        $this->createZipFile($dom, $responseFile);

        if (!is_null($statusDataEntry->getWebappProject())) {
            $this->checkWorker->checkDataEntry($statusDataEntry);
        }

        $statusDataEntry->setSyncable(false);

        $statusDataEntry->setStatus(StatusDataEntry::STATUS_DONE);
        $responseFile->setStatus(UserLinkedJob::STATUS_SUCCESS);
        $this->entityManager->flush();
    }

    /**
     * Create the zip file
     * @param DOMDocument $dom
     * @param ResponseFile $responseFile
     * @throws WritingException
     */
    private function createZipFile(DOMDocument $dom, ResponseFile $responseFile): void
    {
        $tmpDir = $this->getTmpDir($responseFile);
        $tmpFilePath = $tmpDir . '/' . "transfert.xml";
        $filePath = $tmpDir . '/' . $responseFile->getFilePath();

        $zipFile = new ZipArchive();
        if ($zipFile->open($filePath, ZipArchive::CREATE) === true) {
            //Add the annotation pictures to the zip file
            foreach ($this->writerHelper->annotationFiles as $key => $annotationFile) {
                $content = file_get_contents($annotationFile);
                $size = getimagesizefromstring($content);
                //Get the file extension
                $extension = substr($size['mime'], 6);
                //Create the files from the string source
                file_put_contents($tmpDir . '/' . $key, $content);
                $zipFile->addFile($tmpDir . '/' . $key, "metadonnees/$key.$extension");
            }
            file_put_contents($tmpFilePath, $dom->saveXML());
            $zipFile->addFile($tmpFilePath, "transfert.xml");
            $zipFile->addFromString("umpc", "");
            $zipFile->close();
            $this->copyFileFromTmpDir($responseFile);
        } else {
            throw new WritingException("Unable to create zip file");
        }
        $this->cleanTmpDir($responseFile);
    }

    /**
     * @param DataEntryProject $project
     * @param $fileName
     * @return void
     * @throws ReflectionException
     * @throws DOMException
     */
    public function constructProjectZipFile(DataEntryProject $project, $fileName)
    {
        $xml = $this->constructProject($project);

        $zipFile = new ZipArchive();
        if ($zipFile->open($fileName, ZipArchive::CREATE) === true) {
            foreach ($this->writerHelper->markFiles as $markFile) {
                $key = uniqid();
                $content = file_get_contents($markFile[1]);
                $size = getimagesizefromstring($content);
                //Get the file extension
                $extension = substr($size['mime'], 6);
                //Create the files from the string source
                $fileName = "echelles/$key.$extension";
                $markFile[0]->setAttribute('image', $fileName);
                $zipFile->addFromString($fileName, $content);
            }
            $zipFile->addFromString($project->getName() . ".xml", $xml->saveXML());
            $zipFile->close();
        }
    }

    /**
     * Extract the Dom document from an abstract project
     * @param DataEntryProject $projectResult
     * @return DOMDocument
     * @throws ReflectionException|DOMException
     */
    private function constructReturnElement(DataEntryProject $projectResult): DOMDocument
    {
        list($dom, $base, $count, $uriMap) = $this->initDom();
        //The uri map associate php object to their uri in the xml file

        $dom = $this->constructProject($projectResult, $dom, $base, $uriMap, $count);

        $dataEntry = $projectResult->getDataEntry();
        $dataEntryXml = $this->writerHelper->constructDataEntryElement($dom, $dataEntry, $uriMap, $count);

        if ($projectResult instanceof DataEntryProject) {
            $connectedVariablesCount = 0;
            foreach ($projectResult->getConnectedUniqueVariables() as $uniqueVariable) {
                $variableXml = $this->writerHelper->constructUniqueVariableElement($dom, $uniqueVariable, $uriMap, $connectedVariablesCount, false, false, $dataEntry);
                $dataEntryXml->appendChild($variableXml);
                $connectedVariablesCount++;
            }

            foreach ($projectResult->getConnectedGeneratorVariables() as $generatorVariable) {
                $variableXml = $this->writerHelper->constructGeneratorVariableElement($dom, $generatorVariable, $uriMap, $connectedVariablesCount, false, false, $dataEntry);
                $dataEntryXml->appendChild($variableXml);
                $connectedVariablesCount++;
            }
        }

        $base->appendChild($dataEntryXml);

        $this->completeVariableXml($dom, $uriMap);

        return $dom;
    }

    /**
     * @param DataEntryProject $project
     * @param DOMDocument|null $dom
     * @param $base
     * @param array $uriMap
     * @param int $count
     * @return DOMDocument
     * @throws ReflectionException
     * @throws DOMException
     */
    private function constructProject(DataEntryProject $project, DOMDocument $dom = null, &$base = null, array &$uriMap = [], int &$count = 0)
    {
        if ($dom === null) {
            $this->writerHelper->init();
            list($dom, $base, $count, $uriMap) = $this->initDom();
        }

        foreach ($project->getDesktopUsers() as $desktopUser) {
            /** @var DesktopUser $desktopUser */
            $desktopUserXml = $this->writerHelper->constructDesktopUserElement($dom, $desktopUser, $uriMap, $count);
            $base->appendChild($desktopUserXml);
            $count++;
        }

        foreach ($project->getNaturesZHE() as $natureZHE) {
            /** @var NatureZHE $natureZHE */
            $stateCodeXml = $this->writerHelper->constructNatureZHEElement($dom, $natureZHE, $uriMap, $count);
            $base->appendChild($stateCodeXml);
            $count++;
        }

        $plateformXml = $this->writerHelper->constructPlatformElement($dom, $project->getPlatform(), $uriMap, $count);
        $base->appendChild($plateformXml);
        $count++;

        $dataEntryProjectXml = $this->writerHelper->constructDataEntryProjectElement($dom, $project, $uriMap, $count);
        $base->appendChild($dataEntryProjectXml);
        $count++;

        // TODO gérer les variables connectées

        $this->completeVariableXml($dom, $uriMap);
        return $dom;
    }

    /**
     * @param $dom
     * @param array $uriMap
     * @return void
     * @throws ReflectionException
     */
    private function completeVariableXml($dom, array $uriMap): void
    {
        foreach ($this->writerHelper->variablesTuples as $tuple) {
            /** @var Variable $variable */
            $variable = $tuple[0];
            /** @var DOMElement $variableXml */
            $variableXml = $tuple[1];
            foreach ($variable->getCombinationTests() as $test) {
                /** @var CombinationTest $test */
                $variableXml->appendChild($this->writerHelper->constructCombinationTestElement($dom, $test, $uriMap));
            }
            foreach ($variable->getGrowthTests() as $test) {
                /** @var GrowthTest $test */
                $variableXml->appendChild($this->writerHelper->constructGrowthTestElement($dom, $test, $uriMap));
            }
            foreach ($variable->getPreconditionedCalculations() as $test) {
                /** @var PreconditionedCalculation $test */
                $variableXml->appendChild($this->writerHelper->constructPreconditionedCalculationElement($dom, $test, $uriMap));
            }
            foreach ($variable->getRangeTests() as $test) {
                /** @var RangeTest $test */
                $variableXml->appendChild($this->writerHelper->constructRangeTestElement($dom, $test, $uriMap));
            }
            if ($variable->getMaterial() !== null) {
                $variableXml->setAttribute("materiel", $uriMap[$variable->getMaterial()->getUri()]);
            }
            if ($variable->getScale() !== null) {
                $variableXml->appendChild($this->writerHelper->constructScaleElement($dom, $variable->getScale(), $uriMap));
            }
            if ($variable instanceof UniqueVariable && $variable->getValueHintList() !== null) {
                $variableXml->appendChild($this->writerHelper->constructValueHintListElement($dom, $variable->getValueHintList(), $uriMap));
            }
        }
        $this->writerHelper->variablesTuples = [];
    }

    /**
     * @return array
     * @throws DOMException
     */
    private function initDom(): array
    {
        $dom = new DOMDocument("1.0", "UTF-8");
        $dom->preserveWhiteSpace = false;
        //Needed to make the XML human readable
        $dom->formatOutput = true;

        $base = $dom->createElementNS(self::NS_XMI, 'xmi:XMI');
        $base->setAttributeNS(self::NS_XMI, "xmi:version", "2.0");
        $base->setAttribute("xmlns:xsi", self::NS_XSI);
        $base->setAttribute("xmlns:adonis.modeleMetier.plateforme", self::NS_PLATEFORME);
        $base->setAttribute("xmlns:adonis.modeleMetier.projetDeSaisie", self::NS_PROJET_SAISIE);
        $base->setAttribute("xmlns:adonis.modeleMetier.projetDeSaisie.cheminement", self::NS_CHEMINEMENT);
        $base->setAttribute("xmlns:adonis.modeleMetier.projetDeSaisie.variables", self::NS_VARIABLES);
        $base->setAttribute("xmlns:adonis.modeleMetier.saisieTerrain", self::NS_SAISIE_TERRAIN);
        $base->setAttribute("xmlns:adonis.modeleMetier.site", self::NS_SITE);
        $base->setAttribute("xmlns:adonis.modeleMetier.utilisateur", self::NS_UTILISATEUR);
        $base->setAttribute("xmlns:adonis.modeleMetier.conceptsDeBase", self::NS_CONCEPT_DE_BASE);
        $dom->appendChild($base);

        $version = $dom->createElement("adonis.modeleMetier.site:AdonisVersion");
        $version->setAttribute("version", $this->desktopVersion);
        $base->appendChild($version);

        return array($dom, $base, 1, []);
    }

}
