<?php

namespace Mobile\Device\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class Treatment.
 *
 * @ORM\Entity(repositoryClass="Mobile\Device\Repository\TreatmentRepository")
 * @ORM\Table(name="treatment", schema="adonis")
 */
class Treatment extends IdentifiedEntity
{
    public const DEFAULT_SHORT_NAME_FROM_CSV = "";
    public const DEFAULT_REPETITIONS_FROM_CSV = 1;

    /**
     * @var string
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\Column(type="string", nullable=false)
     */
    private $name;

    /**
     * @var string
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\Column(type="string", nullable=false)
     */
    private $shortName;

    /**
     * @var integer
     *
     * @ORM\Column(type="integer", nullable=false)
     */
    private $repetitions;

    /**
     * @var Collection<Modality>
     *
     * @ORM\ManyToMany(targetEntity="Mobile\Device\Entity\Modality", cascade={"persist", "remove", "detach"})
     * @ORM\JoinTable(name="rel_treatment_modality", schema="adonis")
     */
    private $modalities;

    /**
     * @var Protocol
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\Protocol", inversedBy="treatments")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    private $protocol;

    /**
     * Treatment constructor.
     */
    public function __construct()
    {
        $this->modalities = new ArrayCollection();
    }

    /**
     * @return array
     */
    public function getModalitiesValue(): array
    {
        $modalitiesValue = [];
        foreach ($this->getModalities() as $modality) {
            /** @var $modality Modality */
            $modalitiesValue[] = $modality->getValue();
        }
        return $modalitiesValue;
    }

    /**
     * @return Collection<Modality>
     */
    public function getModalities(): Collection
    {
        return $this->modalities;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Treatment
     */
    public function setName(string $name): Treatment
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getShortName(): string
    {
        return $this->shortName;
    }

    /**
     * @param string $shortName
     * @return Treatment
     */
    public function setShortName(string $shortName): Treatment
    {
        $this->shortName = $shortName;
        return $this;
    }

    /**
     * @return int
     */
    public function getRepetitions(): int
    {
        return $this->repetitions;
    }

    /**
     * @param int $repetitions
     * @return Treatment
     */
    public function setRepetitions(int $repetitions): Treatment
    {
        $this->repetitions = $repetitions;
        return $this;
    }

    /**
     * @param Modality $modality
     */
    public function addModality(Modality $modality): void
    {
        if (!$this->modalities->contains($modality)) {
            $this->modalities->add($modality);
        }
    }

    /**
     * @param Modality $modality
     */
    public function removeModality(Modality $modality): void
    {
        if ($this->modalities->contains($modality)) {
            $this->modalities->removeElement($modality);
        }
    }

    /**
     * @return Protocol
     */
    public function getProtocol(): Protocol
    {
        return $this->protocol;
    }

    /**
     * @param Protocol $protocol
     * @return Treatment
     */
    public function setProtocol(Protocol $protocol): Treatment
    {
        $this->protocol = $protocol;
        return $this;
    }
}
