<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2020
 */

declare(strict_types=1);

namespace Mobile\Measure\Entity;

use DateTime;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Project\Entity\ProjectObject;
use Shared\Authentication\Entity\IdentifiedEntity;

/**
 * Class Annotation.
 *
 * @ORM\Entity()
 * @ORM\Table(name="annotation", schema="adonis")
 */
class Annotation extends IdentifiedEntity
{
    public const ANNOT_TYPE_TEXT = 0;
    public const ANNOT_TYPE_IMAG = 1;
    public const ANNOT_TYPE_SONG = 2;

    /**
     * @var ProjectObject
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Project\Entity\ProjectObject", inversedBy="annotations")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    private $projectObject;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $name;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", nullable=false)
     */
    private $type;

    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $value;

    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $image;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     */
    private $timestamp;

    /**
     * @var string[]
     *
     * @ORM\Column(type="array", nullable=false)
     */
    private $categories;

    /**
     * @var string[]
     *
     * @ORM\Column(type="array", nullable=false)
     */
    private $keywords;

    public function __construct()
    {
        $this->keywords = [];
        $this->categories = [];
    }

    /**
     * @return ProjectObject
     */
    public function getProjectObject(): ProjectObject
    {
        return $this->projectObject;
    }

    /**
     * @param ProjectObject $projectObject
     * @return Annotation
     */
    public function setProjectObject(ProjectObject $projectObject): Annotation
    {
        $this->projectObject = $projectObject;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @param string|null $name
     * @return Annotation
     */
    public function setName(?string $name): Annotation
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return int
     */
    public function getType(): int
    {
        return $this->type;
    }

    /**
     * @param int $type
     * @return Annotation
     */
    public function setType(int $type): Annotation
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getValue(): ?string
    {
        return $this->value;
    }

    /**
     * @param string|null $value
     * @return Annotation
     */
    public function setValue(?string $value): Annotation
    {
        $this->value = $value;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getImage(): ?string
    {
        return $this->image;
    }

    /**
     * @param string|null $image
     * @return Annotation
     */
    public function setImage(?string $image): Annotation
    {
        $this->image = $image;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getTimestamp(): DateTime
    {
        return $this->timestamp;
    }

    /**
     * @param DateTime $timestamp
     * @return Annotation
     */
    public function setTimestamp(DateTime $timestamp): Annotation
    {
        $this->timestamp = $timestamp;
        return $this;
    }

    /**
     * @return array
     */
    public function getCategories(): array
    {
        return $this->categories;
    }

    /**
     * @param array $categories
     * @return Annotation
     */
    public function setCategories(array $categories): Annotation
    {
        $this->categories = $categories;
        return $this;
    }

    public function addCategory(string $category): void
    {
        if (!in_array($category, $this->categories)) {
            $this->categories[] = $category;
        }
    }

    /**
     * @return array
     */
    public function getKeywords(): array
    {
        return $this->keywords;
    }

    /**
     * @param array $keywords
     * @return Annotation
     */
    public function setKeywords(array $keywords): Annotation
    {
        $this->keywords = $keywords;
        return $this;
    }

    /**
     * @param string $keyword
     */
    public function addKeyword(string $keyword): void
    {
        if (!in_array($keyword, $this->keywords)) {
            $this->keywords[] = $keyword;
        }
    }

}
