<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2020
 */

declare(strict_types=1);

namespace Mobile\Measure\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Device\Entity\BusinessObject;
use Mobile\Measure\Entity\Variable\Base\Variable;
use Shared\Authentication\Entity\IdentifiedEntity;

/**
 * Class FormField.
 *
 * @ORM\Entity()
 * @ORM\Table(name="field_measure", schema="adonis")
 */
class FormField extends IdentifiedEntity
{
    /**
     * @var Session|null
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Measure\Entity\Session", inversedBy="formFields")
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    private $session;

    /**
     * @var GeneratedField|null
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Measure\Entity\GeneratedField", inversedBy="formFields")
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    private $generatedField;

    /**
     * @var Variable
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Measure\Entity\Variable\Base\Variable")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    private $variable;

    /**
     * @var BusinessObject
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\BusinessObject")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    private $target;

    /**
     * @var Collection<Measure>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Measure", mappedBy="formField", cascade={"persist", "remove", "detach"})
     */
    private $measures;

    /**
     * @var Collection<GeneratedField>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\GeneratedField", mappedBy="formField", cascade={"persist", "remove", "detach"})
     */
    private $generatedFields;

    /**
     * FormField constructor.
     */
    public function __construct()
    {
        $this->measures = new ArrayCollection();
        $this->generatedFields = new ArrayCollection();
    }

    /**
     * @return Session|null
     */
    public function getSession(): ?Session
    {
        return $this->session;
    }

    /**
     * @param Session|null $session
     * @return FormField
     */
    public function setSession(?Session $session): FormField
    {
        $this->session = $session;
        return $this;
    }

    /**
     * @return GeneratedField|null
     */
    public function getGeneratedField(): ?GeneratedField
    {
        return $this->generatedField;
    }

    /**
     * @param GeneratedField|null $generatedField
     * @return FormField
     */
    public function setGeneratedField(?GeneratedField $generatedField): FormField
    {
        $this->generatedField = $generatedField;
        return $this;
    }

    /**
     * @return Variable
     */
    public function getVariable(): Variable
    {
        return $this->variable;
    }

    /**
     * @param Variable $variable
     * @return FormField
     */
    public function setVariable(Variable $variable): FormField
    {
        $this->variable = $variable;
        return $this;
    }

    /**
     * @return BusinessObject
     */
    public function getTarget(): BusinessObject
    {
        return $this->target;
    }

    /**
     * @param BusinessObject $target
     * @return FormField
     */
    public function setTarget(BusinessObject $target): FormField
    {
        $this->target = $target;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getMeasures(): Collection
    {
        return $this->measures;
    }

    /**
     * @param Measure $measure
     * @return $this
     */
    public function addMeasure(Measure $measure): FormField
    {
        if (!$this->measures->contains($measure)) {
            $this->measures->add($measure);
            $measure->setFormField($this);
        }
        return $this;
    }

    /**
     * @param Measure $measure
     * @return $this
     */
    public function removeMeasure(Measure $measure): FormField
    {
        if ($this->measures->contains($measure)) {
            $this->measures->removeElement($measure);
        }
        return $this;
    }

    /**
     * @return Collection
     */
    public function getGeneratedFields(): Collection
    {
        return $this->generatedFields;
    }

    /**
     * @param GeneratedField $generatedField
     * @return $this
     */
    public function addGeneratedField(GeneratedField $generatedField): FormField
    {
        if (!$this->generatedFields->contains($generatedField)) {
            $this->generatedFields->add($generatedField);
            $generatedField->setFormField($this);
        }
        return $this;
    }

    /**
     * @param GeneratedField $generatedField
     * @return $this
     */
    public function removeGeneratedField(GeneratedField $generatedField): FormField
    {
        if ($this->generatedFields->contains($generatedField)) {
            $this->generatedFields->removeElement($generatedField);
            $generatedField->setFormField(null);
        }
        return $this;
    }
}
