<?php

declare(strict_types=1);

namespace Shared\Authentication\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class RelUserSite.
 * @ApiResource(
 *     collectionOperations={
 *         "get"={},
 *         "post"={"security_post_denormalize"="is_granted('REL_USER_SITE_POST', object)"},
 *     },
 *     itemOperations={
 *         "get"={"security"="is_granted('REL_USER_SITE_GET', object)"},
 *         "put"={"security"="is_granted('REL_USER_SITE_EDIT', object)"},
 *         "patch"={"security"="is_granted('REL_USER_SITE_EDIT', object)"},
 *         "delete"={"security"="is_granted('REL_USER_SITE_DELETE', object)"},
 *     }
 * )
 * @ApiFilter(SearchFilter::class, properties={"site": "exact", "user": "exact"})
 *
 * @Gedmo\SoftDeleteable()
 *
 * @ORM\Entity()
 * @ORM\Table(name="rel_user_site", schema="shared")
 */
class RelUserSite
{
    public const EXPERIMENTER = 'expert';
    public const PLATFORM_MANAGER = 'manager';
    public const SITE_ADMIN = 'admin';

    use SoftDeleteableEntity;

    /**
     * @var ?User the user involved in the relationship
     * @Groups({"site_post"})
     * @ORM\Id()
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\User", inversedBy="siteRoles")
     * @ORM\JoinColumn(name="user_id", referencedColumnName="id", nullable=false)
     */
    private ?User $user;

    /**
     * @var ?Site the site involved in the relationship
     * @ORM\Id()
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="userRoles")
     * @ORM\JoinColumn(name="site_id", referencedColumnName="id", nullable=false)
     */
    private ?Site $site;

    /**
     * @var string the role of the user in that site
     * @Groups({"site_post"})
     * @ORM\Column(type="string")
     */
    private string $role;

    /**
     * @return User
     */
    public function getUser(): User
    {
        return $this->user;
    }

    /**
     * @param ?User $user
     * @return RelUserSite
     */
    public function setUser(?User $user): RelUserSite
    {
        $this->user = $user;
        return $this;
    }

    /**
     * @return Site
     */
    public function getSite(): Site
    {
        return $this->site;
    }

    /**
     * @param ?Site $site
     * @return RelUserSite
     */
    public function setSite(?Site $site): RelUserSite
    {
        $this->site = $site;
        return $this;
    }

    /**
     * @return string
     */
    public function getRole(): string
    {
        return $this->role;
    }

    /**
     * @param string $role
     * @return RelUserSite
     */
    public function setRole(string $role): RelUserSite
    {
        $this->role = $role;
        return $this;
    }


}
