<?php

namespace Webapp\Core\Dto\BusinessObject;

use ApiPlatform\Core\Api\IriConverterInterface;
use ApiPlatform\Core\Api\OperationType;
use ApiPlatform\Core\DataTransformer\DataTransformerInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ContainerBagInterface;
use Webapp\Core\Entity\Block;
use Webapp\Core\Entity\Experiment;
use Webapp\Core\Entity\OutExperimentationZone;
use Webapp\Core\Entity\Platform;
use Webapp\Core\Entity\SubBlock;

/**
 * Class ExperimentPostTransformer
 * @package Webapp\Core\Dto\Experiment
 */
class BusinessObjectCsvTransformer implements DataTransformerInterface
{

    public function __construct(IriConverterInterface $iriConverter, ContainerBagInterface $params)
    {
        $this->iriConverter = $iriConverter;
        $this->params = $params;
    }

    /**
     * @param Experiment| Platform $object
     * @param string $to
     * @param array $context
     * @return BusinessObjectCsvOutputDto[]
     */
    public function transform($object, string $to, array $context = []): array
    {
        $output = [];

        if ($object instanceof Platform) {
            $platformName = $object->getName();
            foreach ($object->getExperiments() as $experiment) {
                $this->handleExperiment($experiment, $output, $platformName);
            }
        } else {
            $this->handleExperiment($object, $output);
        }

        return $output;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        if (!$data instanceof Experiment && !$data instanceof Platform) {
            return false;
        }
        return BusinessObjectCsvOutputDto::class === $to &&
            $context['operation_type'] === OperationType::ITEM &&
            $context['item_operation_name'] === 'export';
    }

    /**
     * @param array $output
     * @param $plotContainer
     * @param $experimentName
     * @param $blockName
     * @param $subBlockName
     * @return void
     */
    private function handleUnitPlotContainer(array &$output, $plotContainer, $platformName, $experimentName, $blockName, $subBlockName = null)
    {
        /** @var Block | SubBlock $plotContainer */
        foreach ($plotContainer->getUnitPlots() as $unitPlot) {
            $unitPlotName = $unitPlot->getNumber();
            $modalities = $unitPlot->getTreatment()->getModalities();
            foreach ($unitPlot->getOutExperimentationZones() as $oez) {
                $this->handleOEZ($output, $oez, $experimentName, $blockName, $subBlockName, $unitPlotName);
            }
            foreach ($unitPlot->getIndividuals() as $individual) {
                $output[] = (new BusinessObjectCsvOutputDto())
                    ->setX($individual->getX())
                    ->setY($individual->getY())
                    ->setAppeared($individual->getAppeared())
                    ->setDisapeared($individual->getDisappeared())
                    ->setDead($individual->isDead())
                    ->setId($individual->getIdentifier())
                    ->setTreatment($unitPlot->getTreatment()->getName())
                    ->setShortTreatment($unitPlot->getTreatment()->getShortName())
                    ->setFactor1($modalities->get(0)->getValue())
                    ->setFactor1Id($modalities->get(0)->getIdentifier())
                    ->setFactor1ShortName($modalities->get(0)->getShortName())
                    ->setFactor2($modalities->containsKey(1) ? $modalities->get(1)->getValue() : null)
                    ->setFactor2Id($modalities->containsKey(1) ? $modalities->get(1)->getIdentifier() : null)
                    ->setFactor2ShortName($modalities->containsKey(1) ? $modalities->get(1)->getShortName() : null)
                    ->setFactor3($modalities->containsKey(2) ? $modalities->get(2)->getValue() : null)
                    ->setFactor3Id($modalities->containsKey(2) ? $modalities->get(2)->getIdentifier() : null)
                    ->setFactor3ShortName($modalities->containsKey(2) ? $modalities->get(2)->getShortName() : null)
                    ->setPlatform($platformName)
                    ->setExperiment($experimentName)
                    ->setBlock($blockName)
                    ->setSubBlock($subBlockName)
                    ->setUp($unitPlotName);
            }
        }
        foreach ($plotContainer->getSurfacicUnitPlots() as $surfacicUnitPlot) {
            $unitPlotName = $surfacicUnitPlot->getNumber();
            $modalities = $surfacicUnitPlot->getTreatment()->getModalities();
            $output[] = (new BusinessObjectCsvOutputDto())
                ->setX($surfacicUnitPlot->getX())
                ->setY($surfacicUnitPlot->getY())
                ->setAppeared($surfacicUnitPlot->getAppeared())
                ->setDisapeared($surfacicUnitPlot->getDisappeared())
                ->setDead($surfacicUnitPlot->isDead())
                ->setId($surfacicUnitPlot->getIdentifier())
                ->setTreatment($surfacicUnitPlot->getTreatment()->getName())
                ->setShortTreatment($surfacicUnitPlot->getTreatment()->getShortName())
                ->setFactor1($modalities->get(0)->getValue())
                ->setFactor1Id($modalities->get(0)->getIdentifier())
                ->setFactor1ShortName($modalities->get(0)->getShortName())
                ->setFactor2($modalities->containsKey(1) ? $modalities->get(1)->getValue() : null)
                ->setFactor2Id($modalities->containsKey(1) ? $modalities->get(1)->getIdentifier() : null)
                ->setFactor2ShortName($modalities->containsKey(1) ? $modalities->get(1)->getShortName() : null)
                ->setFactor3($modalities->containsKey(2) ? $modalities->get(2)->getValue() : null)
                ->setFactor3Id($modalities->containsKey(2) ? $modalities->get(2)->getIdentifier() : null)
                ->setFactor3ShortName($modalities->containsKey(2) ? $modalities->get(2)->getShortName() : null)
                ->setPlatform($platformName)
                ->setExperiment($experimentName)
                ->setBlock($blockName)
                ->setSubBlock($subBlockName)
                ->setUp($unitPlotName);
        }
    }

    /**
     * @param array $output
     * @param OutExperimentationZone $oez
     * @param $experimentName
     * @param $blockName
     * @param $subBlockName
     * @param $upName
     */
    private function handleOez(array &$output, OutExperimentationZone $oez, $platformName, $experimentName, $blockName = null, $subBlockName = null, $upName = null)
    {
        $output[] = (new BusinessObjectCsvOutputDto())
            ->setX($oez->getX())
            ->setY($oez->getY())
            ->setZHE($oez->getNature()->getNature())
            ->setPlatform($platformName)
            ->setExperiment($experimentName)
            ->setBlock($blockName)
            ->setSubBlock($subBlockName)
            ->setUp($upName);
    }

    /**
     * @param $experiment
     * @param array $output
     * @param null $platformName
     * @return void
     */
    private function handleExperiment($experiment, array &$output, $platformName = null): void
    {
        $experimentName = $experiment->getName();
        foreach ($experiment->getOutExperimentationZones() as $oez) {
            $this->handleOEZ($output, $oez, $platformName, $experimentName);
        }

        foreach ($experiment->getBlocks() as $block) {
            $blockName = $block->getNumber();
            foreach ($block->getOutExperimentationZones() as $oez) {
                $this->handleOEZ($output, $oez, $platformName, $experimentName, $blockName);
            }
            foreach ($block->getSubBlocks() as $subBlock) {
                $subBlockName = $subBlock->getNumber();
                foreach ($subBlock->getOutExperimentationZones() as $oez) {
                    $this->handleOEZ($output, $oez, $platformName, $experimentName, $blockName, $subBlockName);
                }
                $this->handleUnitPlotContainer($output, $subBlock, $platformName, $experimentName, $blockName, $subBlockName);
            }
            $this->handleUnitPlotContainer($output, $block, $platformName, $experimentName, $blockName);
        }
    }
}
