<?php

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use DateTime;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Enumeration\PathLevelEnum;

/**
 * Class Annotation.
 *
 * @ApiResource(
 *     collectionOperations={
 *          "get"={},
 *     },
 *     itemOperations={
 *          "get"={},
 *     }
 * )
 * @ApiFilter(SearchFilter::class, properties={"session": "exact"})
 *
 * @ORM\Entity()
 * @ORM\Table(name="annotation", schema="webapp")
 */
class Annotation extends IdentifiedEntity
{
    /**
     * @var Measure|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Measure", inversedBy="annotations")
     */
    private ?Measure $targetMeasure;

    /**
     * @var Individual|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Individual")
     */
    private ?Individual $targetIndividual;

    /**
     * @var SurfacicUnitPlot|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SurfacicUnitPlot")
     */
    private ?SurfacicUnitPlot $targetSurfacicUnitPlot;

    /**
     * @var UnitPlot|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\UnitPlot")
     */
    private ?UnitPlot $targetUnitPlot;

    /**
     * @var SubBlock|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SubBlock")
     */
    private ?SubBlock $targetSubBlock;

    /**
     * @var Block|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Block")
     */
    private ?Block $targetBlock;

    /**
     * @var Experiment|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Experiment")
     */
    private ?Experiment $targetExperiment;

    /**
     * @var Session
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Session", inversedBy="annotations")
     * @ORM\JoinColumn(nullable=false)
     */
    private Session $session;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"webapp_data_view"})
     */
    private $name;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", nullable=false)
     * @Groups({"webapp_data_view"})
     */
    private $type;

    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true)
     * @Groups({"webapp_data_view"})
     */
    private $value;

    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true)
     * @Groups({"webapp_data_view"})
     */
    private $image;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     * @Groups({"webapp_data_view"})
     */
    private $timestamp;

    /**
     * @var string[]
     *
     * @ORM\Column(type="array", nullable=false)
     * @Groups({"webapp_data_view", "data_entry_synthesis"})
     */
    private $categories;

    /**
     * @var string[]
     *
     * @ORM\Column(type="array", nullable=false)
     * @Groups({"webapp_data_view"})
     */
    private $keywords;

    public function __construct()
    {
        $this->categories = [];
        $this->keywords = [];
    }

    /**
     * @return Block|Experiment|Individual|Measure|SubBlock|SurfacicUnitPlot|UnitPlot|null
     * @Groups({"webapp_data_view"})
     */
    public function getTarget()
    {
        return $this->targetExperiment ?? $this->targetBlock ?? $this->targetSubBlock ?? $this->targetUnitPlot ?? $this->targetSurfacicUnitPlot ?? $this->targetIndividual ?? $this->targetMeasure;
    }

    /**
     * @Groups({"webapp_data_view", "data_entry_synthesis"})
     * @return string|null
     */
    public function getTargetType(): ?string
    {
        $target = $this->getTarget();
        if ($target instanceof Individual) {
            return PathLevelEnum::INDIVIDUAL;
        } elseif ($target instanceof UnitPlot) {
            return PathLevelEnum::UNIT_PLOT;
        } elseif ($target instanceof SubBlock) {
            return PathLevelEnum::SUB_BLOCK;
        } elseif ($target instanceof Block) {
            return PathLevelEnum::BLOCK;
        } elseif ($target instanceof Experiment) {
            return PathLevelEnum::EXPERIMENT;
        } elseif ($target instanceof SurfacicUnitPlot) {
            return PathLevelEnum::SURFACIC_UNIT_PLOT;
        } else {
            return null;
        }
    }

    /**
     * @param Block|Experiment|Individual|Measure|SubBlock|SurfacicUnitPlot|UnitPlot|null $target
     * @return Annotation
     */
    public function setTarget($target): Annotation
    {
        if ($target instanceof Individual) {
            $this->targetIndividual = $target;
        } elseif ($target instanceof UnitPlot) {
            $this->targetUnitPlot = $target;
        } elseif ($target instanceof SubBlock) {
            $this->targetSubBlock = $target;
        } elseif ($target instanceof Block) {
            $this->targetBlock = $target;
        } elseif ($target instanceof Experiment) {
            $this->targetExperiment = $target;
        } elseif ($target instanceof SurfacicUnitPlot) {
            $this->targetSurfacicUnitPlot = $target;
        } elseif ($target instanceof Measure) {
            $this->targetMeasure = $target;
        }
        return $this;
    }

    /**
     * @return Session
     */
    public function getSession(): Session
    {
        return $this->session;
    }

    /**
     * @param Session $session
     * @return Annotation
     */
    public function setSession(Session $session): Annotation
    {
        $this->session = $session;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @param string|null $name
     * @return Annotation
     */
    public function setName(?string $name): Annotation
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return int
     */
    public function getType(): int
    {
        return $this->type;
    }

    /**
     * @param int $type
     * @return Annotation
     */
    public function setType(int $type): Annotation
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getValue(): ?string
    {
        return $this->value;
    }

    /**
     * @param string|null $value
     * @return Annotation
     */
    public function setValue(?string $value): Annotation
    {
        $this->value = $value;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getImage(): ?string
    {
        return $this->image;
    }

    /**
     * @param string|null $image
     * @return Annotation
     */
    public function setImage(?string $image): Annotation
    {
        $this->image = $image;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getTimestamp(): DateTime
    {
        return $this->timestamp;
    }

    /**
     * @param DateTime $timestamp
     * @return Annotation
     */
    public function setTimestamp(DateTime $timestamp): Annotation
    {
        $this->timestamp = $timestamp;
        return $this;
    }

    /**
     * @return string[]
     */
    public function getCategories(): array
    {
        return $this->categories;
    }

    /**
     * @param string[] $categories
     * @return Annotation
     */
    public function setCategories(array $categories): Annotation
    {
        $this->categories = $categories;
        return $this;
    }

    /**
     * @return string[]
     */
    public function getKeywords(): array
    {
        return $this->keywords;
    }

    /**
     * @param string[] $keywords
     * @return Annotation
     */
    public function setKeywords(array $keywords): Annotation
    {
        $this->keywords = $keywords;
        return $this;
    }

}
