<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\Site;
use Shared\Enumeration\Annotation\EnumType;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Webapp\Core\Validator\UniqueAttributeInParent;

/**
 * Class Device
 * @package Webapp\Core\Entity
 * @ApiResource(
 *     denormalizationContext={"groups"={"semi_automatic_variable"}},
 *     normalizationContext={"groups"={"semi_automatic_variable"}},
 *     collectionOperations={
 *         "get"={},
 *         "post"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())"
 *          },
 *     },
 *     itemOperations={
 *          "get"={},
 *          "patch"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())"
 *          },
 *          "delete"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())"
 *          }
 *     }
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={"site": "exact", "alias": "exact",  "managedVariables": "exact", "project": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"project_explorer_view", "connected_variables"}})
 *
 * @ORM\Entity
 * @ORM\Table(name="device", schema="webapp")
 */
class Device extends IdentifiedEntity
{

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Assert\NotBlank
     * @UniqueAttributeInParent(parentsAttributes={"site.devices", "project.devices"})
     * @Groups({"semi_automatic_variable", "project_explorer_view", "data_explorer_view"})
     */
    private string $alias;

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"semi_automatic_variable"})
     */
    private string $name;

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"semi_automatic_variable"})
     */
    private string $manufacturer;

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"semi_automatic_variable"})
     */
    private string $type;

    /**
     * @var Site|null
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="devices")
     * @Groups({"semi_automatic_variable"})
     */
    protected ?Site $site;

    /**
     * @var Collection | array<SemiAutomaticVariable>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\SemiAutomaticVariable", mappedBy="device", cascade={"persist", "remove"}, orphanRemoval=true)
     * @Groups({"semi_automatic_variable", "project_explorer_view", "connected_variables"})
     */
    private $managedVariables;

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"semi_automatic_variable"})
     * @EnumType(class="Webapp\Core\Enumeration\CommunicationProtocolEnum")
     */
    private string $communicationProtocol;

    /**
     * @var int
     * @ORM\Column(type="integer")
     * @Groups({"semi_automatic_variable"})
     */
    private int $frameLength;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"semi_automatic_variable"})
     */
    private ?string $frameStart;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"semi_automatic_variable"})
     */
    private ?string $frameEnd;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"semi_automatic_variable"})
     */
    private ?string $frameCsv;

    /**
     * @var int|null
     * @ORM\Column(type="integer", nullable=true)
     * @Groups({"semi_automatic_variable"})
     */
    private ?int $baudrate;

    /**
     * @var int|null
     * @ORM\Column(type="integer", nullable=true)
     * @Groups({"semi_automatic_variable"})
     * @EnumType(class="Webapp\Core\Enumeration\BitFormatEnum", nullable=true)
     */
    private ?int $bitFormat;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"semi_automatic_variable"})
     * @EnumType(class="Webapp\Core\Enumeration\FlowControlEnum", nullable=true)
     */
    private ?string $flowControl;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"semi_automatic_variable"})
     * @EnumType(class="Webapp\Core\Enumeration\ParityEnum", nullable=true)
     */
    private ?string $parity;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"semi_automatic_variable"})
     * @EnumType(class="Webapp\Core\Enumeration\StopBitEnum", nullable=true)
     */
    private ?string $stopBit;

    /**
     * @var bool|null
     * @ORM\Column(type="boolean", nullable=true)
     * @Groups({"semi_automatic_variable"})
     */
    private ?bool $remoteControl;

    /**
     * @var Project|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Project", inversedBy="devices")
     * @Groups({"semi_automatic_variable"})
     */
    protected ?Project $project;


    public function __construct()
    {
        $this->managedVariables = new ArrayCollection();
        $this->project = null;
        $this->site = null;
        $this->bitFormat = null;
        $this->flowControl = null;
        $this->parity = null;
        $this->stopBit = null;
        $this->remoteControl = null;
    }

    /**
     * @return string
     */
    public function getAlias(): string
    {
        return $this->alias;
    }

    /**
     * @param string $alias
     * @return Device
     */
    public function setAlias(string $alias): Device
    {
        $this->alias = $alias;
        return $this;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Device
     */
    public function setName(string $name): Device
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getManufacturer(): string
    {
        return $this->manufacturer;
    }

    /**
     * @param string $manufacturer
     * @return Device
     */
    public function setManufacturer(string $manufacturer): Device
    {
        $this->manufacturer = $manufacturer;
        return $this;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @param string $type
     * @return Device
     */
    public function setType(string $type): Device
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return Site|null
     */
    public function getSite(): ?Site
    {
        return $this->site;
    }

    /**
     * @param Site|null $site
     * @return Device
     */
    public function setSite(?Site $site): Device
    {
        $this->site = $site;
        $this->project = null;
        return $this;
    }

    /**
     * @return Collection|SemiAutomaticVariable[]
     */
    public function getManagedVariables()
    {
        return $this->managedVariables;
    }

    /**
     * @param Collection|SemiAutomaticVariable[] $managedVariables
     * @return Device
     */
    public function setManagedVariables($managedVariables)
    {
        $this->managedVariables = $managedVariables;
        foreach ($managedVariables as $managedVariable) {
            $managedVariable->setDevice($this);
        }
        return $this;
    }

    /**
     * @param SemiAutomaticVariable $managedVariable
     * @return Device
     */
    public function addManagedVariable($managedVariable)
    {
        if (!$this->managedVariables->contains($managedVariable)) {
            $this->managedVariables->add($managedVariable);
            $managedVariable->setDevice($this);
        }
        return $this;
    }

    /**
     * @param SemiAutomaticVariable $managedVariable
     * @return Device
     */
    public function removeManagedVariable($managedVariable)
    {
        if ($this->managedVariables->contains($managedVariable)) {
            $this->managedVariables->removeElement($managedVariable);
            $managedVariable->setDevice(null);
        }
        return $this;
    }

    /**
     * @return string
     */
    public function getCommunicationProtocol(): string
    {
        return $this->communicationProtocol;
    }

    /**
     * @param string $communicationProtocol
     * @return Device
     */
    public function setCommunicationProtocol(string $communicationProtocol): Device
    {
        $this->communicationProtocol = $communicationProtocol;
        return $this;
    }

    /**
     * @return int
     */
    public function getFrameLength(): int
    {
        return $this->frameLength;
    }

    /**
     * @param int $frameLength
     * @return Device
     */
    public function setFrameLength(int $frameLength): Device
    {
        $this->frameLength = $frameLength;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getFrameStart(): ?string
    {
        return $this->frameStart;
    }

    /**
     * @param string|null $frameStart
     * @return Device
     */
    public function setFrameStart(?string $frameStart): Device
    {
        $this->frameStart = $frameStart;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getFrameEnd(): ?string
    {
        return $this->frameEnd;
    }

    /**
     * @param string|null $frameEnd
     * @return Device
     */
    public function setFrameEnd(?string $frameEnd): Device
    {
        $this->frameEnd = $frameEnd;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getFrameCsv(): ?string
    {
        return $this->frameCsv;
    }

    /**
     * @param string|null $frameCsv
     * @return Device
     */
    public function setFrameCsv(?string $frameCsv): Device
    {
        $this->frameCsv = $frameCsv;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getBaudrate(): ?int
    {
        return $this->baudrate;
    }

    /**
     * @param int|null $baudrate
     * @return Device
     */
    public function setBaudrate(?int $baudrate): Device
    {
        $this->baudrate = $baudrate;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getBitFormat(): ?int
    {
        return $this->bitFormat;
    }

    /**
     * @param int|null $bitFormat
     * @return Device
     */
    public function setBitFormat(?int $bitFormat): Device
    {
        $this->bitFormat = $bitFormat;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getFlowControl(): ?string
    {
        return $this->flowControl;
    }

    /**
     * @param string|null $flowControl
     * @return Device
     */
    public function setFlowControl(?string $flowControl): Device
    {
        $this->flowControl = $flowControl;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getParity(): ?string
    {
        return $this->parity;
    }

    /**
     * @param string|null $parity
     * @return Device
     */
    public function setParity(?string $parity): Device
    {
        $this->parity = $parity;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getStopBit(): ?string
    {
        return $this->stopBit;
    }

    /**
     * @param string|null $stopBit
     * @return Device
     */
    public function setStopBit(?string $stopBit): Device
    {
        $this->stopBit = $stopBit;
        return $this;
    }

    /**
     * @return bool|null
     */
    public function getRemoteControl(): ?bool
    {
        return $this->remoteControl;
    }

    /**
     * @param bool|null $remoteControl
     * @return Device
     */
    public function setRemoteControl(?bool $remoteControl): Device
    {
        $this->remoteControl = $remoteControl;
        return $this;
    }

    /**
     * @return Project|null
     */
    public function getProject(): ?Project
    {
        return $this->project;
    }

    /**
     * @param Project|null $project
     * @return Device
     */
    public function setProject(?Project $project): Device
    {
        $this->project = $project;
        $this->site = null;
        return $this;
    }
}
