<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Annotation\ApiSubresource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\ExistsFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\CustomFilters\DeletedFilter;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\Site;
use Shared\Authentication\Entity\User;
use Shared\Enumeration\Annotation\EnumType;
use Shared\RightManagement\Annotation\AdvancedRight;
use Shared\RightManagement\Traits\HasOwnerEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Webapp\Core\Annotation\GraphicallyDeletable;
use Webapp\Core\ApiOperation\RestoreObjectOperation;
use Webapp\Core\ApiOperation\UpdateIdentificationCodesOperation;
use Webapp\Core\Dto\BusinessObject\BusinessObjectCsvOutputDto;
use Webapp\Core\Dto\Experiment\ExperimentInputDto;
use Webapp\Core\Dto\Notes\NotesCsvOutputDto;
use Webapp\Core\Entity\Attachment\ExperimentAttachment;
use Webapp\Core\Enumeration\ExperimentStateEnum;
use Webapp\Core\Traits\GraphicallyDeletableEntity;
use Webapp\Core\Validator\Experiment\ExperimentPatchConstraint;
use Webapp\Core\Validator\UniqueAttributeInParent;

/**
 * Class Experiment
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *
 *     collectionOperations={
 *         "get"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER')"
 *          },
 *         "post"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())",
 *              "input"=ExperimentInputDto::class,
 *          },
 *     },
 *     itemOperations={
 *          "get"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER')"
 *          },
 *          "patch"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER')"
 *          },
 *          "delete"={
 *              "security"="object.getState() === 0"
 *          },
 *          "restore"={
 *              "controller"=RestoreObjectOperation::class,
 *              "method"="PATCH",
 *              "path"="/experiments/{id}/restore",
 *              "security"="is_granted('ROLE_SITE_ADMIN')",
 *              "read"=false,
 *              "validate"=false,
 *              "openapi_context"={
 *                  "summary": "Restore deleted protocol",
 *                  "description": "Remove the deleted state"
 *              },
 *          },
 *          "exportNotes"={
 *              "method"="GET",
 *              "path"="/experiments/{id}/notes/exportCsv",
 *              "formats"={"csv"={"text/csv"}},
 *              "pagination_enabled"=false,
 *              "output"=NotesCsvOutputDto::class
 *          },
 *          "export"={
 *              "method"="GET",
 *              "path"="/experiments/{id}/exportCsv",
 *              "formats"={"csv"={"text/csv"}},
 *              "pagination_enabled"=false,
 *              "output"=BusinessObjectCsvOutputDto::class
 *          }
 *     }
 * )
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"design_explorer_view", "platform_full_view", "admin_explorer_view", "note_view"}})
 * @ApiFilter(ExistsFilter::class, properties={"deletedAt"})
 * @ApiFilter(SearchFilter::class, properties={
 *     "site": "exact",
 *     "name": "exact",
 *     "platform": "exact",
 *     "id": "exact",
 *     "state": "exact",
 *     "projects": "exact",
 *     "protocol.name": "exact"
 * })
 * @ApiFilter(DeletedFilter::class)
 *
 * @AdvancedRight(classIdentifier="webapp_experiment", ownerField="owner", siteAttribute="site", parentFields={"platform"})
 *
 * @Gedmo\SoftDeleteable()
 * @GraphicallyDeletable()
 *
 * @ORM\Entity()
 * @ORM\Table(name="experiment", schema="webapp")
 * @ExperimentPatchConstraint
 */
class Experiment extends IdentifiedEntity implements BusinessObject
{
    use HasOwnerEntity;

    use SoftDeleteableEntity;

    use GraphicallyDeletableEntity;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     * @Assert\NotBlank
     * @UniqueAttributeInParent(parentsAttributes={"site.experiments", "platform.experiments"})
     * @Groups({"design_explorer_view", "platform_full_view", "project_explorer_view", "webapp_data_view", "admin_explorer_view", "change_report", "data_entry_synthesis", "platform_synthesis", "project_synthesis", "variable_synthesis"})
     */
    private string $name;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean")
     * @Groups({"platform_full_view", "platform_synthesis"})
     */
    private bool $individualUP;

    /**
     * @var string | null
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"platform_full_view"})
     */
    private ?string $comment;

    /**
     * @var DateTime $created
     *
     * @Gedmo\Timestampable(on="create")
     * @ORM\Column(type="datetime")
     * @Groups({"platform_full_view", "platform_synthesis"})
     */
    private DateTime $created;

    /**
     * @var DateTime|null $validated
     *
     * @Gedmo\Timestampable(on="change", field="state", value=1)
     * @ORM\Column(type="datetime", nullable=true)
     */
    private ?DateTime $validated;

    /**
     * @var Protocol
     *
     * @ORM\OneToOne(targetEntity="Webapp\Core\Entity\Protocol", mappedBy="experiment", cascade={"persist", "remove"}, orphanRemoval=true)
     * @Groups({"design_explorer_view", "platform_full_view", "platform_synthesis"})
     */
    private $protocol;

    /**
     * @var Site|null
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="experiments")
     */
    private ?Site $site;

    /**
     * @var ?User the owner of the entity
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\User")
     * @Groups({"design_explorer_view", "platform_full_view", "platform_synthesis"})
     */
    private ?User $owner;

    /**
     * @var ?Platform
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Platform", inversedBy="experiments")
     * @Groups({"webapp_data_view"})
     */
    private ?Platform $platform;

    /**
     * @var Collection | array<Block>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Block", cascade={"persist", "remove"}, mappedBy="experiment")
     * @Groups({"platform_full_view"})
     */
    private $blocks;

    /**
     * @var Collection | array<OutExperimentationZone>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\OutExperimentationZone", mappedBy="experiment", cascade={"persist", "remove"})
     * @Groups({"platform_full_view"})
     */
    private $outExperimentationZones;

    /**
     * @var Collection|array<Note>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Note", mappedBy="experimentTarget", cascade={"persist", "remove"}, orphanRemoval=true)
     * @Groups({"note_view"})
     * @ApiSubresource()
     */
    private $notes;

    /**
     * @var int|null
     * @ORM\Column(type="integer", nullable=true)
     * @Groups({"platform_full_view"})
     */
    private ?int $color;

    /**
     * @var int
     * @ORM\Column(type="integer")
     * @Groups({"design_explorer_view", "platform_full_view", "platform_synthesis"})
     * @EnumType(class="Webapp\Core\Enumeration\ExperimentStateEnum")
     */
    private int $state;

    /**
     * @var Collection<Project>|array<Project>
     * @Groups({"platform_synthesis"})
     * @ORM\ManyToMany(targetEntity="Webapp\Core\Entity\Project", mappedBy="experiments")
     */
    private $projects;

    /**
     * @var Collection<ExperimentAttachment>|array<ExperimentAttachment>
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Attachment\ExperimentAttachment", mappedBy="experiment", cascade={"persist", "remove"})
     */
    private $experimentAttachments;

    public function __construct()
    {
        $this->blocks = new ArrayCollection();
        $this->outExperimentationZones = new ArrayCollection();
        $this->notes = new ArrayCollection();
        $this->color = null;
        $this->validated = null;
        $this->state = ExperimentStateEnum::CREATED;
        $this->owner = null;
        $this->experimentAttachments = new ArrayCollection();
    }

    /**
     * @Groups({"platform_full_view", "design_explorer_view"})
     * @return mixed
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Experiment
     */
    public function setName(string $name): Experiment
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return bool
     */
    public function isIndividualUP(): bool
    {
        return $this->individualUP;
    }

    /**
     * @param bool $individualUP
     * @return Experiment
     */
    public function setIndividualUP(bool $individualUP): Experiment
    {
        $this->individualUP = $individualUP;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return Experiment
     */
    public function setComment(?string $comment): Experiment
    {
        $this->comment = $comment;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getCreated(): DateTime
    {
        return $this->created;
    }

    /**
     * @param DateTime $created
     * @return Experiment
     */
    public function setCreated(DateTime $created): Experiment
    {
        $this->created = $created;
        return $this;
    }

    /**
     * @return DateTime|null
     */
    public function getValidated(): ?DateTime
    {
        return $this->validated;
    }

    /**
     * @param DateTime|null $validated
     * @return Experiment
     */
    public function setValidated(?DateTime $validated): Experiment
    {
        $this->validated = $validated;
        return $this;
    }

    /**
     * @return Protocol
     */
    public function getProtocol(): Protocol
    {
        return $this->protocol;
    }

    /**
     * @param Protocol $protocol
     * @return Experiment
     */
    public function setProtocol(Protocol $protocol): Experiment
    {
        $this->protocol = $protocol;
        $protocol->setExperiment($this);
        return $this;
    }

    /**
     * @return Site|null
     */
    public function getSite(): ?Site
    {
        return $this->site;
    }

    /**
     * @param Site|null $site
     * @return Experiment
     */
    public function setSite(?Site $site): Experiment
    {
        $this->site = $site;
        $this->platform = null;
        return $this;
    }

    /**
     * @return User|null
     */
    public function getOwner(): ?User
    {
        return $this->owner;
    }

    /**
     * @param User|null $owner
     * @return Experiment
     */
    public function setOwner(?User $owner): Experiment
    {
        $this->owner = $owner;
        return $this;
    }

    /**
     * @return Platform|null
     */
    public function getPlatform(): ?Platform
    {
        return $this->platform;
    }

    /**
     * @param ?Platform $platform
     * @return Experiment
     */
    public function setPlatform(?Platform $platform): Experiment
    {
        $this->site = null;
        $this->platform = $platform;
        return $this;
    }

    /**
     * @return Collection|Block[]
     */
    public function getBlocks()
    {
        return $this->blocks;
    }

    /**
     * @param Collection|Block[] $blocks
     * @return Experiment
     */
    public function setBlocks($blocks)
    {
        $this->blocks = $blocks;
        foreach ($blocks as $block) {
            $block->setExperiment($this);
        }
        return $this;
    }

    /**
     * @param Block $block
     * @return Experiment
     */
    public function addBlocks($block)
    {
        if (!$this->blocks->contains($block)) {
            $this->blocks->add($block);
            $block->setExperiment($this);
        }
        return $this;
    }

    /**
     * @return Collection|Note[]
     */
    public function getNotes()
    {
        return $this->notes;
    }

    /**
     * @param Collection|Note[] $notes
     * @return Experiment
     */
    public function setNotes($notes)
    {
        $this->notes = $notes;
        return $this;
    }

    /**
     * @param Note $note
     * @return Experiment
     */
    public function addNote($note)
    {
        if (!$this->notes->contains($note)) {
            $this->notes->add($note);
            $note->setTarget($this);
        }
        return $this;
    }

    /**
     * @param Note $note
     * @return Experiment
     */
    public function removeNote($note)
    {
        if ($this->notes->contains($note)) {
            $this->notes->removeElement($note);
            $note->setTarget(null);
        }
        return $this;
    }

    /**
     * @return Collection|OutExperimentationZone[]
     */
    public function getOutExperimentationZones()
    {
        return $this->outExperimentationZones;
    }

    /**
     * @param OutExperimentationZone $outExperimentationZone
     * @return Experiment
     */
    public function addOutExperimentationZone($outExperimentationZone)
    {
        if (!$this->outExperimentationZones->contains($outExperimentationZone)) {
            $this->outExperimentationZones->add($outExperimentationZone);
            $outExperimentationZone->setExperiment($this);
        }
        return $this;
    }

    /**
     * @param OutExperimentationZone $outExperimentationZone
     * @return Experiment
     */
    public function removeOutExperimentationZone($outExperimentationZone)
    {
        if ($this->outExperimentationZones->contains($outExperimentationZone)) {
            $this->outExperimentationZones->removeElement($outExperimentationZone);
            $outExperimentationZone->setExperiment(null);
        }
        return $this;
    }

    /**
     * @return int|null
     */
    public function getColor(): ?int
    {
        return $this->color;
    }

    /**
     * @param int|null $color
     * @return Experiment
     */
    public function setColor(?int $color): Experiment
    {
        $this->color = $color;
        return $this;
    }

    /**
     * @return int
     */
    public function getState(): int
    {
        return $this->state;
    }

    /**
     * @param int $state
     * @return Experiment
     */
    public function setState(int $state): Experiment
    {
        $this->state = $state;
        return $this;
    }

    /**
     * @return Collection|Project[]
     */
    public function getProjects()
    {
        return $this->projects;
    }

    public function setDeletedAt(DateTime $deletedAt = null)
    {
        $this->deletedAt = $deletedAt;
        if ($deletedAt === null) {
            foreach ($this->children() as $child) {
                $child->setDeletedAt($deletedAt);
            }
            foreach ($this->getNotes() as $child) {
                $child->setDeletedAt($deletedAt);
            }
            $this->getProtocol()->setDeletedAt($deletedAt);
        }

        return $this;
    }

    /**
     * @return Collection|ExperimentAttachment[]
     */
    public function getExperimentAttachments()
    {
        return $this->experimentAttachments;
    }

    /**
     * @param Collection|ExperimentAttachment[] $experimentAttachments
     * @return Experiment
     */
    public function setExperimentAttachments($experimentAttachments)
    {
        $this->experimentAttachments = $experimentAttachments;
        return $this;
    }

    /**
     * @param ExperimentAttachment $experimentAttachment
     * @return Experiment
     */
    public function addExperimentAttachment($experimentAttachment)
    {
        if (!$this->experimentAttachments->contains($experimentAttachment)) {
            $this->experimentAttachments->add($experimentAttachment);
            $experimentAttachment->setExperiment($this);
        }
        return $this;
    }

    /**
     * @param ExperimentAttachment $experimentAttachment
     * @return Experiment
     */
    public function removeExperimentAttachment($experimentAttachment)
    {
        if ($this->experimentAttachments->contains($experimentAttachment)) {
            $this->experimentAttachments->removeElement($experimentAttachment);
            $experimentAttachment->setExperiment(null);
        }
        return $this;
    }


    function parent(): ?BusinessObject
    {
        return $this->platform;
    }

    function children(): array
    {
        return [...$this->blocks->toArray(), ...$this->outExperimentationZones->toArray()];
    }


}
