<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2021
 */

declare(strict_types=1);

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use DateTime;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\User;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Enumeration\PathLevelEnum;

/**
 * Class FieldMeasure.
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={}
 *     },
 *     itemOperations={
 *         "get"={},
 *         "patch"={}
 *     },
 *     normalizationContext={"groups"={"webapp_data_view", "id_read"}},
 *     denormalizationContext={"groups"={"note_edit"}}
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={"target": "exact"})
 *
 * @Gedmo\SoftDeleteable()
 *
 * @ORM\Entity()
 * @ORM\Table(name="note", schema="webapp")
 */
class Note extends IdentifiedEntity
{
    use SoftDeleteableEntity;

    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true)
     * @Groups({"webapp_data_view", "note_edit"})
     */
    private ?string $text;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean")
     * @Groups({"webapp_data_view", "note_edit"})
     */
    private bool $deleted;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime")
     * @Groups({"webapp_data_view"})
     */
    private DateTime $creationDate;

    /**
     * @var User
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\User")
     * @ORM\JoinColumn(nullable=false)
     */
    private User $creator;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Individual", inversedBy="notes")
     */
    private ?Individual $individualTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\UnitPlot", inversedBy="notes")
     */
    private ?UnitPlot $unitPlotTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SurfacicUnitPlot", inversedBy="notes")
     */
    private ?SurfacicUnitPlot $surfacicUnitPlotTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Block", inversedBy="notes")
     */
    private ?Block $blockTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SubBlock", inversedBy="notes")
     */
    private ?SubBlock $subBlockTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Experiment", inversedBy="notes")
     */
    private ?Experiment $experimentTarget;

    /**
     * @return Block|Experiment|Individual|SubBlock|UnitPlot|SurfacicUnitPlot|null
     */
    public function getTarget()
    {
        return $this->individualTarget ?? $this->unitPlotTarget ?? $this->subBlockTarget ?? $this->blockTarget ?? $this->experimentTarget ?? $this->surfacicUnitPlotTarget;
    }

    /**
     * @return string
     * @Groups({"webapp_data_view"})
     */
    public function getCreatorLogin(): string
    {
        return $this->creator->getUsername();
    }

    /**
     * @return string|null
     * @Groups({"webapp_data_view"})
     */
    public function getCreatorAvatar(): ?string
    {
        return $this->creator->getAvatar();
    }

    /**
     * @return string|null
     */
    public function getTargetType(): ?string
    {
        $target = $this->getTarget();
        if ($target instanceof Individual) {
            return PathLevelEnum::INDIVIDUAL;
        } elseif ($target instanceof UnitPlot) {
            return PathLevelEnum::UNIT_PLOT;
        } elseif ($target instanceof SubBlock) {
            return PathLevelEnum::SUB_BLOCK;
        } elseif ($target instanceof Block) {
            return PathLevelEnum::BLOCK;
        } elseif ($target instanceof Experiment) {
            return PathLevelEnum::EXPERIMENT;
        } elseif ($target instanceof SurfacicUnitPlot) {
            return PathLevelEnum::SURFACIC_UNIT_PLOT;
        } else {
            return null;
        }
    }

    /**
     * @param $target
     * @return Note
     */
    public function setTarget($target): Note
    {
        if ($target instanceof Individual) {
            $this->individualTarget = $target;
        } elseif ($target instanceof UnitPlot) {
            $this->unitPlotTarget = $target;
        } elseif ($target instanceof SubBlock) {
            $this->subBlockTarget = $target;
        } elseif ($target instanceof Block) {
            $this->blockTarget = $target;
        } elseif ($target instanceof Experiment) {
            $this->experimentTarget = $target;
        } elseif ($target instanceof SurfacicUnitPlot) {
            $this->surfacicUnitPlotTarget = $target;
        }
        return $this;
    }

    /**
     * @return string|null
     */
    public function getText(): ?string
    {
        return $this->text;
    }

    /**
     * @param string|null $text
     * @return Note
     */
    public function setText(?string $text): Note
    {
        $this->text = $text;
        return $this;
    }

    /**
     * @return bool
     */
    public function isDeleted(): bool
    {
        return $this->deleted;
    }

    /**
     * @param bool $deleted
     * @return Note
     */
    public function setDeleted(bool $deleted): Note
    {
        $this->deleted = $deleted;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getCreationDate(): DateTime
    {
        return $this->creationDate;
    }

    /**
     * @param DateTime $creationDate
     * @return Note
     */
    public function setCreationDate(DateTime $creationDate): Note
    {
        $this->creationDate = $creationDate;
        return $this;
    }

    /**
     * @return User
     */
    public function getCreator(): User
    {
        return $this->creator;
    }

    /**
     * @param User $creator
     * @return Note
     */
    public function setCreator(User $creator): Note
    {
        $this->creator = $creator;
        return $this;
    }
}
