<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class Treatment
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     itemOperations={
 *          "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER', object.getProtocol().getSite())"},
 *          "patch"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER', object.getProtocol().getSite())",
 *              "denormalization_context"={"groups"={"edit"}}
 *          }
 *     }
 * )
 * @ApiFilter(SearchFilter::class, properties={"protocol": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"design_explorer_view"}})
 *
 * @Gedmo\SoftDeleteable()
 *
 * @ORM\Entity()
 * @ORM\Table(name="treatment", schema="webapp")
 */
class Treatment extends IdentifiedEntity
{
    use SoftDeleteableEntity;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "edit", "change_report", "protocol_synthesis", "protocol_full_view", "data_view_item"})
     */
    private string $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "edit", "protocol_synthesis", "protocol_full_view", "data_view_item"})
     */
    private string $shortName;

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @Groups({"platform_full_view", "protocol_synthesis", "protocol_full_view"})
     */
    private int $repetitions;

    /**
     * @var Collection|array
     *
     * @ORM\ManyToMany(targetEntity="Webapp\Core\Entity\Modality", inversedBy="treatments")
     * @ORM\JoinTable(name="rel_treatment_modality", schema="webapp")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "protocol_synthesis", "protocol_full_view", "data_view_item"})
     */
    private $modalities;

    /**
     * @var Protocol
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Protocol", inversedBy="treatments")
     */
    private Protocol $protocol;

    public function __construct()
    {
        $this->modalities = new ArrayCollection();
    }

    /**
     * @Groups({"platform_full_view"})
     * @return mixed
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Treatment
     */
    public function setName(string $name): Treatment
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getShortName(): string
    {
        return $this->shortName;
    }

    /**
     * @param string $shortName
     * @return Treatment
     */
    public function setShortName(string $shortName): Treatment
    {
        $this->shortName = $shortName;
        return $this;
    }

    /**
     * @return int
     */
    public function getRepetitions(): int
    {
        return $this->repetitions;
    }

    /**
     * @param int $repetitions
     * @return Treatment
     */
    public function setRepetitions(int $repetitions): Treatment
    {
        $this->repetitions = $repetitions;
        return $this;
    }

    /**
     * @return array<Modality>|Collection<Modality>
     */
    public function getModalities()
    {
        return $this->modalities;
    }

    /**
     * @param array<Modality>|Collection<Modality> $modalities
     * @return Treatment
     */
    public function setModalities($modalities)
    {
        $this->modalities = $modalities;
        return $this;
    }


    /**
     * @param Modality $modality
     * @return Treatment
     */
    public function addModalities($modality)
    {
        if (!$this->modalities->contains($modality)) {
            $this->modalities->add($modality);
        }
        return $this;
    }

    /**
     * @param Modality $modality
     * @return Treatment
     */
    public function removeModalities($modality)
    {
        if ($this->modalities->contains($modality)) {
            $this->modalities->removeElement($modality);
        }
        return $this;
    }

    /**
     * @return Protocol
     */
    public function getProtocol(): Protocol
    {
        return $this->protocol;
    }

    /**
     * @param Protocol $protocol
     * @return Treatment
     */
    public function setProtocol(Protocol $protocol): Treatment
    {
        $this->protocol = $protocol;
        return $this;
    }

}
