<?php

declare(strict_types=1);

namespace Webapp\FileManagement\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\Mapping\Column;
use Shared\Authentication\Entity\Site;
use Shared\TransferSync\Entity\StatusDataEntry;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Vich\UploaderBundle\Mapping\Annotation as Vich;
use Webapp\FileManagement\ApiOperation\CreateParsingJobOperation;
use Webapp\FileManagement\Entity\Base\UserLinkedFileJob;
use Webapp\FileManagement\Service\ProjectFileInterface;

/**
 * Class ParsingJob.
 *
 * @ApiResource(
 *     normalizationContext={
 *         "groups"={"request_file_read", "user_linked_file_job_read"}
 *     },
 *     denormalizationContext={
 *         "groups"={"request_file_write"}
 *     },
 *     collectionOperations={
 *         "post"={
 *             "controller"=CreateParsingJobOperation::class,
 *             "deserialize"=false,
 *             "access_control"="is_granted('ROLE_USER')",
 *             "validation_groups"={"Default", "media_object_create"},
 *             "openapi_context"={
 *                 "requestBody"={
 *                     "content"={
 *                         "multipart/form-data"={
 *                             "schema"={
 *                                 "type"="object",
 *                                 "properties"={
 *                                     "file"={
 *                                         "type"="string",
 *                                         "format"="binary"
 *                                     },
 *                                 }
 *                             }
 *                         }
 *                     }
 *                 }
 *             }
 *         },
 *         "get" = {}
 *     },
 *     itemOperations={
 *         "get"={ "is_granted('ROLE_USER') && security"="object.getUser() == user" },
 *         "put"={ "is_granted('ROLE_USER') && security"="object.getUser() == user" },
 *         "delete" = { "security"= "is_granted('ROLE_USER') && object.getUser() == user" }
 *     },
 *     attributes={
 *          "order"={"id":"DESC"},
 *          "security"="is_granted('ROLE_USER')"
 *     }
 * )
 *
 *
 *
 * @ORM\Entity
 * @ORM\Table(name="parsing_job", schema="webapp")
 */
class ParsingJob extends UserLinkedFileJob implements ProjectFileInterface
{

    const TYPE_EXPERIMENT = "experiment";
    const TYPE_EXPERIMENT_CSV = "experiment_csv";
    const TYPE_PROTOCOL_CSV = "protocol_csv";
    const TYPE_DATA_ENTRY_CSV = "data_entry_csv";
    const TYPE_PATH_BASE_CSV = "path_base_csv";
    const TYPE_PLATFORM = "platform";
    const TYPE_PROJECT_DATA_ZIP = "project_data_zip";
    const TYPE_VARIABLE_COLLECTION = "variable_collection";

    /**
     * @Assert\NotNull(groups={"media_object_create"})
     *
     * @Vich\UploadableField(mapping="project_file", fileNameProperty="filePath")
     */
    protected $file;

    /**
     * @var string
     * @ORM\Column(type="string")
     */
    private string $objectType;

    /**
     * @var Site
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="parsingJobs")
     * @ORM\JoinColumn(nullable=false)
     */
    private Site $site;

    /**
     * @var string | null
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private ?string $csvBindings;

    /**
     * @var StatusDataEntry | null
     *
     * @ORM\ManyToOne(targetEntity="Shared\TransferSync\Entity\StatusDataEntry")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?StatusDataEntry $statusDataEntry;

    /**
     * @var array
     *
     * @Groups({"user_linked_file_job_read"})
     * @ORM\Column(type="json", nullable=true)
     */
    private array $errors;

    /**
     * @var string|null
     *
     * @Groups({"user_linked_file_job_read"})
     * @Column(type="string", nullable=true)
     */
    private ?string $objectIri;

    public function __construct()
    {
        $this->objectIri = null;
        $this->csvBindings = null;
        $this->statusDataEntry = null;
        $this->errors = [];
    }

    /**
     * @return string
     */
    public function getObjectType(): string
    {
        return $this->objectType;
    }

    /**
     * @param string $objectType
     * @return ParsingJob
     */
    public function setObjectType(string $objectType): ParsingJob
    {
        $this->objectType = $objectType;
        return $this;
    }

    /**
     * @return Site
     */
    public function getSite(): Site
    {
        return $this->site;
    }

    /**
     * @param Site $site
     * @return ParsingJob
     */
    public function setSite(Site $site): ParsingJob
    {
        $this->site = $site;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getObjectIri(): ?string
    {
        return $this->objectIri;
    }

    /**
     * @param string|null $objectIri
     * @return ParsingJob
     */
    public function setObjectIri(?string $objectIri): ParsingJob
    {
        $this->objectIri = $objectIri;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getCsvBindings(): ?string
    {
        return $this->csvBindings;
    }

    /**
     * @param string|null $csvBindings
     * @return ParsingJob
     */
    public function setCsvBindings(?string $csvBindings): ParsingJob
    {
        $this->csvBindings = $csvBindings;
        return $this;
    }

    /**
     * @return array
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @param array $errors
     * @return ParsingJob
     */
    public function setErrors(array $errors): ParsingJob
    {
        $this->errors = $errors;
        return $this;
    }

    /**
     * @return StatusDataEntry|null
     */
    public function getStatusDataEntry(): ?StatusDataEntry
    {
        return $this->statusDataEntry;
    }

    /**
     * @param StatusDataEntry|null $statusDataEntry
     * @return ParsingJob
     */
    public function setStatusDataEntry(?StatusDataEntry $statusDataEntry): ParsingJob
    {
        $this->statusDataEntry = $statusDataEntry;
        return $this;
    }

}
