<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Annotation\ApiSubresource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Annotation\GraphicallyDeletable;
use Webapp\Core\Traits\GraphicallyDeletableEntity;

/**
 * Class UnitPlot
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *          "get"={}
 *     },
 *     itemOperations={
 *          "get"={},
 *          "patch"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER')",
 *              "denormalization_context"={"groups"={"edit"}}
 *          },
 *     }
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={"block": "exact", "subBlock": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"design_explorer_view", "platform_full_view", "parent_view"}})
 *
 * @Gedmo\SoftDeleteable()
 * @GraphicallyDeletable()
 *
 * @ORM\Entity()
 * @ORM\Table(name="unit_plot", schema="webapp")
 */
class UnitPlot extends IdentifiedEntity implements BusinessObject
{
    use GraphicallyDeletableEntity;

    use SoftDeleteableEntity;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     * @Groups({"design_explorer_view", "platform_full_view", "webapp_data_view", "change_report", "fusion_result"})
     */
    private string $number;

    /**
     * @var Treatment
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Treatment")
     * @ORM\JoinColumn(nullable=false)
     * @Groups({"design_explorer_view", "webapp_data_view", "platform_full_view", "change_report", "edit"})
     */
    private Treatment $treatment;

    /**
     * @var Collection | array<Individual>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Individual", mappedBy="unitPlot", cascade={"persist", "remove"})
     * @Groups({"platform_full_view"})
     */
    private $individuals;

    /**
     * @var ?Block
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Block", inversedBy="unitPlots")
     * @Groups({"webapp_data_view", "parent_view", "change_report"})
     */
    private $block;

    /**
     * @var ?SubBlock
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SubBlock", inversedBy="unitPlots")
     * @Groups({"webapp_data_view", "parent_view", "change_report"})
     */
    private $subBlock;

    /**
     * @var Collection|array<Note>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Note", mappedBy="unitPlotTarget", cascade={"persist", "remove"}, orphanRemoval=true)
     * @Groups({"note_view"})
     * @ApiSubresource()
     */
    private $notes;

    /**
     * @var Collection | array<OutExperimentationZone>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\OutExperimentationZone", mappedBy="unitPlot", cascade={"persist", "remove"})
     * @Groups({"platform_full_view"})
     */
    private $outExperimentationZones;

    /**
     * @var int|null
     * @ORM\Column(type="integer", nullable=true)
     * @Groups({"platform_full_view"})
     */
    private ?int $color;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"platform_full_view", "edit"})
     */
    private ?string $comment;

    public function __construct()
    {
        $this->dead = false;
        $this->individuals = new ArrayCollection();
        $this->outExperimentationZones = new ArrayCollection();
        $this->notes = new ArrayCollection();
        $this->color = null;
        $this->comment = null;
    }

    /**
     * @Groups({"platform_full_view"})
     * @return mixed
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getNumber(): string
    {
        return $this->number;
    }

    /**
     * @param string $number
     * @return UnitPlot
     */
    public function setNumber(string $number): UnitPlot
    {
        $this->number = $number;
        return $this;
    }

    /**
     * @return Treatment
     */
    public function getTreatment(): Treatment
    {
        return $this->treatment;
    }

    /**
     * @param Treatment $treatment
     * @return UnitPlot
     */
    public function setTreatment(Treatment $treatment): UnitPlot
    {
        $this->treatment = $treatment;
        return $this;
    }

    /**
     * @return Collection|Individual[]
     */
    public function getIndividuals()
    {
        return $this->individuals;
    }

    /**
     * @param Individual $individual
     * @return UnitPlot
     */
    public function addIndividual($individual)
    {
        if(!$this->individuals->contains($individual)){
            $this->individuals->add($individual);
            $individual->setUnitPlot($this);
        }
        return $this;
    }

    /**
     * @param Collection|Individual[] $individuals
     * @return UnitPlot
     */
    public function setIndividuals($individuals)
    {
        $this->individuals = $individuals;
        foreach ($individuals as $individual){
            $individual->setUnitPlot($this);
        }
        return $this;
    }

    /**
     * @return Block|null
     */
    public function getBlock(): ?Block
    {
        return $this->block;
    }

    /**
     * @param Block|null $block
     * @return UnitPlot
     */
    public function setBlock(?Block $block): UnitPlot
    {
        $this->block = $block;
        return $this;
    }

    /**
     * @return SubBlock|null
     */
    public function getSubBlock(): ?SubBlock
    {
        return $this->subBlock;
    }

    /**
     * @param SubBlock|null $subBlock
     * @return UnitPlot
     */
    public function setSubBlock(?SubBlock $subBlock): UnitPlot
    {
        $this->subBlock = $subBlock;
        return $this;
    }

    /**
     * @return Collection|Note[]
     */
    public function getNotes()
    {
        return $this->notes;
    }

    /**
     * @param Collection|Note[] $notes
     * @return UnitPlot
     */
    public function setNotes($notes)
    {
        $this->notes = $notes;
        return $this;
    }

    /**
     * @param Note $note
     * @return UnitPlot
     */
    public function addNote($note)
    {
        if (!$this->notes->contains($note)) {
            $this->notes->add($note);
            $note->setTarget($this);
        }
        return $this;
    }

    /**
     * @param Note $note
     * @return UnitPlot
     */
    public function removeNote($note)
    {
        if ($this->notes->contains($note)) {
            $this->notes->removeElement($note);
            $note->setTarget(null);
        }
        return $this;
    }

    /**
     * @return Collection|OutExperimentationZone[]
     */
    public function getOutExperimentationZones()
    {
        return $this->outExperimentationZones;
    }

    /**
     * @param OutExperimentationZone $outExperimentationZone
     * @return UnitPlot
     */
    public function addOutExperimentationZone($outExperimentationZone)
    {
        if (!$this->outExperimentationZones->contains($outExperimentationZone)) {
            $this->outExperimentationZones->add($outExperimentationZone);
            $outExperimentationZone->setUnitPlot($this);
        }
        return $this;
    }

    /**
     * @param OutExperimentationZone $outExperimentationZone
     * @return UnitPlot
     */
    public function removeOutExperimentationZone($outExperimentationZone)
    {
        if ($this->outExperimentationZones->contains($outExperimentationZone)) {
            $this->outExperimentationZones->removeElement($outExperimentationZone);
            $outExperimentationZone->setUnitPlot(null);
        }
        return $this;
    }

    /**
     * @return int|null
     */
    public function getColor(): ?int
    {
        return $this->color;
    }

    /**
     * @param int|null $color
     * @return UnitPlot
     */
    public function setColor(?int $color): UnitPlot
    {
        $this->color = $color;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return UnitPlot
     */
    public function setComment(?string $comment): UnitPlot
    {
        $this->comment = $comment;
        return $this;
    }

    public function setDeletedAt(DateTime $deletedAt = null)
    {
        $this->deletedAt = $deletedAt;
        if($deletedAt === null){
            foreach ($this->children() as $child){
                $child->setDeletedAt($deletedAt);
            }
            foreach ($this->getNotes() as $child){
                $child->setDeletedAt($deletedAt);
            }
        }

        return $this;
    }

    function parent(): ?BusinessObject
    {
        return $this->subBlock ?? $this->block;
    }

    function children(): array
    {
        return [...$this->individuals->toArray(), ...$this->outExperimentationZones->toArray()];
    }
}
