<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2021
 */

declare(strict_types=1);

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\Site;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\ApiOperation\UpdateValueListOperation;

/**
 * Class ValueList
 * @package Webapp\Core\Entity
 * @ApiResource(
 *     collectionOperations={
 *         "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER')"},
 *         "post"={
 *              "security_post_denormalize"="is_granted('POST_VALUE_LIST', object)",
 *          },
 *     },
 *     itemOperations={
 *          "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER')"},
 *          "patch"={
 *              "security_post_denormalize"="is_granted('POST_VALUE_LIST', object)"
 *          },
 *          "update"={
 *              "controller"=UpdateValueListOperation::class,
 *              "method"="PATCH",
 *              "path"="value_lists/{id}/update",
 *              "security"="is_granted('ROLE_SITE_ADMIN')",
 *              "openapi_context"={
 *                  "summary": "Update project value list items",
 *                  "description": "Update the project value list with items in the provided list"
 *              },
 *          },
 *          "delete"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())"
 *          }
 *     }
 * )
 * @ApiFilter(SearchFilter::class, properties={"site": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"project_explorer_view"}})
 *
 * @ORM\Entity()
 * @ORM\Table(name="value_list", schema="webapp")
 */
class ValueList extends IdentifiedEntity
{

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"project_explorer_view", "simple_variable_get", "simple_variable_post", "data_explorer_view"})
     */
    private string $name;

    /**
     * @var Collection<ValueListItem>|ValueListItem[]
     * @Groups({"simple_variable_get", "simple_variable_post"})
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\ValueListItem", mappedBy="list", cascade={"persist", "remove"}, orphanRemoval=true)
     */
    private $values;

    /**
     * @var Site|null
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="valueLists")
     */
    private ?Site $site;

    /**
     * @var SimpleVariable|null
     *
     * @ORM\OneToOne(targetEntity="Webapp\Core\Entity\SimpleVariable", inversedBy="valueList")
     */
    private ?SimpleVariable $variable;

    public function __construct()
    {
        $this->values = new ArrayCollection();
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return ValueList
     */
    public function setName(string $name): ValueList
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return ValueListItem[]|Collection
     */
    public function getValues()
    {
        return $this->values;
    }

    /**
     * @param ValueListItem[]|Collection $values
     * @return ValueList
     */
    public function setValues($values): ValueList
    {
        $this->values = $values;
        foreach ($values as $value) {
            $value->setList($this);
        }
        return $this;
    }

    /**
     * @param ValueListItem $value
     * @return ValueList
     */
    public function addValue(ValueListItem $value): ValueList
    {
        if (!$this->values->contains($value)) {
            $this->values->add($value);
            $value->setList($this);
        }
        return $this;
    }

    /**
     * @param ValueListItem $value
     * @return ValueList
     */
    public function removeValue(ValueListItem $value): ValueList
    {
        if ($this->values->contains($value)) {
            $this->values->removeElement($value);
            $value->setList(null);
        }
        return $this;
    }

    /**
     * @return Site|null
     */
    public function getSite(): ?Site
    {
        return $this->site;
    }

    /**
     * @param Site|null $site
     * @return ValueList
     */
    public function setSite(?Site $site): ValueList
    {
        $this->site = $site;
        $this->variable = null;
        return $this;
    }

    /**
     * @return SimpleVariable|null
     */
    public function getVariable(): ?SimpleVariable
    {
        return $this->variable;
    }

    /**
     * @param SimpleVariable|null $variable
     * @return ValueList
     */
    public function setVariable(?SimpleVariable $variable): ValueList
    {
        $this->variable = $variable;
        $this->site = null;
        return $this;
    }

}
