<?php

namespace Webapp\FileManagement\Service;

use DateTime;
use DOMDocument;
use DOMElement;
use DOMNode;
use ReflectionException;
use Shared\Authentication\Entity\User;
use Webapp\Core\Entity\AbstractVariable;
use Webapp\Core\Entity\Annotation;
use Webapp\Core\Entity\Block;
use Webapp\Core\Entity\Device;
use Webapp\Core\Entity\Experiment;
use Webapp\Core\Entity\Factor;
use Webapp\Core\Entity\FieldGeneration;
use Webapp\Core\Entity\FieldMeasure;
use Webapp\Core\Entity\GeneratorVariable;
use Webapp\Core\Entity\Individual;
use Webapp\Core\Entity\Measure;
use Webapp\Core\Entity\Modality;
use Webapp\Core\Entity\Note;
use Webapp\Core\Entity\OezNature;
use Webapp\Core\Entity\OutExperimentationZone;
use Webapp\Core\Entity\Platform;
use Webapp\Core\Entity\Project;
use Webapp\Core\Entity\ProjectData;
use Webapp\Core\Entity\Protocol;
use Webapp\Core\Entity\SemiAutomaticVariable;
use Webapp\Core\Entity\Session;
use Webapp\Core\Entity\SimpleVariable;
use Webapp\Core\Entity\StateCode;
use Webapp\Core\Entity\SubBlock;
use Webapp\Core\Entity\SurfacicUnitPlot;
use Webapp\Core\Entity\Test;
use Webapp\Core\Entity\Treatment;
use Webapp\Core\Entity\UnitPlot;
use Webapp\Core\Entity\ValueList;
use Webapp\Core\Entity\VariableScale;
use Webapp\Core\Entity\VariableScaleItem;
use Webapp\Core\Enumeration\AnnotationTypeEnum;
use Webapp\Core\Enumeration\CommunicationProtocolEnum;
use Webapp\Core\Enumeration\ExperimentStateEnum;
use Webapp\Core\Enumeration\PathLevelEnum;
use Webapp\Core\Enumeration\SpreadingEnum;
use Webapp\Core\Enumeration\TestTypeEnum;
use Webapp\Core\Enumeration\VariableTypeEnum;

/**
 * Class WriterHelper.
 */
class WebappWriterHelper
{
    /**
     * @param DOMDocument $dom
     * @param User $user
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructConnectedUserElement(DOMDocument $dom, User $user, array &$uriMap): DOMElement
    {
        $desktopUserXml = $dom->createElement("adonis.modeleMetier.utilisateur:Utilisateur");
        $desktopUserXml->setAttribute("nom", $user->getSurname());
        $desktopUserXml->setAttribute("prenom", $user->getName());
        $desktopUserXml->setAttribute("email", $user->getEmail());
        $desktopUserXml->setAttribute("login", $user->getUsername());
        $desktopUserXml->setAttribute("motDePasse", "NO");

        return $desktopUserXml;
    }

    /**
     * @param DOMDocument $dom
     * @param StateCode $stateCode
     * @param array $uriMap
     * @param string $uri
     * @param bool $forDataEntryProject
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructStateCodeElement(DOMDocument $dom, StateCode $stateCode): DOMElement
    {
        $stateCodeXml = $dom->createElement("codesEtats");
        $stateCodeXml->setAttribute("code", $stateCode->getCode());
        $stateCodeXml->setAttribute("intitule", $stateCode->getTitle());
        $stateCodeXml->setAttribute("signification", $stateCode->getMeaning());
        switch ($stateCode->getSpreading()) {
            case SpreadingEnum::INDIVIDUAL:
                $stateCodeXml->setAttribute("propagation", "individu");
                break;
            case SpreadingEnum::UNIT_PLOT:
                $stateCodeXml->setAttribute("propagation", "parcelle");
                break;
        }
        $stateCodeXml->setAttribute("couleur", $this->transformColor($stateCode->getColor() ?? 1));

        return $stateCodeXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Platform $platform
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructPlateformElement(DOMDocument $dom, Platform $platform, array &$uriMap): DOMElement
    {
        $platformXml = $dom->createElement("adonis.modeleMetier.plateforme:Plateforme");
        $platformXml->setAttribute("nom", $platform->getName());
        $platformXml->setAttribute("dateCreation", $platform->getCreated()->format(DateTime::RFC3339_EXTENDED));
        $platformXml->setAttribute("nomSite", $platform->getSiteName());
        $platformXml->setAttribute("nomLieu", $platform->getPlaceName());

        $count = 0;
        foreach ($platform->getExperiments() as $experiment) {
            $uriMap[$experiment->getUri()] = $uriMap[$platform->getUri()] . "/@dispositifs." . $count;
            $variableXml = $this->constructExperimentElement($dom, $experiment, $uriMap);
            $platformXml->appendChild($variableXml);
            $count++;
        }

        return $platformXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Experiment $experiment
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructExperimentElement(DOMDocument $dom, Experiment $experiment, array &$uriMap, $exportExperimentOnly = false): DOMElement
    {
        $this->anomalies = [];
        $experimentXml = $dom->createElement($exportExperimentOnly ? "adonis.modeleMetier.plateforme:Dispositif" : "dispositifs");
        $experimentXml->setAttribute("nom", $experiment->getName());
        $experimentXml->setAttribute("dateCreation", $experiment->getCreated()->format(DateTime::RFC3339_EXTENDED));
        $experimentXml->setAttribute("source", $experiment->getProtocol()->getName());
        $experimentXml->setAttribute("puSurfacique", !$experiment->isIndividualUP() ? "true" : "false");

        if ($experiment->getValidated() !== null) {
            $experimentXml->setAttribute("dateValidation", $experiment->getValidated()->format(DateTime::RFC3339_EXTENDED));
        }
        switch ($experiment->getState()) {
            case ExperimentStateEnum::CREATED:
                $experimentXml->setAttribute("etat", "cree");
            case ExperimentStateEnum::VALIDATED:
                $experimentXml->setAttribute("etat", "valide");
            case ExperimentStateEnum::LOCKED:
                $experimentXml->setAttribute("etat", "verrouille");
            case ExperimentStateEnum::NON_UNLOCKABLE:
                $experimentXml->setAttribute("etat", "nonDeverrouillable");

        }


        $uriMap[$experiment->getProtocol()->getUri()] = $uriMap[$experiment->getUri()] . "/@protocoles";
        $protocolXml = $this->constructProtocolElement($dom, $experiment->getProtocol(), $uriMap);
        $experimentXml->appendChild($protocolXml);

        $count = 0;
        foreach ($experiment->getBlocks() as $block) {
            /** @var Block $block */
            $uriMap[$block->getUri()] = $uriMap[$experiment->getUri()] . "/@blocs." . $count;
            $protocolXml = $this->constructBlockElement($dom, $block, $uriMap);
            $experimentXml->appendChild($protocolXml);
            $count++;
        }
        $count = 0;
        foreach ($experiment->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $experimentXml->appendChild($outExperimentationZoneXml);
            $count++;
        }

        foreach ($experiment->getNotes() as $note) {
            $noteXml = $this->constructNoteElement($dom, $note);
            $experimentXml->appendChild($noteXml);
        }

        return $experimentXml;

    }

    /**
     * @param DOMDocument $dom
     * @param Protocol $protocol
     * @param array $uriMap
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructProtocolElement(DOMDocument $dom, Protocol $protocol, array &$uriMap): DOMElement
    {
        $protocolXml = $dom->createElement("protocoles");
        $protocolXml->setAttribute("nom", $protocol->getName());
        $protocolXml->setAttribute("dateCreation", $protocol->getCreated()->format(DateTime::RFC3339_EXTENDED));
        $protocolXml->setAttribute("objectifs", $protocol->getAim());
        $protocolXml->setAttribute("createur", $uriMap['connected_user']);
        if ($protocol->getAlgorithm() !== null) {
            $protocolXml->setAttribute("algorithmeTirage", $protocol->getAlgorithm()->getName());
        } else {
            $protocolXml->setAttribute("algorithmeTirage", "Sans Tirage");
        }


        $count = 0;
        foreach ($protocol->getFactors() as $factor) {
            /** @var Factor $factor */
            $uriMap[$factor->getUri()] = $uriMap[$protocol->getUri()] . "/@facteurs." . $count;
            $factorXml = $this->constructFactorElement($dom, $factor, $uriMap);
            $protocolXml->appendChild($factorXml);
            $count++;
        }
        $count = 0;
        foreach ($protocol->getTreatments() as $treatment) {
            /** @var Treatment $treatment */
            $uriMap[$treatment->getUri()] = $uriMap[$protocol->getUri()] . "/@traitements." . $count;
            $factorXml = $this->constructTreatmentElement($dom, $treatment, $uriMap);
            $protocolXml->appendChild($factorXml);
            $count++;
        }
        return $protocolXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Factor $factor
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructFactorElement(DOMDocument $dom, Factor $factor, array &$uriMap): DOMElement
    {
        $factorXml = $dom->createElement("facteurs");
        $factorXml->setAttribute("nom", $factor->getName());

        $count = 0;
        foreach ($factor->getModalities() as $modality) {
            /** @var Modality $modality */
            $uriMap[$modality->getUri()] = $uriMap[$factor->getUri()] . "/@modalites." . $count;
            $modalityXml = $this->constructModalityElement($dom, $modality, $uriMap);
            $factorXml->appendChild($modalityXml);
            $count++;
        }
        return $factorXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Modality $modality
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructModalityElement(DOMDocument $dom, Modality $modality, array &$uriMap): DOMElement
    {
        $modalityXml = $dom->createElement("modalites");
        $modalityXml->setAttribute("valeur", $modality->getValue());
        return $modalityXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Treatment $treatment
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructTreatmentElement(DOMDocument $dom, Treatment $treatment, array &$uriMap): DOMElement
    {
        $treatmentXml = $dom->createElement("traitements");
        $treatmentXml->setAttribute("nom", $treatment->getName());
        $treatmentXml->setAttribute("nomCourt", $treatment->getShortName());
        $treatmentXml->setAttribute("nbRepetitions", $treatment->getRepetitions());
        $str = "";
        foreach ($treatment->getModalities() as $modality) {
            /** @var Modality $modality */
            $str = $str . $uriMap[$modality->getUri()] . " ";
        }
        $treatmentXml->setAttribute("combinaisons", $str);

        return $treatmentXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Block $block
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructBlockElement(DOMDocument $dom, Block $block, array &$uriMap): DOMElement
    {
        $blockXml = $dom->createElement("blocs");
        $blockXml->setAttribute("numero", $block->getNumber());

        $count = 0;
        foreach ($block->getSubBlocks() as $subBlock) {
            /** @var SubBlock $subBlock */
            $uriMap[$subBlock->getUri()] = $uriMap[$block->getUri()] . '/@sousBlocs.' . $count;
            $subBlockXml = $this->constructSubBlockElement($dom, $subBlock, $uriMap);
            $blockXml->appendChild($subBlockXml);
            $count++;
        }
        $count = 0;
        foreach ($block->getUnitPlots() as $unitPlot) {
            $uriMap[$unitPlot->getUri()] = $uriMap[$block->getUri()] . "/@parcellesUnitaire." . $count;
            $unitPlotXml = $this->constructUnitPlotElement($dom, $unitPlot, $uriMap);
            $blockXml->appendChild($unitPlotXml);
            $count++;

        }
        foreach ($block->getSurfacicUnitPlots() as $unitPlot) {
            $uriMap[$unitPlot->getUri()] = $uriMap[$block->getUri()] . "/@parcellesUnitaire." . $count;
            $unitPlotXml = $this->constructSurfacicUnitPlotElement($dom, $unitPlot, $uriMap);
            $blockXml->appendChild($unitPlotXml);
            $count++;

        }
        foreach ($block->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $blockXml->appendChild($outExperimentationZoneXml);
            $count++;
        }
        foreach ($block->getNotes() as $note) {
            $noteXml = $this->constructNoteElement($dom, $note);
            $blockXml->appendChild($noteXml);
        }

        return $blockXml;
    }

    /**
     * @param DOMDocument $dom
     * @param SubBlock $subBlock
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructSubBlockElement(DOMDocument $dom, SubBlock $subBlock, array &$uriMap): DOMElement
    {
        $subBlockXml = $dom->createElement("sousBlocs");
        $subBlockXml->setAttribute("numero", $subBlock->getNumber());

        $count = 0;
        foreach ($subBlock->getUnitPlots() as $unitPlot) {
            $uriMap[$unitPlot->getUri()] = $uriMap[$subBlock->getUri()] . "/@parcellesUnitaire." . $count;
            $unitPlotXml = $this->constructUnitPlotElement($dom, $unitPlot, $uriMap);
            $subBlockXml->appendChild($unitPlotXml);
            $count++;
        }
        foreach ($subBlock->getSurfacicUnitPlots() as $unitPlot) {
            $uriMap[$unitPlot->getUri()] = $uriMap[$subBlock->getUri()] . "/@parcellesUnitaire." . $count;
            $unitPlotXml = $this->constructSurfacicUnitPlotElement($dom, $unitPlot, $uriMap);
            $subBlockXml->appendChild($unitPlotXml);
            $count++;
        }
        foreach ($subBlock->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $subBlockXml->appendChild($outExperimentationZoneXml);
            $count++;
        }

        foreach ($subBlock->getNotes() as $note) {
            $noteXml = $this->constructNoteElement($dom, $note);
            $subBlockXml->appendChild($noteXml);
        }

        return $subBlockXml;
    }

    /**
     * @param DOMDocument $dom
     * @param UnitPlot $unitPlot
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructUnitPlotElement(DOMDocument $dom, UnitPlot $unitPlot, array &$uriMap): DOMElement
    {
        $unitPlotXml = $dom->createElement("parcellesUnitaire");
        $unitPlotXml->setAttribute("xsi:type", "adonis.modeleMetier.plateforme:PuIndividuel");
        $unitPlotXml->setAttribute("nbIndividus", $unitPlot->getIndividuals()->count());

        $unitPlotXml->setAttribute("numero", $unitPlot->getNumber());
        $unitPlotXml->setAttribute("traitement", $uriMap[$unitPlot->getTreatment()->getUri()]);

        /*if ($unitParcel->getStateCode() != null) {
            $fakeTab = [];
            $stateCodeXml = $this->constructStateCodeElement($dom, $unitParcel->getStateCode(), $fakeTab, 0, false);
            $unitPlotXml->appendChild($stateCodeXml);
            $uriMap[$unitParcel->getUri() . $unitParcel->getStateCode()->getUri()] = $uriMap[$unitParcel->getUri()] . "/@codeEtat";
        }*/
        $count = 0;
        foreach ($unitPlot->getIndividuals() as $individual) {
            /** @var Individual $individual */
            $uriMap[$individual->getUri()] = $uriMap[$unitPlot->getUri()] . "/@individus." . $count;
            $individualXml = $this->constructIndividualElement($dom, $individual, $uriMap);
            $unitPlotXml->appendChild($individualXml);
            $count++;
        }
        foreach ($unitPlot->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $unitPlotXml->appendChild($outExperimentationZoneXml);
            $count++;
        }
        foreach ($unitPlot->getNotes() as $note) {
            $noteXml = $this->constructNoteElement($dom, $note);
            $unitPlotXml->appendChild($noteXml);
        }

        return $unitPlotXml;
    }

    /**
     * @param DOMDocument $dom
     * @param SurfacicUnitPlot $unitPlot
     * @param array $uriMap
     * @param string $uri
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructSurfacicUnitPlotElement(DOMDocument $dom, SurfacicUnitPlot $unitPlot, array &$uriMap): DOMElement
    {
        $unitPlotXml = $dom->createElement("parcellesUnitaire");

        $unitPlotXml->setAttribute("xsi:type", "adonis.modeleMetier.plateforme:PuSurfacique");
        if (!empty($unitPlot->getIdentifier())) {
            $unitPlotXml->setAttribute("idRfidCodeBarre", $unitPlot->getIdentifier());
        }

        $unitPlotXml->setAttribute("x", $unitPlot->getX());
        $unitPlotXml->setAttribute("y", $unitPlot->getY());
        $unitPlotXml->setAttribute("numero", $unitPlot->getNumber());
        $unitPlotXml->setAttribute("mort", $unitPlot->isDead() ? 'true' : 'false');
        if ($unitPlot->getDisappeared() !== null) {
            $unitPlotXml->setAttribute("dateDisparition", $unitPlot->getDisappeared()->format(DateTime::RFC3339_EXTENDED));
        }
        $unitPlotXml->setAttribute("traitement", $uriMap[$unitPlot->getTreatment()->getUri()]);
        $unitPlotXml->setAttribute("dateApparition", $unitPlot->getAppeared()->format(DateTime::RFC3339_EXTENDED));

        /*if ($unitParcel->getStateCode() != null) {
            $fakeTab = [];
            $stateCodeXml = $this->constructStateCodeElement($dom, $unitParcel->getStateCode(), $fakeTab, 0, false);
            $unitPlotXml->appendChild($stateCodeXml);
            $uriMap[$unitParcel->getUri() . $unitParcel->getStateCode()->getUri()] = $uriMap[$unitParcel->getUri()] . "/@codeEtat";
        }*/
        foreach ($unitPlot->getNotes() as $note) {
            $noteXml = $this->constructNoteElement($dom, $note);
            $unitPlotXml->appendChild($noteXml);
        }

        return $unitPlotXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Individual $individual
     * @param array $uriMap
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructIndividualElement(DOMDocument $dom, Individual $individual, array &$uriMap): DOMElement
    {
        $individualXml = $dom->createElement("individus");
        $individualXml->setAttribute("x", $individual->getX());
        $individualXml->setAttribute("y", $individual->getY());
        $individualXml->setAttribute("numero", $individual->getNumber());
        $individualXml->setAttribute("mort", $individual->isDead() ? 'true' : 'false');
        if ($individual->getDisappeared() !== null) {
            $individualXml->setAttribute("dateDisparition", $individual->getDisappeared()->format(DateTime::RFC3339_EXTENDED));
        }
        $individualXml->setAttribute("dateApparition", $individual->getAppeared()->format(DateTime::RFC3339_EXTENDED));
        if (!empty($individual->getIdentifier())) {
            $individualXml->setAttribute("idRfidCodeBarre", $individual->getIdentifier());
        }
        /*if ($individual->getStateCode() != null) {
            $fakeTab = [];
            $stateCodeXml = $this->constructStateCodeElement($dom, $individual->getStateCode(), $fakeTab, 0, false);
            $individualXml->appendChild($stateCodeXml);
            $uriMap[$individual->getUri() . $individual->getStateCode()->getUri()] = $uriMap[$individual->getUri()] . "/@codeEtat";
        }*/
        foreach ($individual->getNotes() as $note) {
            $noteXml = $this->constructNoteElement($dom, $note);
            $individualXml->appendChild($noteXml);
        }

        return $individualXml;
    }

    /**
     * @param DOMDocument $dom
     * @param OutExperimentationZone $outExperimentationZone
     * @param array $uriMap
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    private function constructOutExperimentationZoneElement(DOMDocument $dom, OutExperimentationZone $outExperimentationZone, array $uriMap): DOMElement
    {
        $outExperimentationZoneXml = $dom->createElement('zheAvecEpaisseurs');
        $outExperimentationZoneXml->setAttribute('x', $outExperimentationZone->getX());
        $outExperimentationZoneXml->setAttribute('y', $outExperimentationZone->getY());
        $outExperimentationZoneXml->setAttribute('numero', $outExperimentationZone->getNumber());
        $outExperimentationZoneXml->setAttribute('natureZhe', $uriMap[$outExperimentationZone->getNature()->getUri()]);
        return $outExperimentationZoneXml;
    }

    /**
     * @param DOMDocument $dom
     * @param OezNature $natureZHE
     * @param array $uriMap
     * @return DOMElement
     * @throws \DOMException
     */
    public function constructOezNatureElement(DOMDocument $dom, OezNature $natureZHE, array &$uriMap): DOMElement
    {
        $natureZHEXml = $dom->createElement('adonis.modeleMetier.plateforme:NatureZhe');
        $natureZHEXml->setAttribute('couleur', $this->transformColor($natureZHE->getColor()));
        $natureZHEXml->setAttribute('nom', $natureZHE->getNature());
        if ($natureZHE->getTexture() !== null) {
            $natureZHEXml->setAttribute('texture', $natureZHE->getTexture());
        }

        return $natureZHEXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Note $note
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    private function constructNoteElement(DOMDocument $dom, Note $note): DOMElement
    {
        $noteXml = $dom->createElement('notes');
        $noteXml->setAttribute("dateCreation", $note->getCreationDate()->format(DateTime::RFC3339_EXTENDED));
        $noteXml->setAttribute("texte", $note->getText());
        $noteXml->setAttribute("onlineUserUri", $note->getCreator()->getUri());
        $noteXml->setAttribute("loginCreateur", $note->getCreator()->getUsername());
        $noteXml->setAttribute("supprime", $note->isDeleted() ? 'true' : 'false');
        return $noteXml;
    }

    /**
     * @param DOMDocument $dom
     * @param ProjectData $projectData
     * @param $uriMap
     * @return DOMElement
     * @throws ReflectionException
     * @throws \DOMException
     */
    public function constructProjectDataElement(DOMDocument $dom, ProjectData $projectData, &$uriMap, &$variablesTuples, &$baseCount, &$desktopUsersXml): DOMElement
    {
        $dataEntryXml = $dom->createElement('adonis.modeleMetier.saisieTerrain:Saisie');
        $dataEntryXml->setAttribute('projetDeSaisie', $uriMap[$projectData->getProject()->getUri()]);
        $dataEntryXml->setAttribute('dateDebut', $projectData->getStart()->format(DateTime::RFC3339_EXTENDED));
        //$dataEntryXml->setAttribute('dateFin', $projectData->get()->format(DateTime::RFC3339_EXTENDED));
        $dataEntryXml->setAttribute('statusSaisie', 'terminee');
        $dataEntryXml->setAttribute('cheminementImpose', 'false');
        $dataEntryXml->setAttribute('plateforme', $uriMap[$projectData->getProject()->getPlatform()->getUri()]);

        $count = 0;
        foreach ($projectData->getSimpleVariables() as $simpleVariable) {
            $uriMap[$simpleVariable->getUri()] = $uriMap[$projectData->getUri()] . "/@variables." . $count++;
            $variableXml = $this->constructSimpleVariableElement($dom, $simpleVariable);
            $dataEntryXml->appendChild($variableXml);
            $variablesTuples[] = [$simpleVariable, $variableXml];
        }

        foreach ($projectData->getGeneratorVariables() as $generatorVariable) {
            $uriMap[$generatorVariable->getUri()] = $uriMap[$projectData->getUri()] . "/@variables." . $count++;
            $variableXml = $this->constructGeneratorVariableElement($dom, $generatorVariable, $uriMap, $variablesTuples);
            $dataEntryXml->appendChild($variableXml);
            $variablesTuples[] = [$generatorVariable, $variableXml];
        }


        $devices = [];
        foreach ($projectData->getSemiAutomaticVariables() as $semiAutomaticVariable) {
            $uriMap[$semiAutomaticVariable->getUri()] = $uriMap[$projectData->getUri()] . "/@variables." . $count++;
            $variableXml = $this->constructSemiAutomaticVariableElement($dom, $semiAutomaticVariable);
            $dataEntryXml->appendChild($variableXml);
            $variablesTuples[] = [$semiAutomaticVariable, $variableXml];
            $devices[$semiAutomaticVariable->getDevice()->getUri()] = $semiAutomaticVariable->getDevice();
        }

        $count = 0;
        foreach ($devices as $device) {
            $uriMap[$device->getUri()] = $uriMap[$projectData->getUri()] . "/@materiel." . $count++;
            $deviceXml = $this->constructDeviceElement($dom, $device, $uriMap);
            $dataEntryXml->appendChild($deviceXml);
        }

        $count = 0;
        foreach ($projectData->getSessions() as $session) {
            $uriMap[$session->getUri()] = $uriMap[$projectData->getUri()] . "/@sessions." . $count++;
            $sessionXml = $this->constructSessionElement($dom, $session, $uriMap, $baseCount, $desktopUsersXml);
            $dataEntryXml->appendChild($sessionXml);
        }

        return $dataEntryXml;
    }

    /**
     * @param DOMDocument $dom
     * @param AbstractVariable $variable
     * @param $uriMap
     * @param bool $isGenerated
     * @param bool $isDataEntry
     * @param ProjectData|null $connectedProjectData
     * @return DOMElement
     * @throws \DOMException
     */
    public function constructVariableElement(DOMDocument $dom, AbstractVariable $variable, bool $isGenerated = false): DOMElement
    {
        /*if ($variable->getProject() === null) {
            $variableXml = $dom->createElement("variablesConnectees");
            $variableXml->setAttribute("prechargee", 'true');
        } else {*/
        $variableXml = $dom->createElement($isGenerated ? "variablesGenerees" : "variables");
        //}
        $variableXml->setAttribute("nom", $variable->getName());
        $variableXml->setAttribute("nomCourt", $variable->getShortName());
        $variableXml->setAttribute("nbRepetitionSaisies", $variable->getRepetitions());
        $variableXml->setAttribute("unite", $variable->getUnit());
        $variableXml->setAttribute("horodatage", "true");
        $variableXml->setAttribute("dateCreation", $variable->getCreated()->format(DateTime::RFC3339_EXTENDED));
        $variableXml->setAttribute("dateDerniereModification", $variable->getLastModified()->format(DateTime::RFC3339_EXTENDED));
        $variableXml->setAttribute("commentaire", $variable->getComment());
        $variableXml->setAttribute("ordre", $variable->getOrder() ?? 1);
        $variableXml->setAttribute("saisieObligatoire", $variable->isMandatory() ? "true" : "false");
        //$variableXml->setAttribute("format", $variable->getFormat()); // TODO gérer le format de la variable
        switch ($variable->getPathLevel()) {
            case PathLevelEnum::INDIVIDUAL:
                $variableXml->setAttribute("uniteParcoursType", 'Individu');
                break;
            case PathLevelEnum::UNIT_PLOT:
            case PathLevelEnum::SURFACIC_UNIT_PLOT:
                //Just do nothing, default type
                //$variableXml->setAttribute("uniteParcoursType", 'parcelle');
                break;
            case PathLevelEnum::SUB_BLOCK:
                $variableXml->setAttribute("uniteParcoursType", 'SousBloc');
                break;
            case PathLevelEnum::BLOCK:
                $variableXml->setAttribute("uniteParcoursType", 'Bloc');
                break;
            case PathLevelEnum::EXPERIMENT:
                $variableXml->setAttribute("uniteParcoursType", 'Dispositif');
                break;
            case PathLevelEnum::PLATFORM:
                $variableXml->setAttribute("uniteParcoursType", 'Plateforme');
                break;
        }
        switch ($variable instanceof GeneratorVariable ? VariableTypeEnum::INTEGER : $variable->getType()) {
            case VariableTypeEnum::ALPHANUMERIC:
                $variableXml->setAttribute("typeVariable", "alphanumerique");
                break;
            case VariableTypeEnum::BOOLEAN:
                $variableXml->setAttribute("typeVariable", "booleen");
                break;
            case VariableTypeEnum::DATE:
                $variableXml->setAttribute("typeVariable", "date");
                break;
            case VariableTypeEnum::HOUR:
                $variableXml->setAttribute("typeVariable", "heure");
                break;
            case VariableTypeEnum::INTEGER:
                $variableXml->setAttribute("typeVariable", "entiere");
                break;
            case VariableTypeEnum::REAL:
                $variableXml->setAttribute("typeVariable", "reel");
                break;
            default:
                $variableXml->setAttribute("typeVariable", $variable->getType());
                break;
        }

        if ($variable->getDefaultTrueValue() !== null) {
            $variableXml->setAttribute("valeurDefaut", $variable->getDefaultTrueValue() ? 'true' : 'false');
        }

        return $variableXml;
    }

    public function constructSimpleVariableElement(DOMDocument $dom, SimpleVariable $variable, bool $isGenerated = false): DOMElement
    {
        return $this->constructVariableElement($dom, $variable, $isGenerated);
    }

    public function constructSemiAutomaticVariableElement(DOMDocument $dom, SemiAutomaticVariable $variable): DOMElement
    {
        $variableXml = $this->constructVariableElement($dom, $variable);
        $variableXml->setAttribute("debut", $variable->getStart());
        $variableXml->setAttribute("fin", $variable->getEnd());
        $variableXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:VariableSemiAuto");
        return $variableXml;
    }

    public function constructGeneratorVariableElement(DOMDocument $dom, GeneratorVariable $variable, &$uriMap, &$variablesTuples, bool $isGenerated = false): DOMElement
    {
        $variableXml = $this->constructVariableElement($dom, $variable, $isGenerated);
        $variableXml->setAttribute("nomGenere", $variable->getGeneratedPrefix());
        $variableXml->setAttribute("extension", $variable->isNumeralIncrement() ? "numerique" : "alphabetique");
        $variableXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:VariableGeneratrice");

        $count = 0;
        foreach ($variable->getGeneratedSimpleVariables() as $simpleVariable) {
            $uriMap[$simpleVariable->getUri()] = $uriMap[$variable->getUri()] . "/@variablesGenerees." . $count++;
            $variableGenereeXml = $this->constructSimpleVariableElement($dom, $simpleVariable, true);
            $variableXml->appendChild($variableGenereeXml);
            $variablesTuples[] = [$simpleVariable, $variableGenereeXml];
        }
        foreach ($variable->getGeneratedGeneratorVariables() as $generatedVariable) {
            $uriMap[$generatedVariable->getUri()] = $uriMap[$variable->getUri()] . "/@variablesGenerees." . $count++;
            $variableGenereeXml = $this->constructGeneratorVariableElement($dom, $generatedVariable, $uriMap, $variablesTuples, true);
            $variableXml->appendChild($variableGenereeXml);
            $variablesTuples[] = [$generatedVariable, $variableGenereeXml];
        }
        return $variableXml;
    }

    public function constructSessionElement(DOMDocument $dom, Session $session, &$uriMap, &$baseCount, &$desktopUsersXml): DOMElement
    {
        $sessionXml = $dom->createElement('sessions');
        $sessionXml->setAttribute('dateDebut', $session->getStartedAt()->format(DateTime::RFC3339_EXTENDED));
        $sessionXml->setAttribute('dateFin', $session->getEndedAt()->format(DateTime::RFC3339_EXTENDED));
        $sessionXml->setAttribute('isAuthenticated', "true");

        if (!isset($uriMap[$session->getUser()->getUri()])) {
            $userXml = $this->constructConnectedUserElement($dom, $session->getUser(), $uriMap);
            $uriMap[$session->getUser()->getUri()] = "/" . $baseCount++;
            $desktopUsersXml[] = $userXml;
        }

        $sessionXml->setAttribute('experimentateur', $uriMap[$session->getUser()->getUri()]);

        $count = 0;
        foreach ($session->getFieldMeasures() as $formField) {
            $this->constructMeasuresElement($dom, $formField, $uriMap, $count, $sessionXml, $session->getUri());
        }

        foreach ($session->getAnnotations() as $annotation) {
            $annotationXml = $this->constructAnnotationElement($dom, $annotation, $uriMap);
            $sessionXml->appendChild($annotationXml);
        }

        return $sessionXml;
    }

    public function constructAnnotationElement(DOMDocument $dom, Annotation $annotation, &$uriMap): DOMElement
    {
        $annotationXml = $dom->createElement("metadonnees");
        $annotationXml->setAttribute("nom", $annotation->getName());
        if ($annotation->getTargetType() === null) {
            $annotationXml->setAttribute('mesureVariable', $uriMap[$annotation->getTarget()->getUri()]);
        } else {
            $annotationXml->setAttribute('objetMetier', $uriMap[$annotation->getTarget()->getUri()]);
        }
        switch ($annotation->getType()) {
            case AnnotationTypeEnum::ANNOT_TYPE_TEXT:
                $annotationXml->setAttribute("nature", "texte");
                $annotationXml->setAttribute("donnee", $annotation->getValue());
                break;
        }
        $annotationXml->setAttribute("date", $annotation->getTimestamp()->format(DateTime::RFC3339_EXTENDED));
        foreach ($annotation->getCategories() as $category) {
            $categorieXml = $dom->createElement("categorie", $category);
            $annotationXml->appendChild($categorieXml);
        }

        foreach ($annotation->getKeywords() as $keyword) {
            $keywordXml = $dom->createElement("motscles", $keyword);
            $annotationXml->appendChild($keywordXml);
        }

        return $annotationXml;
    }

    public function constructMeasuresElement(DOMDocument $dom, FieldMeasure $formField, &$uriMap, &$count, DOMNode $sessionXml, string $sessionURI): void
    {
        foreach ($formField->getMeasures() as $measure) {
            /** @var Measure $measure */
            $measureXml = $this->constructMeasureElement($dom, $measure, $uriMap);
            $variable = $formField->getVariable();
            if ($variable instanceof GeneratorVariable) {
                $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableEntiere");
            } else {
                switch ($variable->getType()) {
                    case VariableTypeEnum::ALPHANUMERIC:
                        $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique");
                        break;
                    case VariableTypeEnum::REAL:
                        $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableReel");
                        if ($measure->getState() === null && $variable->getFormat() !== null && $measure->getValue() !== null) {
                            $measureXml->setAttribute("valeur", number_format(floatval($measure->getValue()), intval($variable->getFormat()), '.', ''));
                        }
                        break;
                    case VariableTypeEnum::BOOLEAN:
                        $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableBooleen");
                        break;
                    case VariableTypeEnum::INTEGER:
                        $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableEntiere");
                        break;
                    case VariableTypeEnum::DATE:
                        $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableDate");
                        break;
                    case VariableTypeEnum::HOUR:
                        $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableHeure");
                        break;
                }
            }
            $measureXml->setAttribute("variable", $uriMap[$variable->getUri()]);
            $measureXml->setAttribute("objetMetier", $uriMap[$formField->getTarget()->getUri()]);
            if ($formField->getFieldParent() !== null) {
                $measureXml->setAttribute("indiceGeneratrice", $formField->getFieldParent()->getIndex());
                $measureXml->setAttribute("mesureGeneratrice", $uriMap[$formField->getFieldParent()->getFormField()->getMeasures()->first()->getUri()]);
            }
            $uriMap[$measure->getUri()] = $uriMap[$sessionURI] . '/@mesureVariables.' . $count;
            $sessionXml->appendChild($measureXml);

            $count++;
        }
        if ($formField->getFieldGenerations()->count() > 0) {
            //In the generator variable case, only one measure is concerned by the formField
            /** @var DOMElement $measureXml */
            $str = "";
            foreach ($formField->getFieldGenerations() as $generatedField) {
                /** @var FieldGeneration $generatedField */
                foreach ($generatedField->getChildren() as $generatedFormField) {
                    /** @var FieldMeasure $generatedFormField */
                    $this->constructMeasuresElement($dom, $generatedFormField, $uriMap, $count, $sessionXml, $sessionURI);
                    foreach ($generatedFormField->getMeasures() as $generatedMeasure) {
                        /** @var Measure $generatedMeasure */
                        $str = $str . $uriMap[$generatedMeasure->getUri()] . " ";
                    }
                }
            }
            $measureXml->setAttribute("mesuresGenerees", $str);
        }
    }

    public function constructMeasureElement(DOMDocument $dom, Measure $measure, &$uriMap): DOMElement
    {
        $measureXml = $dom->createElement("mesureVariables");

        $measureXml->setAttribute("horodatage", $measure->getTimestamp()->format(DateTime::RFC3339_EXTENDED));
        if ($measure->getState() !== null) {
            $measureXml->setAttribute("codeEtat", $uriMap[$measure->getState()->getUri()]);
        } else {
            $measureXml->setAttribute("valeur", $measure->getValue());
        }
        return $measureXml;
    }

    public function constructProjectElement(DOMDocument $dom, Project $dataEntryProject, array &$uriMap, array &$variablesTuples): DOMElement
    {
        $dataEntryProjectXml = $dom->createElement("adonis.modeleMetier.projetDeSaisie:ProjetDeSaisie");
        $dataEntryProjectXml->setAttribute("nom", $dataEntryProject->getName());
        $dataEntryProjectXml->setAttribute("dateCreation", $dataEntryProject->getCreated()->format(DateTime::RFC3339_EXTENDED));
        $dataEntryProjectXml->setAttribute("createur", $uriMap['connected_user']);
        $dataEntryProjectXml->setAttribute("plateforme", $uriMap[$dataEntryProject->getPlatform()->getUri()]);
        $str = "";
        foreach ($dataEntryProject->getExperiments() as $experiment) {
            $str = $str . $uriMap[$experiment->getUri()] . " ";
        }
        $dataEntryProjectXml->setAttribute("dispositifs", $str);

        /*if ( $dataEntryProject->isImprovised()) { // Gérer les projets de saisie improvisés
            $dataEntryProjectXml->setAttribute("etat", "improvise");
        } else {*/
        $dataEntryProjectXml->setAttribute("etat", "transferer");
        //}

        $dataEntryProjectXml->setAttribute("experimentateurs", $uriMap['connected_user']);

        $count = 0;
        foreach ($dataEntryProject->getSimpleVariables() as $variable) {
            $uriMap[$variable->getUri()] = $uriMap[$dataEntryProject->getUri()] . "/@variables." . $count++;
            $variableXml = $this->constructSimpleVariableElement($dom, $variable);
            $dataEntryProjectXml->appendChild($variableXml);
            $variablesTuples[] = [$variable, $variableXml];
        }
        foreach ($dataEntryProject->getGeneratorVariables() as $variable) {
            $uriMap[$variable->getUri()] = $uriMap[$dataEntryProject->getUri()] . "/@variables." . $count++;
            $variableXml = $this->constructGeneratorVariableElement($dom, $variable, $uriMap, $variablesTuples);
            $dataEntryProjectXml->appendChild($variableXml);
            $variablesTuples[] = [$variable, $variableXml];
        }

        $deviceCount = 0;
        foreach ($dataEntryProject->getDevices() as $device) {
            foreach ($device->getManagedVariables() as $semiAutomaticVariable) {
                $uriMap[$semiAutomaticVariable->getUri()] = $uriMap[$dataEntryProject->getUri()] . "/@variables." . $count++;
                $variableXml = $this->constructSemiAutomaticVariableElement($dom, $semiAutomaticVariable);
                $dataEntryProjectXml->appendChild($variableXml);
                $variablesTuples[] = [$semiAutomaticVariable, $variableXml];
            }
            $uriMap[$device->getUri()] = $uriMap[$dataEntryProject->getUri()] . "/@materiel." . $deviceCount++;
            $deviceXml = $this->constructDeviceElement($dom, $device, $uriMap, $count);
            $dataEntryProjectXml->appendChild($deviceXml);
        }

        $count = 0;
        foreach ($dataEntryProject->getStateCodes() as $stateCode) {
            /** @var StateCode $stateCode */
            $uriMap[$stateCode->getUri()] = $uriMap[$dataEntryProject->getUri()] . "/@codesEtats." . $count++;
            $stateCodeXml = $this->constructStateCodeElement($dom, $stateCode, $uriMap);
            $dataEntryProjectXml->appendChild($stateCodeXml);
        }

        return $dataEntryProjectXml;

    }

    private function constructDeviceElement(DOMDocument $dom, Device $device, &$uriMap): DOMElement
    {
        $deviceXml = $dom->createElement("materiel");
        $deviceXml->setAttribute("nom", $device->getName());
        $deviceXml->setAttribute("fabriquant", $device->getManufacturer());
        $deviceXml->setAttribute("appareil", $device->getAlias());
        $deviceXml->setAttribute("type", $device->getType());

        $driverXml = $dom->createElement("driver");

        $driverXml->setAttribute("debutTrame", $device->getFrameStart());
        $driverXml->setAttribute("finTrame", $device->getFrameEnd());
        $driverXml->setAttribute("separateurCsv", $device->getFrameCsv());
        //$driverXml->setAttribute("timeout", $device->get());
        $driverXml->setAttribute("tailleTrame", $device->getFrameLength());
        if ($device->getCommunicationProtocol() !== CommunicationProtocolEnum::RS232) {
            $driverXml->setAttribute("type", $device->getCommunicationProtocol());
        } else {
            $driverXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:DriverCom");
            $driverXml->setAttribute("baudrate", $device->getBaudrate());
            $driverXml->setAttribute("stopbit", $device->getStopbit());
            $driverXml->setAttribute("push", $device->getRemoteControl() ? "true" : "false");
            $driverXml->setAttribute("parity", $device->getParity());
            //$driverXml->setAttribute("request", $device->get());
            $driverXml->setAttribute("flowcontrol", $device->getFlowControl());
            //$driverXml->setAttribute("port", $device->getPort());
            $driverXml->setAttribute("databitsFormat", $device->getBitFormat());
        }

        $deviceXml->appendChild($driverXml);

        $str = "";
        foreach ($device->getManagedVariables() as $semiAutomaticVariable) {
            $str = $str . $uriMap[$semiAutomaticVariable->getUri()] . " ";
        }
        $deviceXml->setAttribute("variables", $str);

        return $deviceXml;
    }

    public function completeVariableXml(DOMDocument $dom, array $uriMap, $variableTuple): void
    {
        /** @var AbstractVariable $variable */
        $variable = $variableTuple[0];
        /** @var DOMElement $variableXml */
        $variableXml = $variableTuple[1];
        foreach ($variable->getTests() as $test) {
            switch ($test->getType()) {
                case TestTypeEnum::COMBINATION:
                    $variableXml->appendChild($this->constructCombinationTestElement($dom, $test, $uriMap));
                    break;
                case TestTypeEnum::GROWTH:
                    $variableXml->appendChild($this->constructGrowthTestElement($dom, $test, $uriMap));
                    break;
                case TestTypeEnum::INTERVAL:
                    $variableXml->appendChild($this->constructRangeTestElement($dom, $test));
                    break;
                case TestTypeEnum::PRECONDITIONED:
                    $variableXml->appendChild($this->constructPreconditionedCalculationElement($dom, $test, $uriMap));
                    break;
            }
        }
        if ($variable instanceof SemiAutomaticVariable) {
            $variableXml->setAttribute("materiel", $uriMap[$variable->getDevice()->getUri()]);
        }
        if ($variable instanceof SimpleVariable) {
            if ($variable->getValueList() !== null) {
                $variableXml->appendChild($this->constructValueListElement($dom, $variable->getValueList()));
            }
            if ($variable->getScale() !== null) {
                $variableXml->appendChild($this->constructScaleElement($dom, $variable->getScale()));
            }
        }

    }

    public function constructCombinationTestElement(DOMDocument $dom, Test $test, $uriMap): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestCombinaisonEntreVariables");
        $testXml->setAttribute('operande1', $uriMap[$test->getCombinedVariable1()->getUri()]);
        $testXml->setAttribute('operande2', $uriMap[$test->getCombinedVariable2()->getUri()]);
        $testXml->setAttribute('nom', 'Test sur combinaison entre variables');
        $testXml->setAttribute('active', 'true');
        switch ($test->getCombinationOperation()) {
            case "-":
                $testXml->setAttribute('operateur', "soustraction");
                break;
            case "/":
                $testXml->setAttribute('operateur', "division");
                break;
            case "*":
                $testXml->setAttribute('operateur', "multiplication");
                break;
        }

        $limiteXml = $dom->createElement('limite');
        $limiteXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $limiteXml->setAttribute('minDouble', $test->getMinLimit());
        $limiteXml->setAttribute('maxDouble', $test->getMaxLimit());

        $testXml->appendChild($limiteXml);

        return $testXml;

    }

    public function constructGrowthTestElement(DOMDocument $dom, Test $test, $uriMap): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestsDeComparaison");
        $testXml->setAttribute('variableDeComparaison', $uriMap[$test->getComparedVariable()->getUri()]);
        $testXml->setAttribute('active', 'true');
        $testXml->setAttribute('nom', 'Test d\'accroissement');

        $ecartXml = $dom->createElement('ecart');
        $ecartXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $ecartXml->setAttribute('minDouble', $test->getMinGrowth());
        $ecartXml->setAttribute('maxDouble', $test->getMaxGrowth());

        $testXml->appendChild($ecartXml);

        return $testXml;
    }

    public function constructPreconditionedCalculationElement(DOMDocument $dom, Test $test, $uriMap): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestConditionnelEntreVariables");
        $testXml->setAttribute('active', 'true');
        $testXml->setAttribute('nom', 'Précalcul conditionnel');
        $testXml->setAttribute('condition', $test->getComparisonOperation());

        $testXml->setAttribute('variableAAffecter', $uriMap[$test->getVariable()->getUri()]);
        $testXml->setAttribute('variableDeComparaison', $uriMap[$test->getComparedVariable1()->getUri()]);

        $affectationValue = $dom->createElement('valeurAAffecter');
        $affectationValue->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique");
        $affectationValue->setAttribute('variable', $uriMap[$test->getVariable()->getUri()]);
        $affectationValue->setAttribute('valeur', $test->getAssignedValue());

        $testXml->appendChild($affectationValue);

        if ($test->getCompareWithVariable()) {
            $testXml->setAttribute('variableAComparer', $uriMap[$test->getComparedVariable2()->getUri()]);
        } else {
            $comparisonValue = $dom->createElement('valeurAComparer');
            $comparisonValue->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique");
            $comparisonValue->setAttribute('variable', $uriMap[$test->getVariable()->getUri()]);
            $comparisonValue->setAttribute('valeur', $test->getComparedValue());
            $testXml->appendChild($comparisonValue);
        }

        return $testXml;
    }

    public function constructRangeTestElement(DOMDocument $dom, Test $test): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestSurIntervalles");
        $testXml->setAttribute('active', 'true');
        $testXml->setAttribute('nom', 'Test sur intervalle');

        $limitXml = $dom->createElement('limiteOptionnel');
        $limitXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $limitXml->setAttribute('minDouble', $test->getProbIntervalMin());
        $limitXml->setAttribute('maxDouble', $test->getProbIntervalMax());

        $testXml->appendChild($limitXml);

        $limitXml = $dom->createElement('limiteObligatoire');
        $limitXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $limitXml->setAttribute('minDouble', $test->getAuthIntervalMin());
        $limitXml->setAttribute('maxDouble', $test->getAuthIntervalMax());

        $testXml->appendChild($limitXml);

        return $testXml;
    }

    public function constructScaleElement(DOMDocument $dom, VariableScale $scale): DOMElement
    {
        $scaleXml = $dom->createElement('echelle');

        $scaleXml->setAttribute("nom", $scale->getName());
        $scaleXml->setAttribute("valeurMin", $scale->getMinValue());
        $scaleXml->setAttribute("valeurMax", $scale->getMaxValue());
        $scaleXml->setAttribute("exclusif", !$scale->isOpen() ? 'true' : 'false');
        /** @var VariableScaleItem $mark */
        foreach ($scale->getValues() as $mark) {
            $markXml = $dom->createElement('notations');
            $markXml->setAttribute('texte', $mark->getText());
            $markXml->setAttribute('valeur', $mark->getValue());
            /*if($mark->getImage()){
                $this->markFiles[] = [$markXml, $mark->getImage()];
                $markXml->setAttribute('image', $mark->getImage());
            }*/
            $scaleXml->appendChild($markXml);
        }
        return $scaleXml;
    }

    public function constructValueListElement(DOMDocument $dom, ValueList $valueHintList): DOMElement
    {
        $valueHintListXml = $dom->createElement('listevaleur');

        $valueHintListXml->setAttribute("nom", $valueHintList->getName());
        foreach ($valueHintList->getValues() as $valueHint) {
            $valueHintXml = $dom->createElement('valeurs', $valueHint->getName());
            $valueHintListXml->appendChild($valueHintXml);
        }
        return $valueHintListXml;
    }

    private function transformColor(int $color): string
    {
        $b = $color % 256;
        $g = round((($color - $b) % 256 ** 2) / 256);
        $r = round(($color - $g - $b) / 256 ** 2);
        return "RGB{" . $r . "," . $g . "," . $b . "}";
    }
}
