<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2020
 */

declare(strict_types=1);

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use DateTime;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class Measure.
 *
 * @ApiResource(
 *     collectionOperations={
 *     },
 *     itemOperations={
 *          "get"={},
 *          "patch"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER')",
 *              "denormalization_context"={"groups"={"measure_edit"}}
 *          },
 *     }
 * )
 *
 * @ORM\Entity()
 * @ORM\Table(name="measure", schema="webapp")
 */
class Measure extends IdentifiedEntity
{

    use OpenSilexEntity;

    /**
     * @var FieldMeasure
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\FieldMeasure", inversedBy="measures")
     * @ORM\JoinColumn(nullable=false)
     */
    private FieldMeasure $formField;

    /**
     * @ORM\Column(type="text", nullable=true)
     * @Groups({"webapp_data_view", "measure_edit", "variable_synthesis", "fusion_result", "data_view_item", "graphical_measure_view"})
     */
    private ?string $value;

    /**
     * Define the repetition number of the variable.
     * @ORM\Column(type="integer", nullable=false)
     * @Groups({"webapp_data_view", "data_view_item"})
     */
    private int $repetition;

    /**
     * @ORM\Column(type="datetime", nullable=false)
     * @Groups({"webapp_data_view", "data_view_item"})
     */
    private DateTime $timestamp;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\StateCode")
     * @ORM\JoinColumn(nullable=true)
     * @Groups({"webapp_data_view", "measure_edit", "data_entry_synthesis", "variable_synthesis", "fusion_result", "data_view_item", "graphical_measure_view"})
     */
    private ?StateCode $state;

    /**
     * @var array<Annotation> | Collection
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Annotation", mappedBy="targetMeasure", cascade={"persist", "remove"})
     * @Groups({"webapp_data_view", "data_entry_synthesis", "data_view_item"})
     */
    private $annotations;

    public function __construct()
    {
        $this->openSilexInstance = null;
        $this->openSilexUri = null;
    }

    /**
     * @return FieldMeasure
     */
    public function getFormField(): FieldMeasure
    {
        return $this->formField;
    }

    /**
     * @param FieldMeasure $formField
     * @return Measure
     */
    public function setFormField(FieldMeasure $formField): Measure
    {
        $this->formField = $formField;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getValue(): ?string
    {
        return $this->value;
    }

    /**
     * @param string|null $value
     * @return Measure
     */
    public function setValue(?string $value): Measure
    {
        $this->value = $value;
        return $this;
    }

    /**
     * @return int
     */
    public function getRepetition(): int
    {
        return $this->repetition;
    }

    /**
     * @param int $repetition
     * @return Measure
     */
    public function setRepetition(int $repetition): Measure
    {
        $this->repetition = $repetition;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getTimestamp(): DateTime
    {
        return $this->timestamp;
    }

    /**
     * @param DateTime $timestamp
     * @return Measure
     */
    public function setTimestamp(DateTime $timestamp): Measure
    {
        $this->timestamp = $timestamp;
        return $this;
    }

    /**
     * @return StateCode|null
     *
     * @psalm-mutation-free
     */
    public function getState(): ?StateCode
    {
        return $this->state;
    }

    /**
     * @param StateCode|null $state
     * @return Measure
     */
    public function setState(?StateCode $state): Measure
    {
        $this->state = $state;
        return $this;
    }

    /**
     * @return Collection|Annotation[]
     */
    public function getAnnotations()
    {
        return $this->annotations;
    }

    /**
     * @param Collection|Annotation[] $annotations
     * @return Measure
     */
    public function setAnnotations($annotations)
    {
        $this->annotations = $annotations;
        return $this;
    }

    /**
     * @param Annotation $annotation
     * @return Measure
     */
    public function addAnnotation($annotation)
    {
        if (!$this->annotations->contains($annotation)) {
            $this->annotations->add($annotation);
            $annotation->setTarget($this);
        }
        return $this;
    }

    /**
     * @param Annotation $annotation
     * @return Measure
     */
    public function removeAnnotation($annotation)
    {
        if ($this->annotations->contains($annotation)) {
            $this->annotations->removeElement($annotation);
            $annotation->setTarget(null);
        }
        return $this;
    }

}
