<?php

namespace Mobile\Device\Entity;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Project\Entity\DesktopUser;
use Shared\Authentication\Entity\IdentifiedEntity;

/**
 * Class Protocol.
 *
 * @ORM\Entity(repositoryClass="Mobile\Device\Repository\ProtocolRepository")
 * @ORM\Table(name="protocol", schema="adonis")
 */
class Protocol extends IdentifiedEntity
{
    public const DEFAULT_PREFIX_FROM_CSV = "p-";
    public const DEFAULT_AIM_FROM_CSV = "";
    public const DEFAULT_ALGORITHM_FROM_CSV = "Sans Tirage";

    /**
     * @var Device
     *
     * @ORM\OneToOne(targetEntity="Mobile\Device\Entity\Device", inversedBy="protocol")
     * @ORM\JoinColumn(nullable=false, onDelete="CASCADE")
     */
    private $device;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $aim;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $algorithm;

    /**
     * @var Collection<array-key, Factor>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Device\Entity\Factor", mappedBy="protocol", cascade={"persist", "remove", "detach"})
     */
    private $factors;

    /**
     * @var Collection<array-key, Treatment>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Device\Entity\Treatment", mappedBy="protocol", cascade={"persist", "remove", "detach"})
     */
    private $treatments;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     */
    private $creationDate;

    /**
     * @var int | null
     *
     * @ORM\Column(type="integer", nullable=true, options={"default": null})
     */
    private $nbIndividualsPerParcel;

    /**
     * @var DesktopUser | null
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Project\Entity\DesktopUser")
     * @ORM\JoinColumn(nullable=true)
     */
    private $creator;

    /**
     * Protocol constructor.
     */
    public function __construct()
    {
        $this->factors = new ArrayCollection();
        $this->treatments = new ArrayCollection();
    }

    /**
     * @return Device
     */
    public function getDevice(): Device
    {
        return $this->device;
    }

    /**
     * @param Device $device
     * @return Protocol
     */
    public function setDevice(Device $device): Protocol
    {
        $this->device = $device;
        return $this;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Protocol
     */
    public function setName(string $name): Protocol
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getAim(): string
    {
        return $this->aim;
    }

    /**
     * @param string $aim
     * @return Protocol
     */
    public function setAim(string $aim): Protocol
    {
        $this->aim = $aim;
        return $this;
    }

    /**
     * @return string
     */
    public function getAlgorithm(): string
    {
        return $this->algorithm;
    }

    /**
     * @param string $algorithm
     * @return Protocol
     */
    public function setAlgorithm(string $algorithm): Protocol
    {
        $this->algorithm = $algorithm;
        return $this;
    }

    /**
     * @return Collection<array-key, Factor>
     */
    public function getFactors(): Collection
    {
        return $this->factors;
    }

    /**
     * @param Factor $factor
     */
    public function addFactor(Factor $factor): void
    {
        if (!$this->factors->contains($factor)) {
            $factor->setProtocol($this);
            $this->factors->add($factor);
        }
    }

    /**
     * @param Factor $factor
     */
    public function removeFactor(Factor $factor): void
    {
        if ($this->factors->contains($factor)) {
            $factor->setProtocol(null);
            $this->factors->removeElement($factor);
        }
    }

    /**
     * @return Collection<array-key, Treatment>
     */
    public function getTreatments(): Collection
    {
        return $this->treatments;
    }

    /**
     * @param Treatment $treatment
     */
    public function addTreatment(Treatment $treatment): void
    {
        if (!$this->treatments->contains($treatment)) {
            $treatment->setProtocol($this);
            $this->treatments->add($treatment);
        }
    }

    /**
     * @param Treatment $treatment
     */
    public function removeTreatment(Treatment $treatment): void
    {
        if ($this->treatments->contains($treatment)) {
            $treatment->setProtocol(null);
            $this->treatments->removeElement($treatment);
        }
    }

    /**
     * @return DateTime
     */
    public function getCreationDate(): DateTime
    {
        return $this->creationDate;
    }

    /**
     * @param DateTime $creationDate
     * @return Protocol
     */
    public function setCreationDate(DateTime $creationDate): Protocol
    {
        $this->creationDate = $creationDate;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getNbIndividualsPerParcel(): ?int
    {
        return $this->nbIndividualsPerParcel;
    }

    /**
     * @param int|null $nbIndividualsPerParcel
     * @return Protocol
     */
    public function setNbIndividualsPerParcel(?int $nbIndividualsPerParcel): Protocol
    {
        $this->nbIndividualsPerParcel = $nbIndividualsPerParcel;
        return $this;
    }

    /**
     * @return DesktopUser
     */
    public function getCreator(): ?DesktopUser
    {
        return $this->creator;
    }

    /**
     * @param DesktopUser|null $creator
     * @return Protocol
     */
    public function setCreator(?DesktopUser $creator): Protocol
    {
        $this->creator = $creator;
        return $this;
    }


}
