<?php

namespace Mobile\Measure\Entity\Variable\Base;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Measure\Entity\Test\Base\Test;
use Mobile\Measure\Entity\Test\CombinationTest;
use Mobile\Measure\Entity\Test\GrowthTest;
use Mobile\Measure\Entity\Test\PreconditionedCalculation;
use Mobile\Measure\Entity\Test\RangeTest;
use Mobile\Measure\Entity\Variable\GeneratorVariable;
use Mobile\Measure\Entity\Variable\Material;
use Mobile\Measure\Entity\Variable\PreviousValue;
use Mobile\Measure\Entity\Variable\Scale;
use Mobile\Project\Entity\DataEntryProject;
use Shared\Authentication\Entity\IdentifiedEntity;
use Webapp\Core\Enumeration\PathLevelEnum;
use Webapp\Core\Enumeration\VariableFormatEnum;
use Webapp\Core\Enumeration\VariableTypeEnum;

/**
 * Class Variable.
 *
 * @ORM\Entity()
 * @ORM\Table(name="variable", schema="adonis")
 * @ORM\InheritanceType("JOINED")
 * @ORM\DiscriminatorColumn(name="discr", type="string")
 * @ORM\DiscriminatorMap({
 *     "GeneratorVariable" = "Mobile\Measure\Entity\Variable\GeneratorVariable",
 *     "UniqueVariable" = "Mobile\Measure\Entity\Variable\UniqueVariable"
 * })
 *
 * @psalm-import-type PathLevelEnumId from PathLevelEnum
 * @psalm-import-type VariableFormatEnumId from VariableFormatEnum
 * @psalm-import-type VariableTypeEnumId from VariableTypeEnum
 */
abstract class Variable extends IdentifiedEntity
{
    /**
     * @var DataEntryProject|null
     */
    protected $project;

    /**
     * @var DataEntryProject|null
     */
    protected $connectedDataEntryProject;

    /**
     * @var GeneratorVariable|null
     */
    protected $generatorVariable;

    /**
     * @var Material|null
     */
    protected $material;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $shortName;

    /**
     * @var integer
     *
     * @ORM\Column(type="integer", nullable=false)
     */
    private $repetitions;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $unit;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", nullable=false)
     */
    private $askTimestamp;

    /**
     * @var string
     * @psalm-var PathLevelEnumId|''
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $pathLevel;

    /**
     * @var string|null
     *
     * @ORM\Column(type="text", nullable=true, options={"default": null})
     */
    private $comment;

    /**
     * @var integer
     *
     * @ORM\Column(type="integer", name="print_order", nullable=false)
     */
    private $order;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", nullable=false)
     */
    private $active;

    /**
     * @var string|null
     * @psalm-var VariableFormatEnumId|numeric-string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $format;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $defaultValue;

    /**
     * @var string
     * @psalm-var VariableTypeEnumId|''
     *
     * @ORM\Column(type="string", nullable=false)
     */
    private $type;

    /**
     * @var boolean
     *
     * @ORM\Column(type="boolean", nullable=false)
     */
    private $mandatory;

    /**
     * @var Collection<array-key, GrowthTest>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Test\GrowthTest", mappedBy="variable", cascade={"persist", "remove", "detach"})
     */
    private $growthTests;

    /**
     * @var Collection<array-key, RangeTest>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Test\RangeTest", mappedBy="variable", cascade={"persist", "remove", "detach"})
     */
    private $rangeTests;

    /**
     * @var Collection<array-key, CombinationTest>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Test\CombinationTest", mappedBy="variable", cascade={"persist", "remove", "detach"})
     */
    private $combinationTests;

    /**
     * @var Collection<array-key, PreconditionedCalculation>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Test\PreconditionedCalculation", mappedBy="variable", cascade={"persist", "remove", "detach"})
     */
    private $preconditionedCalculations;

    /**
     * @var Collection<array-key, PreviousValue>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\PreviousValue", mappedBy="variable", cascade={"persist", "remove", "detach"})
     */
    private $previousValues;

    /**
     * @var Scale|null
     *
     * @ORM\OneToOne(targetEntity="Mobile\Measure\Entity\Variable\Scale", mappedBy="variable", cascade={"persist", "remove", "detach"}, orphanRemoval=true)
     */
    private $scale;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     */
    private $creationDate;

    /**
     * @var DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     */
    private $modificationDate;

    /**
     * @var int|null
     *
     * @ORM\Column(type="integer", nullable=true, name="start_position")
     */
    private $frameStartPosition;

    /**
     * @var int|null
     *
     * @ORM\Column(type="integer", nullable=true, name="end_position")
     */
    private $frameEndPosition;

    /**
     * Variable constructor.
     */
    public function __construct()
    {
        $this->growthTests = new ArrayCollection();
        $this->rangeTests = new ArrayCollection();
        $this->combinationTests = new ArrayCollection();
        $this->preconditionedCalculations = new ArrayCollection();
        $this->previousValues = new ArrayCollection();
    }

    /**
     * @param Variable $variable
     */
    public function merge(Variable $variable)
    {
        $this->shortName = $variable->getShortName();
        $this->repetitions = $variable->getRepetitions();
        $this->unit = $variable->getUnit();
        $this->askTimestamp = $variable->isAskTimestamp();
        $this->pathLevel = $variable->getPathLevel();
        $this->comment = $variable->getComment();
        $this->order = $variable->getOrder();
        $this->active = $variable->isActive();
        $this->format = $variable->getFormat();
        $this->type = $variable->getType();
        $this->mandatory = $variable->isMandatory();
        $this->setScale($variable->getScale());
        $variable->setScale(null);
    }

    /**
     * @return string
     */
    public function getShortName(): string
    {
        return $this->shortName;
    }

    /**
     * @param string $shortName
     * @return static
     */
    public function setShortName(string $shortName): self
    {
        $this->shortName = $shortName;
        return $this;
    }

    /**
     * @return int
     */
    public function getRepetitions(): int
    {
        return $this->repetitions;
    }

    /**
     * @param int $repetitions
     * @return static
     */
    public function setRepetitions(int $repetitions): self
    {
        $this->repetitions = $repetitions;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getUnit(): ?string
    {
        return $this->unit;
    }

    /**
     * @param string|null $unit
     * @return static
     */
    public function setUnit(?string $unit): self
    {
        $this->unit = $unit;
        return $this;
    }

    /**
     * @return bool
     */
    public function isAskTimestamp(): bool
    {
        return $this->askTimestamp;
    }

    /**
     * @param bool $askTimestamp
     * @return static
     */
    public function setAskTimestamp(bool $askTimestamp): self
    {
        $this->askTimestamp = $askTimestamp;
        return $this;
    }

    /**
     * @return string
     * @psalm-return PathLevelEnumId|''
     */
    public function getPathLevel(): string
    {
        return $this->pathLevel;
    }

    /**
     * @param string $pathLevel
     * @psalm-param PathLevelEnumId|'' $pathLevel
     * @return static
     */
    public function setPathLevel(string $pathLevel): self
    {
        $this->pathLevel = $pathLevel;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return static
     */
    public function setComment(?string $comment): self
    {
        $this->comment = $comment;
        return $this;
    }

    /**
     * @return int
     */
    public function getOrder(): int
    {
        return $this->order;
    }

    /**
     * @param int $order
     * @return static
     */
    public function setOrder(int $order): self
    {
        $this->order = $order;
        return $this;
    }

    /**
     * @return bool
     */
    public function isActive(): bool
    {
        return $this->active;
    }

    /**
     * @param bool $active
     * @return static
     */
    public function setActive(bool $active): self
    {
        $this->active = $active;
        return $this;
    }

    /**
     * @return string|null
     * @psalm-return VariableFormatEnumId|numeric-string|null
     */
    public function getFormat(): ?string
    {
        return $this->format;
    }

    /**
     * @param string|null $format
     * @psalm-param VariableFormatEnumId|numeric-string|null $format
     * @return static
     */
    public function setFormat(?string $format): self
    {
        $this->format = $format;
        return $this;
    }

    /**
     * @return string
     * @psalm-return VariableTypeEnumId|''
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @param string $type
     * @psalm-param VariableTypeEnumId|'' $type
     * @return static
     */
    public function setType(string $type): self
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return bool
     */
    public function isMandatory(): bool
    {
        return $this->mandatory;
    }

    /**
     * @param bool $mandatory
     * @return static
     */
    public function setMandatory(bool $mandatory): self
    {
        $this->mandatory = $mandatory;
        return $this;
    }

    /**
     * @return Scale|null
     */
    public function getScale(): ?Scale
    {
        return $this->scale;
    }

    /**
     * @param Scale|null $scale
     * @return static
     */
    public function setScale(?Scale $scale): self
    {
        if ($this->scale !== null) {
            $this->scale->setVariable(null);
        }
        $this->scale = $scale;
        if ($scale !== null) {
            $scale->setVariable($this);
        }
        return $this;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return static
     */
    public function setName(string $name): self
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getDefaultValue(): ?string
    {
        return $this->defaultValue;
    }

    /**
     * @param string|null $defaultValue
     * @return static
     */
    public function setDefaultValue(?string $defaultValue): self
    {
        $this->defaultValue = $defaultValue;
        return $this;
    }

    /**
     * @return DataEntryProject|null
     */
    public function getProject(): ?DataEntryProject
    {
        return $this->project;
    }

    /**
     * @param DataEntryProject|null $project
     * @return static
     */
    public function setProject(?DataEntryProject $project): self
    {
        $this->project = $project;
        return $this;
    }

    /**
     * @return GeneratorVariable|null
     */
    public function getGeneratorVariable(): ?GeneratorVariable
    {
        return $this->generatorVariable;
    }

    /**
     * @param GeneratorVariable|null $generatorVariable
     * @return static
     */
    public function setGeneratorVariable(?GeneratorVariable $generatorVariable): self
    {
        $this->generatorVariable = $generatorVariable;
        return $this;
    }

    /**
     * @return Collection<array-key, GrowthTest>
     */
    public function getGrowthTests(): Collection
    {
        return $this->growthTests;
    }

    /**
     * @param GrowthTest $test
     */
    public function removeGrowthTest(GrowthTest $test): void
    {
        if ($this->growthTests->contains($test)) {
            $this->growthTests->removeElement($test);
        }
    }

    /**
     * @return Collection<array-key, RangeTest>
     */
    public function getRangeTests(): Collection
    {
        return $this->rangeTests;
    }

    /**
     * @param RangeTest $test
     */
    public function removeRangeTest(RangeTest $test): void
    {
        if ($this->rangeTests->contains($test)) {
            $this->rangeTests->removeElement($test);
        }
    }

    /**
     * @return Collection<array-key, CombinationTest>
     */
    public function getCombinationTests(): Collection
    {
        return $this->combinationTests;
    }

    /**
     * @param CombinationTest $test
     */
    public function removeCombinationTest(CombinationTest $test): void
    {
        if ($this->combinationTests->contains($test)) {
            $this->combinationTests->removeElement($test);
        }
    }

    /**
     * @return Collection<array-key, PreconditionedCalculation>
     */
    public function getPreconditionedCalculations(): Collection
    {
        return $this->preconditionedCalculations;
    }

    /**
     * @param PreconditionedCalculation $test
     */
    public function removePreconditionedCalculation(PreconditionedCalculation $test): void
    {
        if ($this->preconditionedCalculations->contains($test)) {
            $this->preconditionedCalculations->removeElement($test);
        }
    }

    /**
     * @param Test $test
     */
    public function addTest(Test $test): void
    {
        if ($test instanceof GrowthTest) {
            $this->addGrowthTest($test);
        } else if ($test instanceof RangeTest) {
            $this->addRangeTest($test);
        } else if ($test instanceof CombinationTest) {
            $this->addCombinationTest($test);
        } else if ($test instanceof PreconditionedCalculation) {
            $this->addPreconditionedCalculation($test);
        }
    }

    /**
     * @param GrowthTest $test
     */
    public function addGrowthTest(GrowthTest $test): void
    {
        if (!$this->growthTests->contains($test)) {
            $this->growthTests->add($test);
            $test->setVariable($this);
        }
    }

    /**
     * @param RangeTest $test
     */
    public function addRangeTest(RangeTest $test): void
    {
        if (!$this->rangeTests->contains($test)) {
            $this->rangeTests->add($test);
            $test->setVariable($this);
        }
    }

    /**
     * @param CombinationTest $test
     */
    public function addCombinationTest(CombinationTest $test): void
    {
        if (!$this->combinationTests->contains($test)) {
            $this->combinationTests->add($test);
            $test->setVariable($this);
        }
    }

    /**
     * @param PreconditionedCalculation $test
     */
    public function addPreconditionedCalculation(PreconditionedCalculation $test): void
    {
        if (!$this->preconditionedCalculations->contains($test)) {
            $this->preconditionedCalculations->add($test);
            $test->setVariable($this);
        }
    }

    /**
     * @return Collection<array-key, PreviousValue>
     */
    public function getPreviousValues(): Collection
    {
        return $this->previousValues;
    }

    /**
     * @param PreviousValue $previousValue
     */
    public function addPreviousValue(PreviousValue $previousValue): void
    {
        if (!$this->previousValues->contains($previousValue)) {
            $previousValue->setVariable($this);
            $this->previousValues->add($previousValue);
        }
    }

    /**
     * @param PreviousValue $previousValue
     */
    public function removePreviousValue(PreviousValue $previousValue): void
    {
        if ($this->previousValues->contains($previousValue)) {
            $this->previousValues->removeElement($previousValue);
        }
    }

    /**
     * @return DataEntryProject|null
     */
    public function getConnectedDataEntryProject(): ?DataEntryProject
    {
        return $this->connectedDataEntryProject;
    }

    /**
     * @param DataEntryProject|null $connectedDataEntryProject
     */
    public function setConnectedDataEntryProject(?DataEntryProject $connectedDataEntryProject): void
    {
        $this->connectedDataEntryProject = $connectedDataEntryProject;
    }

    /**
     * @return DateTime
     */
    public function getCreationDate(): DateTime
    {
        return $this->creationDate;
    }

    /**
     * @param DateTime $creationDate
     * @return Variable
     */
    public function setCreationDate(DateTime $creationDate): Variable
    {
        $this->creationDate = $creationDate;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getModificationDate(): DateTime
    {
        return $this->modificationDate;
    }

    /**
     * @param DateTime $modificationDate
     * @return Variable
     */
    public function setModificationDate(DateTime $modificationDate): Variable
    {
        $this->modificationDate = $modificationDate;
        return $this;
    }

    /**
     * @return Material|null
     */
    public function getMaterial(): ?Material
    {
        return $this->material;
    }

    /**
     * @param Material|null $material
     * @return Variable
     */
    public function setMaterial(?Material $material): Variable
    {
        $this->material = $material;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getFrameStartPosition(): ?int
    {
        return $this->frameStartPosition;
    }

    /**
     * @param int|null $frameStartPosition
     * @return Variable
     */
    public function setFrameStartPosition(?int $frameStartPosition): Variable
    {
        $this->frameStartPosition = $frameStartPosition;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getFrameEndPosition(): ?int
    {
        return $this->frameEndPosition;
    }

    /**
     * @param int|null $frameEndPosition
     * @return Variable
     */
    public function setFrameEndPosition(?int $frameEndPosition): Variable
    {
        $this->frameEndPosition = $frameEndPosition;
        return $this;
    }

}
