<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2021
 */

declare(strict_types=1);

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\CustomFilters\OrSearchFilter;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Enumeration\PathLevelEnum;

/**
 * Class FieldMeasure.
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={}
 *     },
 *     itemOperations={
 *         "get"={}
 *     }
 * )
 *
 * @ApiFilter(OrSearchFilter::class, properties={"search_variable"={"generatorVariable": "exact", "simpleVariable": "exact", "semiAutomaticVariable": "exact"}})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"graphical_measure_view"}})
 *
 * @ORM\Entity()
 * @ORM\Table(name="field_measure", schema="webapp")
 */
class FieldMeasure extends IdentifiedEntity
{
    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Session", inversedBy="fieldMeasures")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?Session $session;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\FieldGeneration", inversedBy="children")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?FieldGeneration $fieldParent;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\GeneratorVariable", cascade={"persist"}, fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?GeneratorVariable $generatorVariable;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SimpleVariable", cascade={"persist"}, fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?SimpleVariable $simpleVariable;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SemiAutomaticVariable", cascade={"persist"}, fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?SemiAutomaticVariable $semiAutomaticVariable;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Individual", fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?Individual $individualTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\UnitPlot", fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?UnitPlot $unitPlotTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SurfacicUnitPlot", fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?SurfacicUnitPlot $surfacicUnitPlotTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Block", fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?Block $blockTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SubBlock", fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?SubBlock $subBlockTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Experiment", fetch="EAGER")
     * @ORM\JoinColumn(nullable=true)
     */
    private ?Experiment $experimentTarget;

    /**
     * @var Collection<array-key, Measure>|Measure[]
     * @Groups({"webapp_data_view", "data_entry_synthesis", "variable_synthesis", "fusion_result", "data_view_item", "graphical_measure_view"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Measure", mappedBy="formField", cascade={"persist", "remove", "detach"}, fetch="EAGER")
     */
    private $measures;

    /**
     * @var Collection<array-key, FieldGeneration>|FieldGeneration[]
     * @Groups({"webapp_data_view", "data_view_item"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\FieldGeneration", mappedBy="formField", cascade={"persist", "remove", "detach"}, fetch="EAGER")
     */
    private $fieldGenerations;

    /**
     * FieldMeasure constructor.
     */
    public function __construct()
    {
        $this->measures = new ArrayCollection();
        $this->fieldGenerations = new ArrayCollection();
    }

    /**
     * @Groups({"webapp_data_view", "data_entry_synthesis", "fusion_result", "data_view_item", "graphical_measure_view"})
     * @return AbstractVariable
     */
    public function getVariable(): AbstractVariable
    {
        return $this->simpleVariable ?? $this->generatorVariable ?? $this->semiAutomaticVariable;
    }

    /**
     * @Groups({"webapp_data_view", "data_entry_synthesis", "variable_synthesis", "fusion_result", "graphical_measure_view"})
     * @return Block|Experiment|Individual|SubBlock|UnitPlot|SurfacicUnitPlot|null
     *
     * @psalm-mutation-free
     */
    public function getTarget()
    {
        return $this->individualTarget ?? $this->unitPlotTarget ?? $this->subBlockTarget ?? $this->blockTarget ?? $this->experimentTarget ?? $this->surfacicUnitPlotTarget;
    }

    /**
     * @Groups({"webapp_data_view", "data_entry_synthesis", "fusion_result"})
     * @return string|null
     */
    public function getTargetType(): ?string
    {
        $target = $this->getTarget();
        if ($target instanceof Individual) {
            return PathLevelEnum::INDIVIDUAL;
        } elseif ($target instanceof UnitPlot) {
            return PathLevelEnum::UNIT_PLOT;
        } elseif ($target instanceof SubBlock) {
            return PathLevelEnum::SUB_BLOCK;
        } elseif ($target instanceof Block) {
            return PathLevelEnum::BLOCK;
        } elseif ($target instanceof Experiment) {
            return PathLevelEnum::EXPERIMENT;
        } elseif ($target instanceof SurfacicUnitPlot) {
            return PathLevelEnum::SURFACIC_UNIT_PLOT;
        } else {
            return null;
        }
    }

    /**
     * @param $target
     * @return FieldMeasure
     */
    public function setTarget($target): FieldMeasure
    {
        if ($target instanceof Individual) {
            $this->setIndividualTarget($target);
        } elseif ($target instanceof UnitPlot) {
            $this->setUnitPlotTarget($target);
        } elseif ($target instanceof SubBlock) {
            $this->setSubBlockTarget($target);
        } elseif ($target instanceof Block) {
            $this->setBlockTarget($target);
        } elseif ($target instanceof Experiment) {
            $this->setExperimentTarget($target);
        } elseif ($target instanceof SurfacicUnitPlot) {
            $this->setSurfacicUnitPlotTarget($target);
        }
        return $this;
    }

    /**
     * @return Session|null
     */
    public function getSession(): ?Session
    {
        return $this->session;
    }

    /**
     * @param Session|null $session
     * @return FieldMeasure
     */
    public function setSession(?Session $session): FieldMeasure
    {
        $this->session = $session;
        return $this;
    }

    /**
     * @return FieldGeneration|null
     */
    public function getFieldParent(): ?FieldGeneration
    {
        return $this->fieldParent;
    }

    /**
     * @param FieldGeneration|null $fieldParent
     * @return FieldMeasure
     */
    public function setFieldParent(?FieldGeneration $fieldParent): FieldMeasure
    {
        $this->fieldParent = $fieldParent;
        return $this;
    }

    /**
     * @param AbstractVariable $variable
     * @return FieldMeasure
     */
    public function setVariable(AbstractVariable $variable): FieldMeasure
    {
        if ($variable instanceof GeneratorVariable) {
            $this->setGeneratorVariable($variable);
        } elseif ($variable instanceof SimpleVariable) {
            $this->setSimpleVariable($variable);
        } elseif ($variable instanceof SemiAutomaticVariable) {
            $this->setSemiAutomaticVariable($variable);
        }
        return $this;
    }

    /**
     * @return GeneratorVariable|null
     */
    public function getGeneratorVariable(): ?GeneratorVariable
    {
        return $this->generatorVariable;
    }

    /**
     * @param GeneratorVariable $generatorVariable
     * @return FieldMeasure
     */
    public function setGeneratorVariable(GeneratorVariable $generatorVariable): FieldMeasure
    {
        $this->simpleVariable = null;
        $this->generatorVariable = $generatorVariable;
        return $this;
    }

    /**
     * @return SimpleVariable|null
     */
    public function getSimpleVariable(): ?SimpleVariable
    {
        return $this->simpleVariable;
    }

    /**
     * @param SimpleVariable $simpleVariable
     * @return FieldMeasure
     */
    public function setSimpleVariable(SimpleVariable $simpleVariable): FieldMeasure
    {
        $this->simpleVariable = $simpleVariable;
        $this->generatorVariable = null;
        return $this;
    }

    /**
     * @return SemiAutomaticVariable|null
     */
    public function getSemiAutomaticVariable(): ?SemiAutomaticVariable
    {
        return $this->semiAutomaticVariable;
    }

    /**
     * @param SemiAutomaticVariable|null $semiAutomaticVariable
     * @return FieldMeasure
     */
    public function setSemiAutomaticVariable(?SemiAutomaticVariable $semiAutomaticVariable): FieldMeasure
    {
        $this->semiAutomaticVariable = $semiAutomaticVariable;
        return $this;
    }

    /**
     * @return Individual|null
     */
    public function getIndividualTarget(): ?Individual
    {
        return $this->individualTarget;
    }

    /**
     * @param Individual|null $individualTarget
     * @return FieldMeasure
     */
    public function setIndividualTarget(?Individual $individualTarget): FieldMeasure
    {
        $this->individualTarget = $individualTarget;
        return $this;
    }

    /**
     * @return UnitPlot|null
     */
    public function getUnitPlotTarget(): ?UnitPlot
    {
        return $this->unitPlotTarget;
    }

    /**
     * @param UnitPlot|null $unitPlotTarget
     * @return FieldMeasure
     */
    public function setUnitPlotTarget(?UnitPlot $unitPlotTarget): FieldMeasure
    {
        $this->unitPlotTarget = $unitPlotTarget;
        return $this;
    }

    /**
     * @return SurfacicUnitPlot|null
     */
    public function getSurfacicUnitPlotTarget(): ?SurfacicUnitPlot
    {
        return $this->surfacicUnitPlotTarget;
    }

    /**
     * @param SurfacicUnitPlot|null $surfacicUnitPlotTarget
     * @return FieldMeasure
     */
    public function setSurfacicUnitPlotTarget(?SurfacicUnitPlot $surfacicUnitPlotTarget): FieldMeasure
    {
        $this->surfacicUnitPlotTarget = $surfacicUnitPlotTarget;
        return $this;
    }

    /**
     * @return Block|null
     */
    public function getBlockTarget(): ?Block
    {
        return $this->blockTarget;
    }

    /**
     * @param Block|null $blockTarget
     * @return FieldMeasure
     */
    public function setBlockTarget(?Block $blockTarget): FieldMeasure
    {
        $this->blockTarget = $blockTarget;
        return $this;
    }

    /**
     * @return SubBlock|null
     */
    public function getSubBlockTarget(): ?SubBlock
    {
        return $this->subBlockTarget;
    }

    /**
     * @param SubBlock|null $subBlockTarget
     * @return FieldMeasure
     */
    public function setSubBlockTarget(?SubBlock $subBlockTarget): FieldMeasure
    {
        $this->subBlockTarget = $subBlockTarget;
        return $this;
    }

    /**
     * @return Experiment|null
     */
    public function getExperimentTarget(): ?Experiment
    {
        return $this->experimentTarget;
    }

    /**
     * @param Experiment|null $experimentTarget
     * @return FieldMeasure
     */
    public function setExperimentTarget(?Experiment $experimentTarget): FieldMeasure
    {
        $this->experimentTarget = $experimentTarget;
        return $this;
    }

    /**
     * @return Collection|Measure[]
     */
    public function getMeasures()
    {
        return $this->measures;
    }

    /**
     * @param Measure $measure
     * @return $this
     */
    public function addMeasure(Measure $measure): FieldMeasure
    {
        if (!$this->measures->contains($measure)) {
            $this->measures->add($measure);
            $measure->setFormField($this);
        }
        return $this;
    }

    /**
     * @param Measure $measure
     * @return $this
     */
    public function removeMeasure(Measure $measure): FieldMeasure
    {
        if ($this->measures->contains($measure)) {
            $this->measures->removeElement($measure);
        }
        return $this;
    }

    /**
     * @return Collection|FieldGeneration[]
     */
    public function getFieldGenerations()
    {
        return $this->fieldGenerations;
    }

    /**
     * @param FieldGeneration $fieldGeneration
     * @return FieldMeasure
     */
    public function addFieldGeneration(FieldGeneration $fieldGeneration): FieldMeasure
    {
        if (!$this->fieldGenerations->contains($fieldGeneration)) {
            $this->fieldGenerations->add($fieldGeneration);
            $fieldGeneration->setFormField($this);
        }
        return $this;
    }

    /**
     * @param FieldGeneration $fieldGeneration
     * @return FieldMeasure
     */
    public function removeFieldGeneration(FieldGeneration $fieldGeneration): FieldMeasure
    {
        if ($this->fieldGenerations->contains($fieldGeneration)) {
            $this->fieldGenerations->removeElement($fieldGeneration);
        }
        return $this;
    }
}
