<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Webapp\Core\Validator\UniqueAttributeInParent;

/**
 * Class Modality
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={}
 *     },
 *     itemOperations={
 *          "get"={},
 *          "patch"={"denormalization_context"={"groups"={"edit"}}}
 *     }
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={"factor": "exact", "treatments": "exact"})
 *
 * @Gedmo\SoftDeleteable()
 *
 * @ORM\Entity
 * @ORM\Table(name="modality", schema="webapp")
 */
class Modality extends IdentifiedEntity
{
    use SoftDeleteableEntity;

    use OpenSilexEntity;

    /**
     * @var string
     *
     * @Assert\NotBlank
     * @UniqueAttributeInParent(parentsAttributes={"factor.modalities"})
     * @ORM\Column(type="string")
     * @Groups({"factor_post", "design_explorer_view", "platform_full_view", "webapp_data_view", "edit", "protocol_synthesis", "protocol_full_view", "data_view_item"})
     */
    private string $value;

    /**
     * @var ?string
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"factor_post","design_explorer_view", "platform_full_view", "webapp_data_view", "edit", "protocol_synthesis", "protocol_full_view", "data_view_item"})
     */
    private ?string $shortName;

    /**
     * @var ?string
     *
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"factor_post", "platform_full_view", "edit", "protocol_full_view"})
     */
    private ?string $identifier;

    /**
     * @var Factor
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Factor", inversedBy="modalities")
     * @ORM\JoinColumn(nullable=false)
     * @Groups({"webapp_data_view", "data_view_item"})
     */
    private $factor;

    /**
     * @var Collection|array
     *
     * @ORM\ManyToMany(targetEntity="Webapp\Core\Entity\Treatment", mappedBy="modalities", cascade={"remove"})
     * @ORM\JoinTable(name="rel_treatment_modality", schema="webapp")
     */
    private $treatments;

    public function __construct()
    {
        $this->value = "";
        $this->shortName = null;
        $this->identifier = null;
        $this->openSilexInstance = null;
        $this->openSilexUri = null;
    }

    /**
     * @Groups({"platform_full_view"})
     * @return mixed
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getValue(): string
    {
        return $this->value;
    }

    /**
     * @param string $value
     * @return Modality
     */
    public function setValue(string $value): Modality
    {
        $this->value = $value;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getShortName(): ?string
    {
        return $this->shortName;
    }

    /**
     * @param string|null $shortName
     * @return Modality
     */
    public function setShortName(?string $shortName): Modality
    {
        $this->shortName = $shortName;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getIdentifier(): ?string
    {
        return $this->identifier;
    }

    /**
     * @param string|null $identifier
     * @return Modality
     */
    public function setIdentifier(?string $identifier): Modality
    {
        $this->identifier = $identifier;
        return $this;
    }

    /**
     * @return Factor
     */
    public function getFactor(): Factor
    {
        return $this->factor;
    }

    /**
     * @param Factor $factor
     * @return Modality
     */
    public function setFactor(Factor $factor): Modality
    {
        $this->factor = $factor;
        return $this;
    }

    /**
     * @return array|Collection
     */
    public function getTreatments()
    {
        return $this->treatments;
    }

    /**
     * @param array|Collection $treatments
     * @return Modality
     */
    public function setTreatments($treatments)
    {
        $this->treatments = $treatments;
        return $this;
    }

}
