<?php


namespace Webapp\Core\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;

/**
 * Class PathBase
 * @package Webapp\Core\Entity
 *
 * @ORM\Entity
 * @ORM\Table(name="path_filter_node", schema="webapp")
 */
class PathFilterNode extends IdentifiedEntity
{
    /**
     * @var PathBase|null
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\PathBase", inversedBy="pathFilterNodes"  )
     */
    private ?PathBase $pathBase;

    /**
     * @var PathFilterNode|null
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\PathFilterNode", inversedBy="branches")
     */
    private ?PathFilterNode $parentFilter;

    /**
     * @var int
     * @ORM\Column(type="integer")
     */
    private int $type; //TODO FILTERKEYENUM

    /**
     * @var string
     * @ORM\Column(type="string")
     */
    private string $text;

    /**
     * @var Collection|array<PathFilterNode>
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\PathFilterNode", mappedBy="parentFilter", cascade={"remove", "persist"}, orphanRemoval=true )
     */
    private $branches;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     */
    private ?string $operator; //TODO FilterComparatorsEnum,

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     */
    private ?string $value;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\GeneratorVariable", cascade={"persist"})
     * @ORM\JoinColumn(nullable=true)
     */
    private ?GeneratorVariable $generatorVariable;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SimpleVariable", cascade={"persist"})
     * @ORM\JoinColumn(nullable=true)
     */
    private ?SimpleVariable $simpleVariable;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SemiAutomaticVariable", cascade={"persist"})
     * @ORM\JoinColumn(nullable=true)
     */
    private ?SemiAutomaticVariable $semiAutomaticVariable;

    public function __construct($type, $text)
    {
        $this->type = $type; // TODO delete when https://github.com/api-platform/core/issues/3974 is solved
        $this->text = $text; // TODO delete when https://github.com/api-platform/core/issues/3974 is solved
        $this->branches = new ArrayCollection();
        $this->operator = null;
        $this->value = null;
        $this->generatorVariable = null;
        $this->semiAutomaticVariable = null;
        $this->simpleVariable = null;
    }

    /**
     * @return PathBase|null
     */
    public function getPathBase(): ?PathBase
    {
        return $this->pathBase;
    }

    /**
     * @param PathBase|null $pathBase
     * @return PathFilterNode
     */
    public function setPathBase(?PathBase $pathBase): PathFilterNode
    {
        $this->pathBase = $pathBase;
        return $this;
    }

    /**
     * @return int
     */
    public function getType(): int
    {
        return $this->type;
    }

    /**
     * @param int $type
     * @return PathFilterNode
     */
    public function setType(int $type): PathFilterNode
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return string
     */
    public function getText(): string
    {
        return $this->text;
    }

    /**
     * @param string $text
     * @return PathFilterNode
     */
    public function setText(string $text): PathFilterNode
    {
        $this->text = $text;
        return $this;
    }

    /**
     * @return Collection|PathFilterNode[]
     */
    public function getBranches()
    {
        return $this->branches;
    }

    /**
     * @param Collection|PathFilterNode[] $branches
     * @return PathFilterNode
     */
    public function setBranches($branches)
    {
        $this->branches = $branches;
        return $this;
    }


    /**
     * @param PathFilterNode $branche
     * @return PathFilterNode
     */
    public function addBranche($branche)
    {
        if (!$this->branches->contains($branche)) {
            $this->branches->add($branche);
            $branche->setParentFilter($this);
        }
        return $this;
    }

    /**
     * @param PathFilterNode $branche
     * @return PathFilterNode
     */
    public function removeBranche($branche)
    {
        if ($this->branches->contains($branche)) {
            $this->branches->removeElement($branche);
            $branche->setParentFilter(null);
        }
        return $this;
    }

    /**
     * @return string|null
     */
    public function getOperator(): ?string
    {
        return $this->operator;
    }

    /**
     * @param string|null $operator
     * @return PathFilterNode
     */
    public function setOperator(?string $operator): PathFilterNode
    {
        $this->operator = $operator;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getValue(): ?string
    {
        return $this->value;
    }

    /**
     * @param string|null $value
     * @return PathFilterNode
     */
    public function setValue(?string $value): PathFilterNode
    {
        $this->value = $value;
        return $this;
    }

    /**
     * @return PathFilterNode|null
     */
    public function getParentFilter(): ?PathFilterNode
    {
        return $this->parentFilter;
    }

    /**
     * @param PathFilterNode|null $parentFilter
     * @return PathFilterNode
     */
    public function setParentFilter(?PathFilterNode $parentFilter): PathFilterNode
    {
        $this->parentFilter = $parentFilter;
        return $this;
    }

    /**
     * @return SimpleVariable|null
     */
    public function getVariable(): ?AbstractVariable
    {
        return $this->simpleVariable ?? $this->semiAutomaticVariable ?? $this->generatorVariable;
    }

    /**
     * @param AbstractVariable|null $variable
     * @return PathFilterNode
     */
    public function setVariable(?AbstractVariable $variable): ?PathFilterNode
    {
        $this->simpleVariable = null;
        $this->semiAutomaticVariable = null;
        $this->generatorVariable = null;
        if ($variable instanceof SimpleVariable) {
            $this->simpleVariable = $variable;
        } else if ($variable instanceof SemiAutomaticVariable) {
            $this->semiAutomaticVariable = $variable;
        } else if ($variable instanceof GeneratorVariable) {
            $this->generatorVariable = $variable;
        }
        return $this;
    }

}
