<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2021
 */

declare(strict_types=1);

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\User;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\ApiOperation\ProjectWeightOperation;

/**
 * Class Session.
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={},
 *          "getWeight"={
 *                "controller"=ProjectWeightOperation::class,
 *                "method"="GET",
 *                "path"="/sessions/weight",
 *                "read"=false,
 *                "validate"=false,
 *                "openapi_context"={
 *                    "summary": "Restore deleted protocol",
 *                    "description": "Remove the deleted state"
 *                },
 *           },
 *     },
 *     itemOperations={
 *         "get"={},
 *         "patch"={"denormalization_context"={"groups"={"edit"}}}
 *     },
 * )
 *
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"webapp_data_view", "id_read"}})
 * @ApiFilter(SearchFilter::class, properties={"projectData": "exact", "id": "exact"})
 * @ORM\Entity()
 * @ORM\Table(name="session_data", schema="webapp")
 */
class Session extends IdentifiedEntity
{
    /**
     * @var ProjectData
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\ProjectData", inversedBy="sessions")
     * @ORM\JoinColumn(nullable=false)
     */
    private $projectData;

    /**
     * @var User
     *
     * @Groups({"data_explorer_view", "data_entry_synthesis", "variable_synthesis"})
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\User")
     */
    private User $user;

    /**
     * @var DateTime
     * @Groups({"webapp_data_view", "data_explorer_view", "data_entry_synthesis"})
     * @ORM\Column(type="datetime")
     */
    private $startedAt;

    /**
     * @var DateTime
     * @Groups({"webapp_data_view", "data_explorer_view", "data_entry_synthesis"})
     * @ORM\Column(type="datetime")
     */
    private $endedAt;

    /**
     * @var Collection<array-key, FieldMeasure>
     * @Groups({"webapp_data_view", "data_entry_synthesis", "variable_synthesis"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\FieldMeasure", mappedBy="session", cascade={"persist", "remove", "detach"})
     */
    private $fieldMeasures;

    /**
     * @var Collection|array<Annotation>
     * @Groups({"webapp_data_view", "data_explorer_view", "data_entry_synthesis"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Annotation", mappedBy="session", cascade={"persist", "remove", "detach"})
     */
    private $annotations;

    /**
     * @var string|null
     * @Groups({"edit"})
     * @ORM\Column(type="string", nullable=true)
     */
    private ?string $comment;

    /**
     * Session constructor.
     */
    public function __construct()
    {
        $this->fieldMeasures = new ArrayCollection();
        $this->annotations = new ArrayCollection();
        $this->comment = null;
    }

    /**
     * @return ProjectData
     */
    public function getProjectData(): ProjectData
    {
        return $this->projectData;
    }

    /**
     * @param ProjectData $projectData
     * @return Session
     */
    public function setProjectData(ProjectData $projectData): Session
    {
        $this->projectData = $projectData;
        return $this;
    }

    /**
     * @Groups({"webapp_data_view"})
     * @return string
     */
    public function getUserName(): string
    {
        return $this->user->getUsername();
    }

    /**
     * @return User
     */
    public function getUser(): User
    {
        return $this->user;
    }

    /**
     * @param User $user
     * @return Session
     */
    public function setUser(User $user): Session
    {
        $this->user = $user;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getStartedAt(): DateTime
    {
        return $this->startedAt;
    }

    /**
     * @param DateTime $startedAt
     * @return Session
     */
    public function setStartedAt(DateTime $startedAt): Session
    {
        $this->startedAt = $startedAt;
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getEndedAt(): DateTime
    {
        return $this->endedAt;
    }

    /**
     * @param DateTime $endedAt
     * @return Session
     */
    public function setEndedAt(DateTime $endedAt): Session
    {
        $this->endedAt = $endedAt;
        return $this;
    }

    /**
     * @return Collection<array-key,FieldMeasure>
     */
    public function getFieldMeasures(): Collection
    {
        return $this->fieldMeasures;
    }

    /**
     * @param FieldMeasure $fieldMeasure
     *
     * @return Session
     */
    public function addFieldMeasure(FieldMeasure $fieldMeasure): Session
    {
        if (!$this->fieldMeasures->contains($fieldMeasure)) {
            $this->fieldMeasures->add($fieldMeasure);
            $fieldMeasure->setSession($this);
        }
        return $this;
    }

    /**
     * @param FieldMeasure $fieldMeasure
     *
     * @return Session
     */
    public function removeFieldMeasure(FieldMeasure $fieldMeasure): Session
    {
        if ($this->fieldMeasures->contains($fieldMeasure)) {
            $this->fieldMeasures->removeElement($fieldMeasure);
            $fieldMeasure->setSession(null);
        }
        return $this;
    }

    /**
     * @return Collection|Annotation[]
     */
    public function getAnnotations()
    {
        return $this->annotations;
    }

    /**
     * @param Collection|Annotation[] $annotations
     * @return Session
     */
    public function setAnnotations($annotations)
    {
        $this->annotations = $annotations;
        return $this;
    }

    /**
     * @param Annotation $annotation
     * @return Session
     */
    public function addAnnotation($annotation)
    {
        if (!$this->annotations->contains($annotation)) {
            $this->annotations->add($annotation);
            $annotation->setSession($this);
        }
        return $this;
    }

    /**
     * @param Annotation $annotation
     * @return Session
     */
    public function removeAnnotation($annotation)
    {
        if ($this->annotations->contains($annotation)) {
            $this->annotations->removeElement($annotation);
            $annotation->setSession(null);
        }
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return Session
     */
    public function setComment(?string $comment): Session
    {
        $this->comment = $comment;
        return $this;
    }

}
