<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2020
 */

declare(strict_types=1);

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiProperty;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\Site;
use Shared\Enumeration\Annotation\EnumType;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Validator\UniqueAttributeInParent;

/**
 * Class StateCode
 * @package Webapp\Core\Entity
 * @ApiResource(
 *     collectionOperations={
 *         "get"={},
 *         "post"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())",
 *          },
 *     },
 *     itemOperations={
 *          "get"={},
 *          "patch"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())"
 *          },
 *          "delete"={
 *              "security"="!object.isPermanent() && is_granted('ROLE_PLATFORM_MANAGER', object.getSite())"
 *          }
 *     }
 * )
 * @ApiFilter(SearchFilter::class, properties={"site": "exact", "code": "exact", "project": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"project_explorer_view"}})
 *
 * @ORM\Entity()
 * @ORM\Table(name="state_code", schema="webapp")
 */
class StateCode extends IdentifiedEntity
{
    /**
     * @var int
     * @ORM\Column(type="integer")
     * @Groups({"project_explorer_view", "state_code_post", "webapp_data_view", "data_entry_synthesis", "variable_synthesis", "data_view_item", "graphical_measure_view"})
     * @UniqueAttributeInParent(parentsAttributes={"project.stateCodes", "site.stateCodes"})
     */
    private int $code;

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"project_explorer_view", "state_code_post", "data_entry_synthesis", "variable_synthesis", "fusion_result"})
     */
    private string $title;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @Groups({"state_code_post"})
     */
    private ?string $meaning = null;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     * @EnumType(class="Webapp\Core\Enumeration\SpreadingEnum", nullable=true)
     * @Groups({"state_code_post"})
     */
    private ?string $spreading = null;

    /**
     * @var int|null
     * @ORM\Column(type="integer", nullable=true)
     * @Groups({"state_code_post", "graphical_measure_view"})
     */
    private ?int $color = null;

    /**
     * @var bool
     * @ORM\Column(type="boolean")
     * @ApiProperty(writable=false)
     * @Groups({"project_explorer_view", "data_entry_synthesis"})
     */
    private bool $permanent;

    /**
     * @var Site|null
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="stateCodes")
     * @Groups({"state_code_post"})
     */
    private ?Site $site;

    /**
     * @var Project|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Project", inversedBy="stateCodes")
     * @Groups({"state_code_post"})
     */
    private ?Project $project;

    public function __construct()
    {
        $this->site = null;
        $this->project = null;
        $this->permanent = false;
    }

    /**
     * @return int
     */
    public function getCode(): int
    {
        return $this->code;
    }

    /**
     * @param int $code
     * @return StateCode
     */
    public function setCode(int $code): StateCode
    {
        $this->code = $code;
        return $this;
    }

    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @param string $title
     * @return StateCode
     */
    public function setTitle(string $title): StateCode
    {
        $this->title = $title;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getMeaning(): ?string
    {
        return $this->meaning;
    }

    /**
     * @param string|null $meaning
     * @return StateCode
     */
    public function setMeaning(?string $meaning): StateCode
    {
        $this->meaning = $meaning;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getSpreading(): ?string
    {
        return $this->spreading;
    }

    /**
     * @param string|null $spreading
     * @return StateCode
     */
    public function setSpreading(?string $spreading): StateCode
    {
        $this->spreading = $spreading;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getColor(): ?int
    {
        return $this->color;
    }

    /**
     * @param int|null $color
     * @return StateCode
     */
    public function setColor(?int $color): StateCode
    {
        $this->color = $color;
        return $this;
    }

    /**
     * @return Site|null
     */
    public function getSite(): ?Site
    {
        return $this->site;
    }

    /**
     * @param Site|null $site
     * @return StateCode
     */
    public function setSite(?Site $site): StateCode
    {
        $this->site = $site;
        $this->project = null;
        return $this;
    }

    /**
     * @return Project|null
     */
    public function getProject(): ?Project
    {
        return $this->project;
    }

    /**
     * @param Project|null $project
     * @return StateCode
     */
    public function setProject(?Project $project): StateCode
    {
        $this->project = $project;
        $this->site = null;
        return $this;
    }

    /**
     * @return bool
     */
    public function isPermanent(): bool
    {
        return $this->permanent;
    }

    /**
     * @param bool $permanent
     * @return StateCode
     */
    public function setPermanent(bool $permanent): StateCode
    {
        $this->permanent = $permanent;
        return $this;
    }
}
