<?php

namespace Webapp\FileManagement\Service;

use Doctrine\ORM\EntityManagerInterface;
use SimpleXMLElement;
use Vich\UploaderBundle\Storage\StorageInterface;
use Webapp\FileManagement\Entity\Base\UserLinkedFileJob;
use Webapp\FileManagement\Entity\RequestFile;
use Webapp\FileManagement\Exception\ParsingException;
use ZipArchive;

/**
 * Class WebappParserService.
 */
abstract class AbstractReaderService extends AbstractFileService
{
    /**
     * @var int State at which the process has throw an error.
     */
    public int $currentParsingStateError;

    public function __construct(EntityManagerInterface $entityManager,
                                DirectoryNamer         $directoryNameManager,
                                StorageInterface       $storage,
                                string                 $tmpDir)
    {
        parent::__construct($entityManager, $directoryNameManager, $storage, $tmpDir);
    }

    /**
     * Method to extract the simpleXMLElement from a zip project file
     * @param UserLinkedFileJob $parsingJob
     * @return SimpleXMLElement
     * @throws ParsingException
     */
    protected function getSimpleXMLElement(UserLinkedFileJob $parsingJob): SimpleXMLElement
    {
        $tmpDir = $this->getTmpDir($parsingJob);
        if (str_ends_with($parsingJob->getFilePath(), ".zip")) {
            $zip = new ZipArchive;
            $zipFilePath = $tmpDir . $parsingJob->getFilePath();
            $res = $zip->open($zipFilePath);
            if ($res === TRUE) {
                $zip->extractTo($tmpDir);
                $zip->close();
            } else {
                throw new ParsingException("Error during unzipping", RequestFile::ERROR_DURING_UNZIPPING);
            }
        }
        $xmlFilePath = file_exists($tmpDir . $parsingJob->getName() . '.xml') ? $tmpDir . $parsingJob->getName() . '.xml' : $tmpDir . 'transfert.xml';
        if (!file_exists($xmlFilePath)) {
            throw new ParsingException("File not found");
        }
        $xml = simplexml_load_file($xmlFilePath);
        if (false === $xml) {
            throw new ParsingException("Invalid Xml");
        }

        return $xml;
    }

    protected function copyFileInTmpDir(UserLinkedFileJob $projectFile)
    {
        copy($this->storage->resolvePath($projectFile, 'file'), $this->getTmpDir($projectFile) . $projectFile->getFilePath());
    }
}
