<?php

namespace Mobile\Device\Entity;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Measure\Entity\Variable\StateCode;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class UnitParcel.
 *
 * @ORM\Entity(repositoryClass="Mobile\Device\Repository\UnitParcelRepository")
 * @ORM\Table(name="unit_plot", schema="adonis")
 */
class UnitParcel extends BusinessObject
{
    /**
     * @var int
     *
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\Column(type="integer", nullable=false)
     */
    private $x;

    /**
     * @var int
     *
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\Column(type="integer", nullable=false)
     */
    private $y;

    /**
     * @var StateCode|null
     *
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Measure\Entity\Variable\StateCode", cascade={"persist", "remove", "detach"})
     * @ORM\JoinColumn(nullable=true)
     */
    private $stateCode;

    /**
     * @var Collection<array-key, Individual>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Device\Entity\Individual", mappedBy="unitParcel", cascade={"persist", "remove", "detach"})
     */
    private $individuals;

    /**
     * @var SubBlock|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\SubBlock", inversedBy="unitParcels")
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    private $subBlock;

    /**
     * @var Block|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\Block", inversedBy="unitParcels")
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    private $block;

    /**
     * @var Treatment|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\Treatment")
     * @ORM\JoinColumn(nullable=true)
     */
    private $treatment;

    /**
     * @var string | null
     * @ORM\Column(type="string", nullable=true)
     */
    private $ident;

    /**
     * @var Collection<array-key, OutExperimentationZone>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Device\Entity\OutExperimentationZone", mappedBy="unitParcel", cascade={"persist", "remove", "detach"})
     */
    private $outExperimentationZones;

    /**
     * @var DateTime|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $apparitionDate;

    /**
     * @var DateTime|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $demiseDate;

    /**
     * UnitParcel constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->individuals = new ArrayCollection();
        $this->outExperimentationZones = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getX(): int
    {
        return $this->x;
    }

    /**
     * @param int $x
     * @return UnitParcel
     */
    public function setX(int $x): UnitParcel
    {
        $this->x = $x;
        return $this;
    }

    /**
     * @return int
     */
    public function getY(): int
    {
        return $this->y;
    }

    /**
     * @param int $y
     * @return UnitParcel
     */
    public function setY(int $y): UnitParcel
    {
        $this->y = $y;
        return $this;
    }

    /**
     * @return StateCode|null
     */
    public function getStateCode(): ?StateCode
    {
        return $this->stateCode;
    }

    /**
     * @param StateCode|null $stateCode
     * @return UnitParcel
     */
    public function setStateCode(?StateCode $stateCode): UnitParcel
    {
        $this->stateCode = $stateCode;
        return $this;
    }

    /**
     * @return Collection<array-key, Individual>
     */
    public function getIndividuals(): Collection
    {
        return $this->individuals;
    }

    /**
     * @param Individual $individual
     */
    public function addIndividual(Individual $individual): void
    {
        if (!$this->individuals->contains($individual)) {
            $individual->setUnitParcel($this);
            $this->individuals->add($individual);
        }
    }

    /**
     * @param Individual $individual
     */
    public function removeIndividual(Individual $individual): void
    {
        if ($this->individuals->contains($individual)) {
            $individual->setUnitParcel(null);
            $this->individuals->removeElement($individual);
        }
    }

    /**
     * @return SubBlock|null
     */
    public function getSubBlock(): ?SubBlock
    {
        return $this->subBlock;
    }

    /**
     * @param SubBlock|null $subBlock
     * @return UnitParcel
     */
    public function setSubBlock(?SubBlock $subBlock): UnitParcel
    {
        $this->subBlock = $subBlock;
        return $this;
    }

    /**
     * @return Block|null
     */
    public function getBlock(): ?Block
    {
        return $this->block;
    }

    /**
     * @param Block|null $block
     * @return UnitParcel
     */
    public function setBlock(?Block $block): UnitParcel
    {
        $this->block = $block;
        return $this;
    }

    /**
     * @return Treatment|null
     */
    public function getTreatment(): ?Treatment
    {
        return $this->treatment;
    }

    /**
     * @param Treatment|null $treatment
     * @return UnitParcel
     */
    public function setTreatment(?Treatment $treatment): UnitParcel
    {
        $this->treatment = $treatment;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getIdent(): ?string
    {
        return $this->ident;
    }

    /**
     * @param string|null $ident
     * @return UnitParcel
     */
    public function setIdent(?string $ident): UnitParcel
    {
        $this->ident = $ident;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getOutExperimentationZones(): Collection
    {
        return $this->outExperimentationZones;
    }

    /**
     * @param Collection $outExperimentationZones
     * @return UnitParcel
     */
    public function setOutExperimentationZones(Collection $outExperimentationZones): UnitParcel
    {
        $this->outExperimentationZones = $outExperimentationZones;
        return $this;
    }

    /**
     * @param OutExperimentationZone $outExperimentationZone
     */
    public function addOutExperimentationZone(OutExperimentationZone $outExperimentationZone): void
    {
        if (!$this->outExperimentationZones->contains($outExperimentationZone)) {
            $outExperimentationZone->setUnitParcel($this);
            $this->outExperimentationZones->add($outExperimentationZone);
        }
    }

    /**
     * @param OutExperimentationZone $outExperimentationZone
     */
    public function removeOutExperimentationZone(OutExperimentationZone $outExperimentationZone): void
    {
        if ($this->outExperimentationZones->contains($outExperimentationZone)) {
            $outExperimentationZone->setUnitParcel(null);
            $this->outExperimentationZones->removeElement($outExperimentationZone);
        }
    }

    public function getApparitionDate(): ?DateTime
    {
        return $this->apparitionDate;
    }

    public function setApparitionDate(?DateTime $apparitionDate): UnitParcel
    {
        $this->apparitionDate = $apparitionDate;
        return $this;
    }

    /**
     * @return DateTime|null
     */
    public function getDemiseDate(): ?DateTime
    {
        return $this->demiseDate;
    }

    /**
     * @param DateTime|null $demiseDate
     * @return UnitParcel
     */
    public function setDemiseDate(?DateTime $demiseDate): UnitParcel
    {
        $this->demiseDate = $demiseDate;
        return $this;
    }

    public function parent(): ?BusinessObject
    {
        return $this->getSubBlock() ?? $this->getBlock();
    }
}
