<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2021
 */

declare(strict_types=1);

namespace Shared\TransferSync\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Device\Entity\Individual;
use Mobile\Device\Entity\UnitParcel;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * Class IndividualStructureChange
 * Declaration of changes between one request project business object and its state in response project.
 * @package Shared\TransferSync\Entity
 *
 * @ApiResource(
 *     collectionOperations={},
 *     itemOperations={
 *         "get"={
 *             "security"="is_granted('STRUCTURE_CHANGE_GET_PATCH', object)",
 *         },
 *         "patch"={
 *             "security"="is_granted('STRUCTURE_CHANGE_GET_PATCH', object)",
 *             "denormalization_context"={"groups"={"individual_structure_change_patch"}},
 *         }
 *     }
 * )
 *
 * @ORM\Entity()
 * @ORM\Table(name="rel_individual_change", schema="shared")
 */
class IndividualStructureChange extends IdentifiedEntity
{
    /**
     * @var Individual|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\Individual")
     */
    private ?Individual $requestIndividual;

    /**
     * @var Individual|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\Individual")
     */
    private ?Individual $responseIndividual;
    /**
     * @var UnitParcel|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\UnitParcel")
     */
    private ?UnitParcel $requestUnitParcel;

    /**
     * @var UnitParcel|null
     * @Groups({"status_project_webapp_return_data"})
     * @ORM\ManyToOne(targetEntity="Mobile\Device\Entity\UnitParcel")
     */
    private ?UnitParcel $responseUnitParcel;

    /**
     * @var StatusDataEntry
     * @ORM\ManyToOne(targetEntity="Shared\TransferSync\Entity\StatusDataEntry", inversedBy="changes")
     * @ORM\JoinColumn(nullable=false)
     */
    private StatusDataEntry $statusDataEntry;

    /**
     * @var bool
     * @Groups({"individual_structure_change_patch"})
     * @ORM\Column(type="boolean", options={"default"=false})
     */
    private bool $approved;

    /**
     * IndividualStructureChange constructor.
     */
    public function __construct()
    {
        $this->approved = false;
        $this->requestIndividual = null;
        $this->requestUnitParcel = null;
        $this->responseIndividual = null;
        $this->responseUnitParcel = null;
    }

    /**
     * @return Individual|null
     */
    public function getRequestIndividual(): ?Individual
    {
        return $this->requestIndividual;
    }

    /**
     * @param Individual|null $requestIndividual
     * @return IndividualStructureChange
     */
    public function setRequestIndividual(?Individual $requestIndividual): IndividualStructureChange
    {
        $this->requestIndividual = $requestIndividual;
        return $this;
    }

    /**
     * @return Individual|null
     */
    public function getResponseIndividual(): ?Individual
    {
        return $this->responseIndividual;
    }

    /**
     * @param Individual|null $responseIndividual
     * @return IndividualStructureChange
     */
    public function setResponseIndividual(?Individual $responseIndividual): IndividualStructureChange
    {
        $this->responseIndividual = $responseIndividual;
        return $this;
    }

    /**
     * @return UnitParcel|null
     */
    public function getRequestUnitParcel(): ?UnitParcel
    {
        return $this->requestUnitParcel;
    }

    /**
     * @param UnitParcel|null $requestUnitParcel
     * @return IndividualStructureChange
     */
    public function setRequestUnitParcel(?UnitParcel $requestUnitParcel): IndividualStructureChange
    {
        $this->requestUnitParcel = $requestUnitParcel;
        return $this;
    }

    /**
     * @return UnitParcel|null
     */
    public function getResponseUnitParcel(): ?UnitParcel
    {
        return $this->responseUnitParcel;
    }

    /**
     * @param UnitParcel|null $responseUnitParcel
     * @return IndividualStructureChange
     */
    public function setResponseUnitParcel(?UnitParcel $responseUnitParcel): IndividualStructureChange
    {
        $this->responseUnitParcel = $responseUnitParcel;
        return $this;
    }

    /**
     * @return StatusDataEntry
     */
    public function getStatusDataEntry(): StatusDataEntry
    {
        return $this->statusDataEntry;
    }

    /**
     * @param StatusDataEntry $statusDataEntry
     * @return IndividualStructureChange
     */
    public function setStatusDataEntry(StatusDataEntry $statusDataEntry): IndividualStructureChange
    {
        $this->statusDataEntry = $statusDataEntry;
        return $this;
    }

    /**
     * @return bool
     */
    public function isApproved(): bool
    {
        return $this->approved;
    }

    /**
     * @param bool $approved
     * @return IndividualStructureChange
     */
    public function setApproved(bool $approved): IndividualStructureChange
    {
        $this->approved = $approved;
        return $this;
    }
}
