<?php

namespace Webapp\Core\Entity\DataView;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use DateTime;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\CustomFilters\OrSearchFilter;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Entity\Block;
use Webapp\Core\Entity\Experiment;
use Webapp\Core\Entity\FieldMeasure;
use Webapp\Core\Entity\GeneratorVariable;
use Webapp\Core\Entity\Individual;
use Webapp\Core\Entity\ProjectData;
use Webapp\Core\Entity\SemiAutomaticVariable;
use Webapp\Core\Entity\Session;
use Webapp\Core\Entity\SimpleVariable;
use Webapp\Core\Entity\SubBlock;
use Webapp\Core\Entity\SurfacicUnitPlot;
use Webapp\Core\Entity\Treatment;
use Webapp\Core\Entity\UnitPlot;
use Webapp\Core\Enumeration\PathLevelEnum;

/**
 * @ApiResource(
 *      normalizationContext={"groups"={"data_view_item"}},
 *      collectionOperations={
 *          "get"={},
 *      },
 *      itemOperations={
 *          "get"={},
 *      },
 *  )
 *
 *
 * @ApiFilter(SearchFilter::class, properties={
 *     "projectData": "exact",
 *     "session": "exact",
 * })
 *
 * @ORM\Entity(readOnly=true)
 * @ORM\Table(name="view_data_item", schema="webapp")
 */
class DataViewItem
{
    /**
     * @ORM\Id()
     * @Groups({"data_view_item"})
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\FieldMeasure")
     */
    private FieldMeasure $fieldMeasure;

    /**
     * @Groups({"data_view_item"})
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\ProjectData")
     */
    private ProjectData $projectData;

    /**
     * @Groups({"data_view_item"})
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Session")
     */
    private Session $session;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Experiment")
     */
    private Experiment $experimentTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Block")
     */
    private Block $blockTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SubBlock")
     */
    private SubBlock $subBlockTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SurfacicUnitPlot")
     */
    private SurfacicUnitPlot $surfacicUnitPlotTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\UnitPlot")
     */
    private UnitPlot $unitPlotTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Individual")
     */
    private Individual $individualTarget;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SimpleVariable")
     */
    private SimpleVariable $simpleVariable;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\GeneratorVariable")
     */
    private GeneratorVariable $generatorVariable;

    /**
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\SemiAutomaticVariable")
     */
    private SemiAutomaticVariable $semiAutomaticVariable;

    /**
     * @Groups({"data_view_item"})
     * @ORM\Column()
     */
    private int $repetition;

    /**
     * @Groups({"data_view_item"})
     * @ORM\Column()
     */
    private string $value;

    /**
     * @Groups({"data_view_item"})
     * @ORM\Column()
     */
    private int $code;

    /**
     * @Groups({"data_view_item"})
     * @ORM\Column()
     */
    private DateTime $timestamp;

    /**
     * @Groups({"data_view_item"})
     * @ORM\Column()
     */
    private string $username;

    /**
     * @Groups({"data_view_item"})
     * @return string|null
     */
    public function getTargetType(): ?string
    {
        $target = $this->getTarget();
        if ($target instanceof Individual) {
            return PathLevelEnum::INDIVIDUAL;
        } elseif ($target instanceof SurfacicUnitPlot) {
            return PathLevelEnum::SURFACIC_UNIT_PLOT;
        } elseif ($target instanceof UnitPlot) {
            return PathLevelEnum::UNIT_PLOT;
        } elseif ($target instanceof SubBlock) {
            return PathLevelEnum::SUB_BLOCK;
        } elseif ($target instanceof Block) {
            return PathLevelEnum::BLOCK;
        } elseif ($target instanceof Experiment) {
            return PathLevelEnum::EXPERIMENT;
        } else {
            return null;
        }
    }

    /**
     * @Groups({"data_view_item"})
     * @return Block|Experiment|Individual|SubBlock|SurfacicUnitPlot|UnitPlot
     */
    public function getTarget()
    {
        return $this->individualTarget ?? $this->surfacicUnitPlotTarget ?? $this->unitPlotTarget ?? $this->subBlockTarget ?? $this->blockTarget ?? $this->experimentTarget;
    }

    /**
     *
     * @Groups({"data_view_item"})
     * @return Block|Experiment|GeneratorVariable|SemiAutomaticVariable|SimpleVariable|SurfacicUnitPlot
     */
    public function getVariable()
    {
        return $this->generatorVariable ?? $this->semiAutomaticVariable ?? $this->simpleVariable ?? $this->blockTarget ?? $this->experimentTarget ?? $this->surfacicUnitPlotTarget;
    }

    public function getFieldMeasure(): FieldMeasure
    {
        return $this->fieldMeasure;
    }

    public function getProjectData(): ProjectData
    {
        return $this->projectData;
    }

    public function getSession(): Session
    {
        return $this->session;
    }

    public function getRepetition(): int
    {
        return $this->repetition;
    }

    public function getValue(): string
    {
        return $this->value;
    }

    public function getCode(): int
    {
        return $this->code;
    }

    public function getTimestamp(): DateTime
    {
        return $this->timestamp;
    }

    public function getUsername(): string
    {
        return $this->username;
    }

    public function getDataViewBusinessObject(): DataViewBusinessObject
    {
        return $this->dataViewBusinessObject;
    }

}
