<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\Site;
use Shared\Authentication\Entity\User;
use Shared\RightManagement\Annotation\AdvancedRight;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Webapp\Core\Validator\UniqueAttributeInParent;

/**
 * Class UserGroup
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={},
 *         "post"={"security_post_denormalize"="is_granted('ROLE_SITE_ADMIN', object.getSite())"}
 *     },
 *     itemOperations={
 *         "get"={},
 *         "patch"={"security"="is_granted('ROLE_SITE_ADMIN', object.getSite())"},
 *         "delete"={"security"="is_granted('ROLE_SITE_ADMIN', object.getSite())"},
 *     }
 *
 * )
 * @ApiFilter(SearchFilter::class, properties={"site": "exact", "label": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"admin_explorer_view"}})
 *
 * @AdvancedRight(siteAttribute="site")
 *
 * @ORM\Entity()
 * @ORM\Table(name="user_group", schema="webapp")
 */
class UserGroup extends IdentifiedEntity
{
    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"admin_explorer_view"})
     * @Assert\NotBlank
     * @UniqueAttributeInParent(parentsAttributes={"site.userGroups"})
     */
    private string $name;

    /**
     * @var Site
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="userGroups")
     */
    private Site $site;

    /**
     * @var array|Collection<User>
     * @ORM\ManyToMany(targetEntity="Shared\Authentication\Entity\User", inversedBy="groups")
     * @ORM\JoinTable(name="rel_user_user_group", schema="webapp")
     */
    private $users;

    public function __construct()
    {
        $this->users = new ArrayCollection();
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return UserGroup
     */
    public function setName(string $name): UserGroup
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return Site
     */
    public function getSite(): Site
    {
        return $this->site;
    }

    /**
     * @param Site $site
     * @return UserGroup
     */
    public function setSite(Site $site): UserGroup
    {
        $this->site = $site;
        return $this;
    }

    /**
     * @return array|Collection
     */
    public function getUsers()
    {
        return $this->users;
    }

    /**
     * @param array|Collection $users
     * @return UserGroup
     */
    public function setUsers($users)
    {
        $this->users = $users;
        return $this;
    }
}
