<?php

namespace Mobile\Measure\Entity\Variable;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Mobile\Measure\Entity\Variable\Base\Variable;
use Mobile\Project\Entity\DataEntryProject;

/**
 * Class GeneratorVariable.
 *
 * @ORM\Entity(repositoryClass="Mobile\Measure\Repository\Variable\GeneratorVariableRepository")
 * @ORM\Table(name="variable_generator", schema="adonis")
 */
class GeneratorVariable extends Variable
{
    /**
     * @var DataEntryProject|null
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Project\Entity\DataEntryProject", inversedBy="generatorVariables")
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    protected $project;

    /**
     * @var DataEntryProject|null
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Project\Entity\DataEntryProject", inversedBy="connectedGeneratorVariables")
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    protected $connectedDataEntryProject;

    /**
     * @var GeneratorVariable|null
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Measure\Entity\Variable\GeneratorVariable", inversedBy="generatorVariables")
     * @ORM\JoinColumn(nullable=true, onDelete="CASCADE")
     */
    protected $generatorVariable;
    /**
     * @var Material | null
     *
     * @ORM\ManyToOne(targetEntity="Mobile\Measure\Entity\Variable\Material", inversedBy="generatorVariables")
     */
    protected $material;
    /**
     * @var Collection<array-key, UniqueVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\UniqueVariable", mappedBy="generatorVariable", cascade={"persist", "remove", "detach"})
     */
    private $uniqueVariables;
    /**
     * @var Collection<array-key, GeneratorVariable>
     *
     * @ORM\OneToMany(targetEntity="Mobile\Measure\Entity\Variable\GeneratorVariable", mappedBy="generatorVariable", cascade={"persist", "remove", "detach"})
     */
    private $generatorVariables;
    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    private $generatedPrefix;
    /**
     * @var bool
     *
     * @ORM\Column(type="boolean")
     */
    private $numeralIncrement;

    /**
     * GeneratorVariable constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->uniqueVariables = new ArrayCollection();
        $this->generatorVariables = new ArrayCollection();
    }

    /**
     * @return Collection<array-key, UniqueVariable>
     */
    public function getUniqueVariables(): Collection
    {
        return $this->uniqueVariables;
    }

    /**
     * @param UniqueVariable $variable
     */
    public function removeUniqueVariable(UniqueVariable $variable): void
    {
        if ($this->uniqueVariables->contains($variable)) {
            $this->uniqueVariables->removeElement($variable);
        }
    }

    /**
     * @return Collection<array-key, GeneratorVariable>
     */
    public function getGeneratorVariables(): Collection
    {
        return $this->generatorVariables;
    }

    /**
     * @param GeneratorVariable $variable
     */
    public function removeGeneratorVariable(GeneratorVariable $variable): void
    {
        if ($this->generatorVariables->contains($variable)) {
            $this->generatorVariables->removeElement($variable);
        }
    }

    /**
     * @return string
     */
    public function getGeneratedPrefix(): string
    {
        return $this->generatedPrefix;
    }

    /**
     * @param string $generatedPrefix
     * @return GeneratorVariable
     */
    public function setGeneratedPrefix(string $generatedPrefix): GeneratorVariable
    {
        $this->generatedPrefix = $generatedPrefix;
        return $this;
    }

    /**
     * @param Variable $variable
     */
    public function addGeneratedVariable(Variable $variable)
    {
        if ($variable instanceof UniqueVariable) {
            $this->addUniqueVariable($variable);
        } else if ($variable instanceof GeneratorVariable) {
            $this->addGeneratorVariable($variable);
        }
    }

    /**
     * @param UniqueVariable $variable
     */
    public function addUniqueVariable(UniqueVariable $variable): void
    {
        if (!$this->uniqueVariables->contains($variable)) {
            $variable->setGeneratorVariable($this);
            $this->uniqueVariables->add($variable);
        }
    }

    /**
     * @param GeneratorVariable $variable
     */
    public function addGeneratorVariable(GeneratorVariable $variable): void
    {
        if (!$this->generatorVariables->contains($variable)) {
            $variable->setGeneratorVariable($this);
            $this->generatorVariables->add($variable);
        }
    }

    /**
     * @return bool
     */
    public function isNumeralIncrement(): bool
    {
        return $this->numeralIncrement;
    }

    /**
     * @param bool $numeralIncrement
     * @return GeneratorVariable
     */
    public function setNumeralIncrement(bool $numeralIncrement): GeneratorVariable
    {
        $this->numeralIncrement = $numeralIncrement;
        return $this;
    }
}
