<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class PathBase
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={},
 *         "post"={}
 *     },
 *     itemOperations={
 *         "get"={},
 *         "patch"={},
 *         "delete"={}
 *     }
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={"project": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"project_explorer_view"}})
 *
 * @ORM\Entity
 * @ORM\Table(name="path_base", schema="webapp")
 */
class PathBase extends IdentifiedEntity
{
    /**
     * @var string
     *
     * @ORM\Column(type="string", nullable=false)
     * @Assert\NotBlank
     * @Groups({"project_explorer_view"})
     */
    private string $name;

    /**
     * @var Project
     * @ORM\OneToOne(targetEntity="Webapp\Core\Entity\Project", inversedBy="pathBase")
     * @Assert\NotBlank
     */
    private Project $project;

    /**
     * @var Collection|array<PathFilterNode>
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\PathFilterNode", mappedBy="pathBase", cascade={"remove", "persist"}, orphanRemoval=true)
     */
    private $pathFilterNodes;

    /**
     * @var bool
     * @ORM\Column(type="boolean")
     */
    private bool $askWhenEntering;

    /**
     * @var Collection|array<PathLevelAlgorithm>
     * @Groups({"variable_synthesis"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\PathLevelAlgorithm", mappedBy="pathBase", cascade={"remove", "persist"}, orphanRemoval=true)
     */
    private $pathLevelAlgorithms;

    /**
     * @var array
     * @ORM\Column(type="simple_array", nullable=true)
     */
    private array $selectedIris;

    /**
     * @var array
     * @ORM\Column(type="simple_array", nullable=true)
     */
    private array $orderedIris;

    /**
     * @var Collection<array-key, PathUserWorkflow>
     * @Groups({"project_explorer_view", "project_synthesis", "variable_synthesis"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\PathUserWorkflow", mappedBy="pathBase", cascade={"remove", "persist"})
     */
    private $userPaths;

    public function __construct()
    {
        $this->pathFilterNodes = new ArrayCollection();
        $this->pathLevelAlgorithms = new ArrayCollection();
        $this->userPaths = new ArrayCollection();
        $this->selectedIris = [];
        $this->orderedIris = [];
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return PathBase
     */
    public function setName(string $name): PathBase
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return Project
     */
    public function getProject(): Project
    {
        return $this->project;
    }

    /**
     * @param Project $project
     * @return PathBase
     */
    public function setProject(Project $project): PathBase
    {
        $this->project = $project;
        return $this;
    }

    /**
     * @return Collection|PathFilterNode[]
     */
    public function getPathFilterNodes()
    {
        return $this->pathFilterNodes;
    }

    /**
     * @param Collection|PathFilterNode[] $pathFilterNodes
     * @return PathBase
     */
    public function setPathFilterNodes($pathFilterNodes)
    {
        $this->pathFilterNodes = $pathFilterNodes;
        return $this;
    }

    /**
     * @param PathFilterNode $pathFilterNode
     * @return PathBase
     */
    public function addPathFilterNode($pathFilterNode)
    {
        if (!$this->pathFilterNodes->contains($pathFilterNode)) {
            $this->pathFilterNodes->add($pathFilterNode);
            $pathFilterNode->setPathBase($this);
        }
        return $this;
    }

    /**
     * @param PathFilterNode $pathFilterNode
     * @return PathBase
     */
    public function removePathFilterNode($pathFilterNode)
    {
        if ($this->pathFilterNodes->contains($pathFilterNode)) {
            $this->pathFilterNodes->removeElement($pathFilterNode);
            $pathFilterNode->setPathBase(null);
        }
        return $this;
    }

    /**
     * @return bool
     */
    public function isAskWhenEntering(): bool
    {
        return $this->askWhenEntering;
    }

    /**
     * @param bool $askWhenEntering
     * @return PathBase
     */
    public function setAskWhenEntering(bool $askWhenEntering): PathBase
    {
        $this->askWhenEntering = $askWhenEntering;
        return $this;
    }

    /**
     * @return Collection|PathLevelAlgorithm[]
     */
    public function getPathLevelAlgorithms()
    {
        return $this->pathLevelAlgorithms;
    }

    /**
     * @param Collection|PathLevelAlgorithm[] $pathLevelAlgorithms
     * @return PathBase
     */
    public function setPathLevelAlgorithms($pathLevelAlgorithms)
    {
        $this->pathLevelAlgorithms = $pathLevelAlgorithms;
        return $this;
    }

    /**
     * @param PathLevelAlgorithm $pathLevelAlgorithm
     * @return PathBase
     */
    public function addPathLevelAlgorithm($pathLevelAlgorithm)
    {
        if (!$this->pathLevelAlgorithms->contains($pathLevelAlgorithm)) {
            $this->pathLevelAlgorithms->add($pathLevelAlgorithm);
            $pathLevelAlgorithm->setPathBase($this);
        }
        return $this;
    }

    /**
     * @param PathLevelAlgorithm $pathLevelAlgorithm
     * @return PathBase
     */
    public function removePathLevelAlgorithm($pathLevelAlgorithm)
    {
        if ($this->pathLevelAlgorithms->contains($pathLevelAlgorithm)) {
            $this->pathLevelAlgorithms->removeElement($pathLevelAlgorithm);
            $pathLevelAlgorithm->setPathBase(null);
        }
        return $this;
    }

    /**
     * @return array
     */
    public function getSelectedIris(): array
    {
        return $this->selectedIris;
    }

    /**
     * @param array $selectedIris
     * @return PathBase
     */
    public function setSelectedIris(array $selectedIris): PathBase
    {
        $this->selectedIris = $selectedIris;
        return $this;
    }

    /**
     * @return array
     */
    public function getOrderedIris(): array
    {
        return $this->orderedIris;
    }

    /**
     * @param array $orderedIris
     * @return PathBase
     */
    public function setOrderedIris(array $orderedIris): PathBase
    {
        $this->orderedIris = $orderedIris;
        return $this;
    }

    /**
     * @return Collection<array-key, PathUserWorkflow>
     */
    public function getUserPaths()
    {
        return $this->userPaths;
    }

    /**
     * @param Collection<array-key, PathUserWorkflow> $userPaths
     * @return PathBase
     */
    public function setUserPaths($userPaths)
    {
        $this->userPaths = $userPaths;
        return $this;
    }

    /**
     * @param PathUserWorkflow $pathUserWorkflow
     * @return PathBase
     */
    public function addPathUserWorkflow($pathUserWorkflow)
    {
        if (!$this->userPaths->contains($pathUserWorkflow)) {
            $this->userPaths->add($pathUserWorkflow);
            $pathUserWorkflow->setPathBase($this);
        }
        return $this;
    }

    /**
     * @param PathUserWorkflow $pathUserWorkflow
     * @return PathBase
     */
    public function removePathUserWorkflow($pathUserWorkflow)
    {
        if ($this->userPaths->contains($pathUserWorkflow)) {
            $this->userPaths->removeElement($pathUserWorkflow);
            $pathUserWorkflow->setPathBase(null);
        }
        return $this;
    }

}
