<?php

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Enumeration\Annotation\EnumType;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\Enumeration\PathLevelEnum;

/**
 * Class RequiredAnnotation.
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER')"},
 *         "post"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER')"
 *          }
 *     },
 *     itemOperations={
 *         "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER')"},
 *         "patch"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER')"
 *          },
 *         "delete"={"is_granted('ROLE_PLATFORM_MANAGER')"},
 *     }
 * )
 *
 * @ORM\Entity()
 * @ORM\Table(name="required_annotation", schema="webapp")
 *
 * @psalm-import-type PathLevelEnumId from PathLevelEnum
 */
class RequiredAnnotation extends IdentifiedEntity
{
    /**
     * @var Project|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Project", inversedBy="requiredAnnotations")
     */
    private ?Project $project;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", nullable=false)
     * @EnumType(class="Webapp\Core\Enumeration\AnnotationKindEnum")
     * @Groups({"project_explorer_view"})
     */
    private int $type;

    /**
     * @var string
     * @psalm-var PathLevelEnumId|''
     *
     * @ORM\Column(type="string", nullable=false)
     * @Groups({"project_explorer_view", "data_entry_synthesis"})
     */
    private string $level;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", nullable=false)
     */
    private bool $askWhenEntering;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    private string $comment;

    /**
     * @return Project|null
     */
    public function getProject(): ?Project
    {
        return $this->project;
    }

    /**
     * @param Project|null $project
     * @return RequiredAnnotation
     */
    public function setProject(?Project $project): RequiredAnnotation
    {
        $this->project = $project;
        return $this;
    }

    /**
     * @return int
     */
    public function getType(): int
    {
        return $this->type;
    }

    /**
     * @param int $type
     * @return RequiredAnnotation
     */
    public function setType(int $type): RequiredAnnotation
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return string
     * @psalm-return PathLevelEnumId|''
     */
    public function getLevel(): string
    {
        return $this->level;
    }

    /**
     * @param string $level
     * @psalm-param PathLevelEnumId|'' $level
     * @return RequiredAnnotation
     */
    public function setLevel(string $level): RequiredAnnotation
    {
        $this->level = $level;
        return $this;
    }

    /**
     * @return bool
     */
    public function isAskWhenEntering(): bool
    {
        return $this->askWhenEntering;
    }

    /**
     * @param bool $askWhenEntering
     * @return RequiredAnnotation
     */
    public function setAskWhenEntering(bool $askWhenEntering): RequiredAnnotation
    {
        $this->askWhenEntering = $askWhenEntering;
        return $this;
    }

    /**
     * @return string
     */
    public function getComment(): string
    {
        return $this->comment;
    }

    /**
     * @param string $comment
     * @return RequiredAnnotation
     */
    public function setComment(string $comment): RequiredAnnotation
    {
        $this->comment = $comment;
        return $this;
    }
}
