<?php

namespace Webapp\FileManagement\Service;

use DateTime;
use DateTimeInterface;
use Doctrine\Common\Collections\Collection;
use DOMDocument;
use DOMElement;
use DOMNode;
use Exception;
use Mobile\Device\Entity\Anomaly;
use Mobile\Device\Entity\Block;
use Mobile\Device\Entity\Device;
use Mobile\Device\Entity\EntryNote;
use Mobile\Device\Entity\Factor;
use Mobile\Device\Entity\Individual;
use Mobile\Device\Entity\Modality;
use Mobile\Device\Entity\OutExperimentationZone;
use Mobile\Device\Entity\Protocol;
use Mobile\Device\Entity\SubBlock;
use Mobile\Device\Entity\Treatment;
use Mobile\Device\Entity\UnitParcel;
use Mobile\Measure\Entity\Annotation;
use Mobile\Measure\Entity\DataEntry;
use Mobile\Measure\Entity\FormField;
use Mobile\Measure\Entity\GeneratedField;
use Mobile\Measure\Entity\Measure;
use Mobile\Measure\Entity\Session;
use Mobile\Measure\Entity\Test\CombinationTest;
use Mobile\Measure\Entity\Test\GrowthTest;
use Mobile\Measure\Entity\Test\PreconditionedCalculation;
use Mobile\Measure\Entity\Test\RangeTest;
use Mobile\Measure\Entity\Variable\Base\Variable;
use Mobile\Measure\Entity\Variable\Driver;
use Mobile\Measure\Entity\Variable\GeneratorVariable;
use Mobile\Measure\Entity\Variable\Mark;
use Mobile\Measure\Entity\Variable\Material;
use Mobile\Measure\Entity\Variable\Scale;
use Mobile\Measure\Entity\Variable\StateCode;
use Mobile\Measure\Entity\Variable\UniqueVariable;
use Mobile\Measure\Entity\Variable\ValueHint;
use Mobile\Measure\Entity\Variable\ValueHintList;
use Mobile\Project\Entity\DataEntryProject;
use Mobile\Project\Entity\DesktopUser;
use Mobile\Project\Entity\NatureZHE;
use Mobile\Project\Entity\Platform;
use Mobile\Project\Entity\Workpath;
use ReflectionException;
use Webapp\Core\Enumeration\PathLevelEnum;
use Webapp\Core\Enumeration\VariableTypeEnum;

/**
 * Class WriterHelper.
 */
class WriterHelper
{
    public array $objectAnnotations;

    public array $annotationFiles;

    public array $markFiles;

    public array $variablesTuples;

    private array $anomalies;

    public function init()
    {
        $this->annotationFiles = [];
        $this->markFiles = [];
        $this->objectAnnotations = [];
        $this->variablesTuples = [];
    }

    /**
     * @param DOMDocument $dom
     * @param DataEntry $dataEntry
     * @param $uriMap
     * @param $uriIndex
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructDataEntryElement(DOMDocument $dom, DataEntry $dataEntry, &$uriMap, $uriIndex): DOMElement
    {
        $dataEntryXml = $dom->createElement('adonis.modeleMetier.saisieTerrain:Saisie');
        $dataEntryXml->setAttribute('projetDeSaisie', $uriMap[$dataEntry->getProject()->getUri()]);
        $dataEntryXml->setAttribute('dateDebut', $dataEntry->getStartedAt()->format(DateTimeInterface::RFC3339_EXTENDED));
        $dataEntryXml->setAttribute('dateFin', $dataEntry->getEndedAt()->format(DateTimeInterface::RFC3339_EXTENDED));
        $dataEntryXml->setAttribute('statusSaisie', 'terminee');
        $dataEntryXml->setAttribute('cheminementImpose', $dataEntry->getProject() instanceof DataEntryProject && $dataEntry->getProject()->getWorkpaths()->count() > 0 ? 'true' : 'false');
        $dataEntryXml->setAttribute('plateforme', $uriMap[$dataEntry->getProject()->getPlatform()->getUri()]);

        $uriMap[$dataEntry->getUri()] = "/" . $uriIndex;
        $count = 0;
        foreach ($dataEntry->getProject()->getUniqueVariables() as $uniqueVariable) {
            $variableXml = $this->constructUniqueVariableElement($dom, $uniqueVariable, $uriMap, $count);
            $dataEntryXml->appendChild($variableXml);
            $this->variablesTuples[] = [$uniqueVariable, $variableXml];
            $count++;
        }

        foreach ($dataEntry->getProject()->getGeneratorVariables() as $generatorVariable) {
            $variableXml = $this->constructGeneratorVariableElement($dom, $generatorVariable, $uriMap, $count);
            $dataEntryXml->appendChild($variableXml);
            $this->variablesTuples[] = [$generatorVariable, $variableXml];
            $count++;
        }

        $count = 0;
        foreach ($dataEntry->getProject()->getMaterials() as $material) {
            $materialXml = $this->constructMaterialElement($dom, $material, $uriMap, $count);
            $dataEntryXml->appendChild($materialXml);
            $count++;
        }

        $count = 0;

        foreach ($dataEntry->getSessions() as $session) {
            $sessionXml = $this->constructSessionElement($dom, $session, $uriMap, $count, $count === $dataEntry->getSessions()->count() - 1);
            $dataEntryXml->appendChild($sessionXml);
            $count++;
        }

        return $dataEntryXml;
    }

    /**
     * @param DOMDocument $dom
     * @param UniqueVariable $uniqueVariable
     * @param $uriMap
     * @param $uriIndex
     * @param bool $isGenerated
     * @param bool $isDataEntry
     * @param DataEntry|null $connectedDataEntry
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructUniqueVariableElement(DOMDocument $dom, UniqueVariable $uniqueVariable, &$uriMap, $uriIndex, bool $isGenerated = false, bool $isDataEntry = true, DataEntry $connectedDataEntry = null): DOMElement
    {
        $variableXML = $this->constructVariableElement($dom, $uniqueVariable, $uriMap, $uriIndex, $isGenerated, $isDataEntry, $connectedDataEntry);

        $valueHintListXML = $this->constructValueHintListXML($uniqueVariable, $dom);
        if (!is_null($valueHintListXML)) {
            $variableXML->appendChild($valueHintListXML);
        }

        return $variableXML;
    }

    /**
     * @param DOMDocument $dom
     * @param Variable $variable
     * @param $uriMap
     * @param $uriIndex
     * @param bool $isGenerated
     * @param bool $isDataEntry
     * @param DataEntry|null $connectedDataEntry
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructVariableElement(DOMDocument $dom, Variable $variable, &$uriMap, $uriIndex, bool $isGenerated = false, bool $isDataEntry = true, DataEntry $connectedDataEntry = null): DOMElement
    {
        if ($connectedDataEntry !== null) {
            $variableXml = $dom->createElement("variablesConnectees");
        } else {
            $variableXml = $dom->createElement($isGenerated ? "variablesGenerees" : "variables");
        }
        $variableXml->setAttribute("nom", $variable->getName());
        $variableXml->setAttribute("nomCourt", $variable->getShortName());
        $variableXml->setAttribute("nbRepetitionSaisies", $variable->getRepetitions());
        $variableXml->setAttribute("unite", $variable->getUnit());
        $variableXml->setAttribute("horodatage", $variable->isAskTimestamp() ? "true" : "false");
        $variableXml->setAttribute("uniteParcoursType", $variable->getPathLevel());
        $variableXml->setAttribute("dateCreation", $variable->getCreationDate()->format(DateTime::RFC3339_EXTENDED));
        $variableXml->setAttribute("dateDerniereModification", $variable->getModificationDate()->format(DateTime::RFC3339_EXTENDED));
        $variableXml->setAttribute("commentaire", $variable->getComment());
        $variableXml->setAttribute("ordre", $variable->getOrder());
        $variableXml->setAttribute("active", $variable->isActive() ? "true" : "false");
        $variableXml->setAttribute("format", $variable->getFormat());
        switch ($variable->getType()) {
            case VariableTypeEnum::ALPHANUMERIC:
                $variableXml->setAttribute("typeVariable", "alphanumerique");
                break;
            case VariableTypeEnum::BOOLEAN:
                $variableXml->setAttribute("typeVariable", "booleen");
                break;
            case VariableTypeEnum::DATE:
                $variableXml->setAttribute("typeVariable", "date");
                break;
            case VariableTypeEnum::HOUR:
                $variableXml->setAttribute("typeVariable", "heure");
                break;
            case VariableTypeEnum::INTEGER:
                $variableXml->setAttribute("typeVariable", "entiere");
                break;
            case VariableTypeEnum::REAL:
                $variableXml->setAttribute("typeVariable", "reel");
                break;
            default:
                $variableXml->setAttribute("typeVariable", $variable->getType());
                break;
        }

        $variableXml->setAttribute("saisieObligatoire", $variable->isMandatory() ? "true" : "false");
        if ($variable->getMaterial() !== null) {
            $variableXml->setAttribute("debut", $variable->getFrameStartPosition());
            $variableXml->setAttribute("fin", $variable->getFrameEndPosition());
            $variableXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:VariableSemiAuto");
        }

        if ($variable->getDefaultValue() !== null && $variable->getDefaultValue() != "") {
            $variableXml->setAttribute("valeurDefaut", $variable->getDefaultValue());
        }
        if ($variable->getConnectedDataEntryProject() !== null) {
            $variableXml->setAttribute("prechargee", 'true');
        }
        if ($isGenerated) {
            $uriMap[$variable->getUri()] = $uriMap[$variable->getGeneratorVariable()->getUri()] . "/@variablesGenerees." . $uriIndex;
        } else if ($connectedDataEntry !== null) {
            $uriMap[$variable->getUri()] = $uriMap[$connectedDataEntry->getUri()] . "/@variablesConnectees." . $uriIndex;
        } else {
            //Variables have to be duplicated in the project and in the data entry
            $uriMap[$variable->getUri()] = $uriMap[$isDataEntry ?
                    $variable->getProject()->getDataEntry()->getUri() :
                    $variable->getProject()->getUri()] . "/@variables." . $uriIndex;
        }

        return $variableXml;
    }

    /**
     * @param DOMDocument $dom
     * @param GeneratorVariable $generatorVariable
     * @param $uriMap
     * @param $uriIndex
     * @param bool $isGenerated
     * @param bool $isDataEntry
     * @param DataEntry|null $connectedDataEntry
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructGeneratorVariableElement(DOMDocument $dom, GeneratorVariable $generatorVariable, &$uriMap, $uriIndex, bool $isGenerated = false, bool $isDataEntry = true, DataEntry $connectedDataEntry = null): DOMElement
    {
        $variableXml = $this->constructVariableElement($dom, $generatorVariable, $uriMap, $uriIndex, $isGenerated, $isDataEntry, $connectedDataEntry);
        $variableXml->setAttribute("nomGenere", $generatorVariable->getGeneratedPrefix());
        $variableXml->setAttribute("extension", $generatorVariable->isNumeralIncrement() ? "numerique" : "alphabetique");
        $variableXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:VariableGeneratrice");

        $count = 0;
        foreach ($generatorVariable->getUniqueVariables() as $uniqueVariable) {
            /** @var UniqueVariable $uniqueVariable */
            $variableGenereeXml = $this->constructUniqueVariableElement($dom, $uniqueVariable, $uriMap, $count, true);
            $variableXml->appendChild($variableGenereeXml);
            $this->variablesTuples[] = [$uniqueVariable, $variableGenereeXml];
            $count++;
        }
        foreach ($generatorVariable->getGeneratorVariables() as $generatorVariable) {
            /** @var GeneratorVariable $generatorVariable */
            $variableGenereeXml = $this->constructGeneratorVariableElement($dom, $generatorVariable, $uriMap, $count, true);
            $variableXml->appendChild($variableGenereeXml);
            $this->variablesTuples[] = [$generatorVariable, $variableGenereeXml];
            $count++;
        }
        return $variableXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Material $material
     * @param $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    private function constructMaterialElement(DOMDocument $dom, Material $material, &$uriMap, int $count): DOMElement
    {
        $materialXml = $dom->createElement("materiel");
        $materialXml->setAttribute("nom", $material->getName());
        $materialXml->setAttribute("fabriquant", $material->getManufacturer());
        $materialXml->setAttribute("appareil", $material->getPhysicalDevice());
        $materialXml->setAttribute("type", $material->getType());

        $driverXml = $this->constructDriverElement($dom, $material->getDriver());
        $materialXml->appendChild($driverXml);

        $str = "";
        foreach ($material->getUniqueVariables() as $uniqueVariable) {
            /** @var UniqueVariable $uniqueVariable */
            $str = $str . $uriMap[$uniqueVariable->getUri()] . " ";
        }
        foreach ($material->getGeneratorVariables() as $generatorVariable) {
            /** @var GeneratorVariable $generatorVariable */
            $str = $str . $uriMap[$generatorVariable->getUri()] . " ";
        }
        $materialXml->setAttribute("variables", $str);

        $uriMap[$material->getUri()] = $uriMap[$material->getProject()->getUri()] . "/@materiel." . $count;

        return $materialXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Driver $driver
     * @return DOMElement
     */
    private function constructDriverElement(DOMDocument $dom, Driver $driver): DOMElement
    {
        $driverXml = $dom->createElement("driver");

        $driverXml->setAttribute("debutTrame", $driver->getFrameStart());
        $driverXml->setAttribute("finTrame", $driver->getFrameEnd());
        $driverXml->setAttribute("separateurCsv", $driver->getCsvSeparator());
        $driverXml->setAttribute("timeout", $driver->getTimeout());
        $driverXml->setAttribute("tailleTrame", $driver->getFrameLength());
        if ($driver->getType() !== Driver::RS232) {
            $driverXml->setAttribute("type", $driver->getType());
        } else {
            $driverXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:DriverCom");
            $driverXml->setAttribute("baudrate", $driver->getBaudrate());
            $driverXml->setAttribute("stopbit", $driver->getStopbit());
            $driverXml->setAttribute("push", $driver->getPush() ? "true" : "false");
            $driverXml->setAttribute("parity", $driver->getParity());
            $driverXml->setAttribute("request", $driver->getRequest());
            $driverXml->setAttribute("flowcontrol", $driver->getFlowControl());
            $driverXml->setAttribute("port", $driver->getPort());
            $driverXml->setAttribute("databitsFormat", $driver->getDatabitsFormat());
        }
        return $driverXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Session $session
     * @param $uriMap
     * @param $uriIndex
     * @param $isLast
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructSessionElement(DOMDocument $dom, Session $session, &$uriMap, $uriIndex, $isLast): DOMElement
    {
        $sessionXml = $dom->createElement('sessions');
        $sessionXml->setAttribute('dateDebut', $session->getStartedAt()->format(DateTime::RFC3339_EXTENDED));
        $sessionXml->setAttribute('dateFin', $session->getEndedAt()->format(DateTime::RFC3339_EXTENDED));
        $sessionXml->setAttribute('isAuthenticated', "true");
        $uriMap[$session->getUri()] = $uriMap[$session->getDataEntry()->getUri()] . "/@sessions." . $uriIndex;

        $count = 0;
        foreach ($session->getFormFields() as $formField) {
            $this->constructMeasuresElement($dom, $formField, $uriMap, $count, $sessionXml, $session->getUri());
        }

        if ($isLast) {
            foreach ($this->objectAnnotations as $annotationXml) {
                $sessionXml->appendChild($annotationXml);
            }
        }

        return $sessionXml;
    }

    /**
     * @param DOMDocument $dom
     * @param FormField $formField
     * @param $uriMap
     * @param $count
     * @param DOMNode $sessionXml
     * @param string $sessionURI
     * @throws ReflectionException
     */
    public function constructMeasuresElement(DOMDocument $dom, FormField $formField, &$uriMap, &$count, DOMNode $sessionXml, string $sessionURI): void
    {
        foreach ($formField->getMeasures() as $measure) {
            /** @var Measure $measure */
            $measureXml = $this->constructMeasureElement($dom, $measure, $uriMap);
            $variable = $formField->getVariable();
            switch ($variable->getType()) {
                case VariableTypeEnum::ALPHANUMERIC:
                    $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique");
                    break;
                case VariableTypeEnum::REAL:
                    $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableReel");
                    if ($measure->getState() === null && $variable->getFormat() !== null && $measure->getValue() !== null) {
                        $measureXml->setAttribute("valeur", number_format(floatval($measure->getValue()), intval($variable->getFormat()), '.', ''));
                    }
                    break;
                case VariableTypeEnum::BOOLEAN:
                    $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableBooleen");
                    break;
                case VariableTypeEnum::INTEGER:
                    $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableEntiere");
                    break;
                case VariableTypeEnum::DATE:
                    $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableDate");
                    break;
                case VariableTypeEnum::HOUR:
                    $measureXml->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableHeure");
                    break;
            }
            $measureXml->setAttribute("variable", $uriMap[$variable->getUri()]);
            $measureXml->setAttribute("objetMetier", $uriMap[$formField->getTarget()->getUri()]);
            if ($formField->getGeneratedField() !== null) {
                $measureXml->setAttribute("indiceGeneratrice", $formField->getGeneratedField()->getIndex());
                $measureXml->setAttribute("mesureGeneratrice", $uriMap[$formField->getGeneratedField()->getFormField()->getMeasures()->first()->getUri()]);
            }
            $uriMap[$measure->getUri()] = $uriMap[$sessionURI] . '/@mesureVariables.' . $count;
            $sessionXml->appendChild($measureXml);

            foreach ($measure->getAnnotations() as $annotation) {
                /** @var Annotation $annotation */
                $annotationXml = $this->constructAnnotationElement($dom, $annotation, $uriMap);
                $annotationXml->setAttribute("mesureVariable", $uriMap[$measure->getUri()]);
                $sessionXml->appendChild($annotationXml);
            }

            $count++;
        }
        if ($formField->getGeneratedFields()->count() > 0) {
            //In the generator variable case, only one measure is concerned by the formField
            /** @var DOMElement $measureXml */
            $str = "";
            foreach ($formField->getGeneratedFields() as $generatedField) {
                /** @var GeneratedField $generatedField */
                foreach ($generatedField->getFormFields() as $generatedFormField) {
                    /** @var FormField $generatedFormField */
                    $this->constructMeasuresElement($dom, $generatedFormField, $uriMap, $count, $sessionXml, $sessionURI);
                    foreach ($generatedFormField->getMeasures() as $generatedMeasure) {
                        /** @var Measure $generatedMeasure */
                        $str = $str . $uriMap[$generatedMeasure->getUri()] . " ";
                    }
                }
            }
            $measureXml->setAttribute("mesuresGenerees", $str);
        }
    }

    /**
     * @param DOMDocument $dom
     * @param Measure $measure
     * @param $uriMap
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructMeasureElement(DOMDocument $dom, Measure $measure, &$uriMap): DOMElement
    {
        $measureXml = $dom->createElement("mesureVariables");

        $measureXml->setAttribute("horodatage", $measure->getTimestamp()->format(DateTime::RFC3339_EXTENDED));
        if ($measure->getState() !== null) {
            if ($measure->getState()->getPropagation() === null) {
                $measureXml->setAttribute("codeEtat", $uriMap[$measure->getState()->getUri()]);
            } else {
                $measureXml->setAttribute("codeEtat", $uriMap[$measure->getFormField()->getTarget()->getUri() . $measure->getState()->getUri()]);
            }

        } else {
            $measureXml->setAttribute("valeur", $measure->getValue());
        }
        return $measureXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Annotation $annotation
     * @param $uriMap
     * @return DOMElement
     * @throws Exception
     */
    public function constructAnnotationElement(DOMDocument $dom, Annotation $annotation, $uriMap): DOMElement
    {
        $annotationXml = $dom->createElement("metadonnees");
        $annotationXml->setAttribute("nom", $annotation->getName());
        switch ($annotation->getType()) {
            case Annotation::ANNOT_TYPE_TEXT:
                $annotationXml->setAttribute("nature", "texte");
                $annotationXml->setAttribute("donnee", $annotation->getValue());
                break;
            case Annotation::ANNOT_TYPE_IMAG:
                $annotationXml->setAttribute("nature", "photo");
                $annotationXml->setAttribute("donnee", count($this->annotationFiles));
                $this->annotationFiles[] = $annotation->getImage();
                break;
        }
        $annotationXml->setAttribute("date", $annotation->getTimestamp()->format(DateTime::RFC3339_EXTENDED));
        foreach ($annotation->getCategories() as $category) {
            $categorieXml = $dom->createElement("categorie", $category);
            $annotationXml->appendChild($categorieXml);
        }

        foreach ($annotation->getKeywords() as $keyword) {
            $keywordXml = $dom->createElement("motscles", $keyword);
            $annotationXml->appendChild($keywordXml);
        }

        return $annotationXml;
    }

    /**
     * @param DOMDocument $dom
     * @param DataEntryProject $dataEntryProject
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructDataEntryProjectElement(DOMDocument $dom, DataEntryProject $dataEntryProject, array &$uriMap, int $count): DOMElement
    {
        $dataEntryProjectXml = $dom->createElement("adonis.modeleMetier.projetDeSaisie:ProjetDeSaisie");
        $dataEntryProjectXml->setAttribute("nom", $dataEntryProject->getName());
        $dataEntryProjectXml->setAttribute("dateCreation", $dataEntryProject->getCreationDate()->format(DateTime::RFC3339_EXTENDED));
        $dataEntryProjectXml->setAttribute("createur", $uriMap[$dataEntryProject->getCreator()->getUri()]);
        $dataEntryProjectXml->setAttribute("plateforme", $uriMap[$dataEntryProject->getPlatform()->getUri()]);

        $str = "";
        foreach ($dataEntryProject->getPlatform()->getDevices() as $device) {
            /** @var Device $device */
            $str = $str . $uriMap[$device->getUri()] . " ";
        }
        $dataEntryProjectXml->setAttribute("dispositifs", $str);
        if ($dataEntryProject->isImprovised()) {
            $dataEntryProjectXml->setAttribute("etat", "improvise");
        } else {
            $dataEntryProjectXml->setAttribute("etat", "transferer");
        }
        $str = "";
        foreach ($dataEntryProject->getDesktopUsers() as $desktopUser) {
            /** @var DesktopUser $desktopUser */
            $str = $str . $uriMap[$desktopUser->getUri()] . " ";
        }
        $dataEntryProjectXml->setAttribute("experimentateurs", $str);

        $uriMap[$dataEntryProject->getUri()] = "/$count";

        $count = 0;
        foreach ($dataEntryProject->getUniqueVariables() as $variable) {
            /** @var UniqueVariable $variable */
            $variableXml = $this->constructUniqueVariableElement($dom, $variable, $uriMap, $count, false, false);
            $dataEntryProjectXml->appendChild($variableXml);
            $this->variablesTuples[] = [$variable, $variableXml];
            $count++;
        }
        foreach ($dataEntryProject->getGeneratorVariables() as $variable) {
            /** @var GeneratorVariable $variable */
            $variableXml = $this->constructGeneratorVariableElement($dom, $variable, $uriMap, $count, false, false);
            $dataEntryProjectXml->appendChild($variableXml);
            $this->variablesTuples[] = [$variable, $variableXml];
            $count++;
        }

        $count = 0;
        foreach ($dataEntryProject->getMaterials() as $material) {
            $materialXml = $this->constructMaterialElement($dom, $material, $uriMap, $count);
            $dataEntryProjectXml->appendChild($materialXml);
            $count++;
        }

        $count = 0;
        foreach ($dataEntryProject->getStateCodes() as $stateCode) {
            /** @var StateCode $stateCode */
            $stateCodeXml = $this->constructStateCodeElement($dom, $stateCode, $uriMap, $count);
            $dataEntryProjectXml->appendChild($stateCodeXml);
            $count++;
        }

        if ($dataEntryProject->getResponseFile() === null && $dataEntryProject->getWorkpaths()->count() > 0) {
            $workpathLibXml = $dom->createElement('sectionCheminements');
            $workpathLibXml->setAttribute("name", "Cheminement");
            $dataEntryProjectXml->appendChild($workpathLibXml);
            $count++;
            foreach ($dataEntryProject->getWorkpaths() as $workpath) {
                $workpathXml = $this->constructWorkpathElement($dom, $workpath, $dataEntryProject->getDesktopUsers(), $uriMap);
                $workpathLibXml->appendChild($workpathXml);
            }
            $dataEntryProjectXml->setAttribute("cheminementCalcules", join(" ", array_map(function ($index) use ($dataEntryProject, $uriMap) {
                return $uriMap[$dataEntryProject->getUri()] . "/" . "@sectionCheminements.0/@cheminementCalcules." . $index;
            }, array_keys(array_fill(0, $dataEntryProject->getWorkpaths()->count(), 0)))));
        }

        return $dataEntryProjectXml;

    }

    /**
     * @param DOMDocument $dom
     * @param StateCode $stateCode
     * @param array $uriMap
     * @param int $count
     * @param bool $forDataEntryProject
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructStateCodeElement(DOMDocument $dom, StateCode $stateCode, array &$uriMap, int $count, bool $forDataEntryProject = true): DOMElement
    {
        if ($forDataEntryProject) {
            $stateCodeXml = $dom->createElement("codesEtats");
        } else {
            $stateCodeXml = $dom->createElement("codeEtat");
        }
        $stateCodeXml->setAttribute("code", $stateCode->getCode());
        $stateCodeXml->setAttribute("intitule", $stateCode->getLabel());
        $stateCodeXml->setAttribute("signification", $stateCode->getDescription());
        if ($stateCode->getPropagation() !== null) {
            switch ($stateCode->getPropagation()) {
                case PathLevelEnum::INDIVIDUAL:
                    $stateCodeXml->setAttribute("propagation", "individu");
                    break;
                case PathLevelEnum::UNIT_PLOT:
                    $stateCodeXml->setAttribute("propagation", "parcelle");
                    break;
                default:
                    $stateCodeXml->setAttribute("propagation", $stateCode->getPropagation());
                    break;
            }
        }
        $stateCodeXml->setAttribute("couleur", $stateCode->getColor());

        if ($forDataEntryProject) {
            $uriMap[$stateCode->getUri()] = $uriMap[$stateCode->getProject()->getUri()] . "/@codesEtats." . $count;
        }

        return $stateCodeXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Platform $platform
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructPlatformElement(DOMDocument $dom, Platform $platform, array &$uriMap, int $count): DOMElement
    {
        $platformXml = $dom->createElement("adonis.modeleMetier.plateforme:Plateforme");
        $platformXml->setAttribute("nom", $platform->getName());
        $platformXml->setAttribute("dateCreation", $platform->getCreationDate()->format(DateTime::RFC3339_EXTENDED));
        $platformXml->setAttribute("nomSite", $platform->getSite());
        $platformXml->setAttribute("nomLieu", $platform->getPlace());
        $uriMap[$platform->getUri()] = "/$count";

        $count = 0;
        foreach ($platform->getDevices() as $device) {
            /** @var Device $device */
            $variableXml = $this->constructExperimentElement($dom, $device, $uriMap, $count);
            $platformXml->appendChild($variableXml);
            $count++;
        }

        return $platformXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Device $device
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructExperimentElement(DOMDocument $dom, Device $device, array &$uriMap, int $count): DOMElement
    {
        $this->anomalies = [];
        $deviceXml = $dom->createElement("dispositifs");
        $deviceXml->setAttribute("nom", $device->getName());
        $deviceXml->setAttribute("dateCreation", $device->getCreationDate()->format(DateTime::RFC3339_EXTENDED));
        $deviceXml->setAttribute("source", $device->getProtocol()->getName());
        $deviceXml->setAttribute("etat", "nonDeverrouillable");

        if ($device->getValidationDate() !== null) {
            $deviceXml->setAttribute("dateValidation", $device->getValidationDate()->format(DateTime::RFC3339_EXTENDED));
        }
        $uriMap[$device->getUri()] = $uriMap[$device->getPlatform()->getUri()] . "/@dispositifs." . $count;

        $protocolXml = $this->constructProtocolElement($dom, $device->getProtocol(), $uriMap);
        $deviceXml->appendChild($protocolXml);

        $count = 0;
        foreach ($device->getBlocks() as $block) {
            /** @var Block $block */
            $protocolXml = $this->constructBlockElement($dom, $block, $uriMap, $count);
            $deviceXml->appendChild($protocolXml);
            $count++;
        }
        $count = 0;
        foreach ($device->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $deviceXml->appendChild($outExperimentationZoneXml);
            $count++;
        }
        foreach ($device->getAnnotations() as $annotation) {
            $annotationXml = $this->constructAnnotationElement($dom, $annotation, $uriMap);
            $annotationXml->setAttribute("objetMetier", $uriMap[$device->getUri()]);
            $this->objectAnnotations[] = $annotationXml;
        }

        foreach ($this->anomalies as $anomaly) {
            /** @var Anomaly $anomaly */
            $anomalyXml = $this->constructAnomalyElement($dom, $anomaly, $uriMap);
            $deviceXml->appendChild($anomalyXml);
        }

        foreach ($device->getNotes() as $note) {
            /** @var EntryNote $note */
            $noteXml = $this->constructNoteElement($dom, $note);
            $deviceXml->appendChild($noteXml);
        }

        return $deviceXml;

    }

    /**
     * @param DOMDocument $dom
     * @param Protocol $protocol
     * @param array $uriMap
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructProtocolElement(DOMDocument $dom, Protocol $protocol, array &$uriMap): DOMElement
    {
        $protocolXml = $dom->createElement("protocoles");
        $protocolXml->setAttribute("nom", $protocol->getName());
        $protocolXml->setAttribute("dateCreation", $protocol->getCreationDate()->format(DateTime::RFC3339_EXTENDED));
        $protocolXml->setAttribute("objectifs", $protocol->getAim());
        $protocolXml->setAttribute("createur", $uriMap[$protocol->getCreator()->getUri()]);
        $protocolXml->setAttribute("algorithmeTirage", $protocol->getAlgorithm());

        $uriMap[$protocol->getUri()] = $uriMap[$protocol->getDevice()->getUri()] . "/@protocoles";

        $count = 0;
        foreach ($protocol->getFactors() as $factor) {
            /** @var Factor $factor */
            $factorXml = $this->constructFactorElement($dom, $factor, $uriMap, $count);
            $protocolXml->appendChild($factorXml);
            $count++;
        }
        $count = 0;
        foreach ($protocol->getTreatments() as $treatment) {
            /** @var Treatment $treatment */
            $factorXml = $this->constructTreatmentElement($dom, $treatment, $uriMap, $count);
            $protocolXml->appendChild($factorXml);
            $count++;
        }
        return $protocolXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Factor $factor
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructFactorElement(DOMDocument $dom, Factor $factor, array &$uriMap, int $count): DOMElement
    {
        $factorXml = $dom->createElement("facteurs");
        $factorXml->setAttribute("nom", $factor->getName());
        $uriMap[$factor->getUri()] = $uriMap[$factor->getProtocol()->getUri()] . "/@facteurs." . $count;

        $count = 0;
        foreach ($factor->getModalities() as $modality) {
            /** @var Modality $modality */
            $modalityXml = $this->constructModalityElement($dom, $modality, $uriMap, $count);
            $factorXml->appendChild($modalityXml);
            $count++;
        }
        return $factorXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Modality $modality
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructModalityElement(DOMDocument $dom, Modality $modality, array &$uriMap, int $count): DOMElement
    {
        $modalityXml = $dom->createElement("modalites");
        $modalityXml->setAttribute("valeur", $modality->getValue());
        $uriMap[$modality->getUri()] = $uriMap[$modality->getFactor()->getUri()] . "/@modalites." . $count;
        return $modalityXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Treatment $treatment
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructTreatmentElement(DOMDocument $dom, Treatment $treatment, array &$uriMap, int $count): DOMElement
    {
        $treatmentXml = $dom->createElement("traitements");
        $treatmentXml->setAttribute("nom", $treatment->getName());
        $treatmentXml->setAttribute("nomCourt", $treatment->getShortName());
        $treatmentXml->setAttribute("nbRepetitions", $treatment->getRepetitions());
        $str = "";
        foreach ($treatment->getModalities() as $modality) {
            /** @var Modality $modality */
            $str = $str . $uriMap[$modality->getUri()] . " ";
        }
        $treatmentXml->setAttribute("combinaisons", $str);
        $uriMap[$treatment->getUri()] = $uriMap[$treatment->getProtocol()->getUri()] . "/@traitements." . $count;

        return $treatmentXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Block $block
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructBlockElement(DOMDocument $dom, Block $block, array &$uriMap, int $count): DOMElement
    {
        $blockXml = $dom->createElement("blocs");
        $blockXml->setAttribute("numero", $block->getName());

        $uriMap[$block->getUri()] = $uriMap[$block->getDevice()->getUri()] . "/@blocs." . $count;

        $count = 0;
        foreach ($block->getSubBlocks() as $subBlock) {
            /** @var SubBlock $subBlock */
            $subBlockXml = $this->constructSubBlockElement($dom, $subBlock, $uriMap, $count);
            $blockXml->appendChild($subBlockXml);
            $count++;
        }
        $count = 0;
        foreach ($block->getUnitParcels() as $unitParcel) {
            /** @var UnitParcel $unitParcel */
            if ($unitParcel->getTreatment() === null) {
                //Case when the unit parcel comes from a treatment error
                foreach ($unitParcel->getOutExperimentationZones() as $outExperimentationZone) {
                    /** @var OutExperimentationZone $outExperimentationZone */
                    $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
                    $blockXml->appendChild($outExperimentationZoneXml);
                }
            } else {
                $unitParcelXml = $this->constructUnitParcelElement($dom, $unitParcel, $uriMap, $count);
                $blockXml->appendChild($unitParcelXml);
                $count++;
            }

        }
        foreach ($block->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $blockXml->appendChild($outExperimentationZoneXml);
            $count++;
        }
        foreach ($block->getAnnotations() as $annotation) {
            $annotationXml = $this->constructAnnotationElement($dom, $annotation, $uriMap);
            $annotationXml->setAttribute("objetMetier", $uriMap[$block->getUri()]);
            $this->objectAnnotations[] = $annotationXml;
        }
        foreach ($block->getNotes() as $note) {
            /** @var EntryNote $note */
            $noteXml = $this->constructNoteElement($dom, $note);
            $blockXml->appendChild($noteXml);
        }

        return $blockXml;
    }

    /**
     * @param DOMDocument $dom
     * @param SubBlock $subBlock
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructSubBlockElement(DOMDocument $dom, SubBlock $subBlock, array &$uriMap, int $count): DOMElement
    {
        $subBlockXml = $dom->createElement("sousBlocs");
        $subBlockXml->setAttribute("numero", $subBlock->getName());

        $uriMap[$subBlock->getUri()] = $uriMap[$subBlock->getBlock()->getUri()] . '/@sousBlocs.' . $count;

        $count = 0;
        foreach ($subBlock->getUnitParcels() as $unitParcel) {
            /** @var UnitParcel $unitParcel */
            if ($unitParcel->getTreatment() === null) {
                //Case when the unit parcel comes from a treatment error
                foreach ($unitParcel->getOutExperimentationZones() as $outExperimentationZone) {
                    /** @var OutExperimentationZone $outExperimentationZone */
                    $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
                    $subBlockXml->appendChild($outExperimentationZoneXml);
                    $count++;
                }
            } else {
                $unitParcelXml = $this->constructUnitParcelElement($dom, $unitParcel, $uriMap, $count);
                $subBlockXml->appendChild($unitParcelXml);
                $count++;
            }
        }
        foreach ($subBlock->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $subBlockXml->appendChild($outExperimentationZoneXml);
            $count++;
        }
        foreach ($subBlock->getAnnotations() as $annotation) {
            $annotationXml = $this->constructAnnotationElement($dom, $annotation, $uriMap);
            $annotationXml->setAttribute("objetMetier", $uriMap[$subBlock->getUri()]);
            $this->objectAnnotations[] = $annotationXml;
        }

        foreach ($subBlock->getNotes() as $note) {
            /** @var EntryNote $note */
            $noteXml = $this->constructNoteElement($dom, $note);
            $subBlockXml->appendChild($noteXml);
        }

        return $subBlockXml;
    }

    /**
     * @param DOMDocument $dom
     * @param UnitParcel $unitParcel
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructUnitParcelElement(DOMDocument $dom, UnitParcel $unitParcel, array &$uriMap, int $count): DOMElement
    {
        $unitParcelXml = $dom->createElement("parcellesUnitaire");
        $device = $unitParcel->getSubBlock() === null ? $unitParcel->getBlock()->getDevice() : $unitParcel->getSubBlock()->getBlock()->getDevice();
        if ($device->isIndividualPU()) {
            $unitParcelXml->setAttribute("xsi:type", "adonis.modeleMetier.plateforme:PuIndividuel");
            $unitParcelXml->setAttribute("nbIndividus", $unitParcel->getIndividuals()->count());
        } else {
            $unitParcelXml->setAttribute("xsi:type", "adonis.modeleMetier.plateforme:PuSurfacique");
            if (!empty($unitParcel->getIdent())) {
                $unitParcelXml->setAttribute("idRfidCodeBarre", $unitParcel->getIdent());
            }
        }

        $unitParcelXml->setAttribute("x", $unitParcel->getX());
        $unitParcelXml->setAttribute("y", $unitParcel->getY());
        $unitParcelXml->setAttribute("numero", $unitParcel->getName());
        if ($unitParcel->getDemiseDate() !== null) {
            $unitParcelXml->setAttribute("dateDisparition", $unitParcel->getDemiseDate()->format(DateTime::RFC3339_EXTENDED));
        }
        if ($device->isIndividualPU() && $unitParcel->getApparitionDate() !== null) {
            $unitParcelXml->setAttribute("dateApparition", $unitParcel->getApparitionDate()->format(DateTime::RFC3339_EXTENDED));
        }
        //parcels without treatments are filtered by parents object
        $unitParcelXml->setAttribute("traitement", $uriMap[$unitParcel->getTreatment()->getUri()]);

        $uriMap[$unitParcel->getUri()] = $unitParcel->getSubBlock() === null ?
            $uriMap[$unitParcel->getBlock()->getUri()] . "/@parcellesUnitaire." . $count :
            $uriMap[$unitParcel->getSubBlock()->getUri()] . "/@parcellesUnitaire." . $count;

        if ($unitParcel->getStateCode() != null) {
            $fakeTab = [];
            $stateCodeXml = $this->constructStateCodeElement($dom, $unitParcel->getStateCode(), $fakeTab, 0, false);
            $unitParcelXml->appendChild($stateCodeXml);
            $uriMap[$unitParcel->getUri() . $unitParcel->getStateCode()->getUri()] = $uriMap[$unitParcel->getUri()] . "/@codeEtat";
        }
        $count = 0;
        foreach ($unitParcel->getIndividuals() as $individual) {
            /** @var Individual $individual */
            $individualXml = $this->constructIndividualElement($dom, $individual, $uriMap, $count);
            $unitParcelXml->appendChild($individualXml);
            $count++;
        }
        foreach ($unitParcel->getOutExperimentationZones() as $outExperimentationZone) {
            /** @var OutExperimentationZone $outExperimentationZone */
            $outExperimentationZoneXml = $this->constructOutExperimentationZoneElement($dom, $outExperimentationZone, $uriMap);
            $unitParcelXml->appendChild($outExperimentationZoneXml);
            $count++;
        }
        foreach ($unitParcel->getAnnotations() as $annotation) {
            $annotationXml = $this->constructAnnotationElement($dom, $annotation, $uriMap);
            $annotationXml->setAttribute("objetMetier", $uriMap[$unitParcel->getUri()]);
            $this->objectAnnotations[] = $annotationXml;
        }
        if ($unitParcel->getAnomaly() !== null) {
            $this->anomalies[] = $unitParcel->getAnomaly();
        }
        foreach ($unitParcel->getNotes() as $note) {
            /** @var EntryNote $note */
            $noteXml = $this->constructNoteElement($dom, $note);
            $unitParcelXml->appendChild($noteXml);
        }

        return $unitParcelXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Individual $individual
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructIndividualElement(DOMDocument $dom, Individual $individual, array &$uriMap, int $count): DOMElement
    {
        $uriMap[$individual->getUri()] = $uriMap[$individual->getUnitParcel()->getUri()] . "/@individus." . $count;
        $individualXml = $dom->createElement("individus");
        $individualXml->setAttribute("x", $individual->getX());
        $individualXml->setAttribute("y", $individual->getY());
        $individualXml->setAttribute("numero", $individual->getName());
        if ($individual->getDemiseDate() !== null) {
            $individualXml->setAttribute("dateDisparition", $individual->getDemiseDate()->format(DateTime::RFC3339_EXTENDED));
        }
        $individualXml->setAttribute("dateApparition", $individual->getApparitionDate()->format(DateTime::RFC3339_EXTENDED));
        if (!empty($individual->getIdent())) {
            $individualXml->setAttribute("idRfidCodeBarre", $individual->getIdent());
        }
        if ($individual->getStateCode() != null) {
            $fakeTab = [];
            $stateCodeXml = $this->constructStateCodeElement($dom, $individual->getStateCode(), $fakeTab, 0, false);
            $individualXml->appendChild($stateCodeXml);
            $uriMap[$individual->getUri() . $individual->getStateCode()->getUri()] = $uriMap[$individual->getUri()] . "/@codeEtat";
        }
        foreach ($individual->getAnnotations() as $annotation) {
            $annotationXml = $this->constructAnnotationElement($dom, $annotation, $uriMap);
            $annotationXml->setAttribute("objetMetier", $uriMap[$individual->getUri()]);
            $this->objectAnnotations[] = $annotationXml;
        }
        if ($individual->getAnomaly() !== null) {
            $this->anomalies[] = $individual->getAnomaly();
        }
        foreach ($individual->getNotes() as $note) {
            /** @var EntryNote $note */
            $noteXml = $this->constructNoteElement($dom, $note);
            $individualXml->appendChild($noteXml);
        }

        return $individualXml;
    }

    /**
     * @param DOMDocument $dom
     * @param OutExperimentationZone $outExperimentationZone
     * @param array $uriMap
     * @return DOMElement
     * @throws ReflectionException
     */
    private function constructOutExperimentationZoneElement(DOMDocument $dom, OutExperimentationZone $outExperimentationZone, array $uriMap): DOMElement
    {
        $outExperimentationZoneXml = $dom->createElement('zheAvecEpaisseurs');
        $outExperimentationZoneXml->setAttribute('x', $outExperimentationZone->getX());
        $outExperimentationZoneXml->setAttribute('y', $outExperimentationZone->getY());
        $outExperimentationZoneXml->setAttribute('numero', $outExperimentationZone->getNumber());
        $outExperimentationZoneXml->setAttribute('natureZhe', $uriMap[$outExperimentationZone->getNatureZHE()->getUri()]);
        return $outExperimentationZoneXml;
    }

    /**
     * @param DOMDocument $dom
     * @param DesktopUser $desktopUser
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructDesktopUserElement(DOMDocument $dom, DesktopUser $desktopUser, array &$uriMap, int $count): DOMElement
    {
        $desktopUserXml = $dom->createElement("adonis.modeleMetier.utilisateur:Utilisateur");
        $desktopUserXml->setAttribute("nom", $desktopUser->getName());
        $desktopUserXml->setAttribute("prenom", $desktopUser->getFirstname());
        $desktopUserXml->setAttribute("email", $desktopUser->getEmail());
        $desktopUserXml->setAttribute("login", $desktopUser->getLogin());
        $desktopUserXml->setAttribute("motDePasse", "1f83246b9b6a1409d273de4901c98514");

        $uriMap[$desktopUser->getUri()] = "/$count";

        return $desktopUserXml;
    }

    /**
     * @param DOMDocument $dom
     * @param CombinationTest $test
     * @param $uriMap
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructCombinationTestElement(DOMDocument $dom, CombinationTest $test, $uriMap): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestCombinaisonEntreVariables");
        $testXml->setAttribute('operande1', $uriMap[$test->getFirstOperand()->getUri()]);
        $testXml->setAttribute('operande2', $uriMap[$test->getSecondOperand()->getUri()]);
        $testXml->setAttribute('nom', 'Test sur combinaison entre variables');
        $testXml->setAttribute('active', $test->isActive() ? 'true' : 'false');
        switch ($test->getOperator()) {
            case "-":
                $testXml->setAttribute('operateur', "soustraction");
                break;
            case "/":
                $testXml->setAttribute('operateur', "division");
                break;
            case "*":
                $testXml->setAttribute('operateur', "multiplication");
                break;
        }

        $limiteXml = $dom->createElement('limite');
        $limiteXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $limiteXml->setAttribute('minDouble', $test->getLowLimit());
        $limiteXml->setAttribute('maxDouble', $test->getHighLimit());

        $testXml->appendChild($limiteXml);

        return $testXml;

    }

    /**
     * @param DOMDocument $dom
     * @param GrowthTest $test
     * @param $uriMap
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructGrowthTestElement(DOMDocument $dom, GrowthTest $test, $uriMap): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestsDeComparaison");
        $testXml->setAttribute('variableDeComparaison', $uriMap[$test->getComparisonVariable()->getUri()]);
        $testXml->setAttribute('active', $test->isActive() ? 'true' : 'false');
        $testXml->setAttribute('nom', 'Test d\'accroissement');

        $ecartXml = $dom->createElement('ecart');
        $ecartXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $ecartXml->setAttribute('minDouble', $test->getMinDiff());
        $ecartXml->setAttribute('maxDouble', $test->getMaxDiff());

        $testXml->appendChild($ecartXml);

        return $testXml;
    }

    /**
     * @param DOMDocument $dom
     * @param PreconditionedCalculation $test
     * @param $uriMap
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructPreconditionedCalculationElement(DOMDocument $dom, PreconditionedCalculation $test, $uriMap): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestConditionnelEntreVariables");
        $testXml->setAttribute('active', $test->isActive() ? 'true' : 'false');
        $testXml->setAttribute('nom', 'Précalcul conditionnel');
        $testXml->setAttribute('condition', $test->getCondition());

        $testXml->setAttribute('variableAAffecter', $uriMap[$test->getVariable()->getUri()]);
        $testXml->setAttribute('variableDeComparaison', $uriMap[$test->getComparisonVariable()->getUri()]);

        $affectationValue = $dom->createElement('valeurAAffecter');
        $affectationValue->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique");
        $affectationValue->setAttribute('variable', $uriMap[$test->getVariable()->getUri()]);
        $affectationValue->setAttribute('valeur', $test->getAffectationValue());

        $testXml->appendChild($affectationValue);

        if ($test->getComparedValue() === null) {
            $testXml->setAttribute('variableAComparer', $uriMap[$test->getComparedVariable()->getUri()]);
        } else {
            $comparisonValue = $dom->createElement('valeurAComparer');
            $comparisonValue->setAttribute("xsi:type", "adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique");
            $comparisonValue->setAttribute('variable', $uriMap[$test->getComparisonVariable()->getUri()]);
            $comparisonValue->setAttribute('valeur', $test->getComparedValue());
            $testXml->appendChild($comparisonValue);
        }

        return $testXml;
    }

    /**
     * @param DOMDocument $dom
     * @param RangeTest $test
     * @param $uriMap
     * @return DOMElement
     */
    public function constructRangeTestElement(DOMDocument $dom, RangeTest $test, $uriMap): DOMElement
    {
        $testXml = $dom->createElement("tests");
        $testXml->setAttribute("xsi:type", "adonis.modeleMetier.projetDeSaisie.variables:TestSurIntervalles");
        $testXml->setAttribute('active', $test->isActive() ? 'true' : 'false');
        $testXml->setAttribute('nom', 'Test sur intervalle');

        $limitXml = $dom->createElement('limiteOptionnel');
        $limitXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $limitXml->setAttribute('minDouble', $test->getOptionalMinValue());
        $limitXml->setAttribute('maxDouble', $test->getOptionalMaxValue());

        $testXml->appendChild($limitXml);

        $limitXml = $dom->createElement('limiteObligatoire');
        $limitXml->setAttribute("xsi:type", "adonis.modeleMetier.conceptsDeBase:MinMaxDouble");
        $limitXml->setAttribute('minDouble', $test->getMandatoryMinValue());
        $limitXml->setAttribute('maxDouble', $test->getMandatoryMaxValue());

        $testXml->appendChild($limitXml);

        return $testXml;
    }

    /**
     * @param DOMDocument $dom
     * @param NatureZHE $natureZHE
     * @param array $uriMap
     * @param int $count
     * @return DOMElement
     * @throws ReflectionException
     */
    public function constructNatureZHEElement(DOMDocument $dom, NatureZHE $natureZHE, array &$uriMap, int $count): DOMElement
    {
        $natureZHEXml = $dom->createElement('adonis.modeleMetier.plateforme:NatureZhe');
        $natureZHEXml->setAttribute('couleur', $natureZHE->getColor());
        $natureZHEXml->setAttribute('nom', $natureZHE->getName());
        if ($natureZHE->getTexture() !== null) {
            $natureZHEXml->setAttribute('texture', $natureZHE->getTexture());
        }

        $uriMap[$natureZHE->getUri()] = "/$count";
        return $natureZHEXml;
    }

    /**
     * @param DOMDocument $dom
     * @param Anomaly $anomaly
     * @param array $uriMap
     * @return DOMElement
     * @throws ReflectionException
     */
    private function constructAnomalyElement(DOMDocument $dom, Anomaly $anomaly, array $uriMap): DOMElement
    {
        $anomalyXml = $dom->createElement('anomalies');
        $anomalyXml->setAttribute('objetMetier', $uriMap[$anomaly->getBusinessObject()->getUri()]);
        $anomalyXml->setAttribute('description', $anomaly->getDescription());
        if ($anomaly->getType() === Anomaly::TYPE_BAD_TREATMENT) {
            $anomalyXml->setAttribute('typeAnomalie', 'individuMauvaisTraitement');

            if ($anomaly->getConstatedTreatment() !== null) {
                $anomalyXml->setAttribute('traitementConstate', $uriMap[$anomaly->getConstatedTreatment()->getUri()]);
            }
        }

        return $anomalyXml;
    }

    /**
     * @param DOMDocument $dom
     * @param EntryNote $note
     * @return DOMElement
     * @throws ReflectionException
     */
    private function constructNoteElement(DOMDocument $dom, EntryNote $note): DOMElement
    {
        $noteXml = $dom->createElement('notes');
        $noteXml->setAttribute("dateCreation", $note->getCreationDate()->format(DateTime::RFC3339_EXTENDED));
        $noteXml->setAttribute("texte", $note->getText());
        $noteXml->setAttribute("onlineUserUri", $note->getCreator()->getUri());
        $noteXml->setAttribute("loginCreateur", $note->getCreator()->getUsername());
        $noteXml->setAttribute("supprime", $note->isDeleted() ? 'true' : 'false');
        return $noteXml;
    }

    public function constructScaleElement(DOMDocument $dom, Scale $scale, array $uriMap): DOMElement
    {
        $scaleXml = $dom->createElement('echelle');

        $scaleXml->setAttribute("nom", $scale->getName());
        $scaleXml->setAttribute("valeurMin", $scale->getMinValue());
        $scaleXml->setAttribute("valeurMax", $scale->getMaxValue());
        $scaleXml->setAttribute("exclusif", $scale->isExclusive() ? 'true' : 'false');
        /** @var Mark $mark */
        foreach ($scale->getMarks() as $mark) {
            $markXml = $dom->createElement('notations');
            $markXml->setAttribute('texte', $mark->getText());
            $markXml->setAttribute('valeur', $mark->getValue());
            if ($mark->getImage()) {
                $this->markFiles[] = [$markXml, $mark->getImage()];
                $markXml->setAttribute('image', $mark->getImage());
            }
            $scaleXml->appendChild($markXml);
        }
        return $scaleXml;
    }

    /**
     * @param UniqueVariable $uniqueVariable
     * @param DOMDocument $dom
     * @return DOMElement|false|null
     */
    private function constructValueHintListXML(UniqueVariable $uniqueVariable, DOMDocument $dom)
    {
        $valueHintListXML = null;
        $valueHintList = $uniqueVariable->getValueHintList();
        if (!is_null($valueHintList)) {
            $valueHintListXML = $dom->createElement("listevaleur");
            $valueHintListXML->setAttribute("nom", $valueHintList->getName());
            foreach ($uniqueVariable->getValueHintList()->getValueHints() as $valueHint) {
                /* @var $valueHint ValueHint */
                $valueHintXML = $dom->createElement("valeurs");
                $valueHintXML->nodeValue = $valueHint->getValue();
                $valueHintListXML->appendChild($valueHintXML);
            }
        }
        return $valueHintListXML;
    }

    public function constructValueHintListElement(DOMDocument $dom, ValueHintList $valueHintList, array $uriMap): DOMElement
    {
        $valueHintListXml = $dom->createElement('listevaleur');

        $valueHintListXml->setAttribute("nom", $valueHintList->getName());
        /** @var ValueHint $valueHint */
        foreach ($valueHintList->getValueHints() as $valueHint) {
            $valueHintXml = $dom->createElement('valeurs', $valueHint->getValue());
            $valueHintListXml->appendChild($valueHintXml);
        }
        return $valueHintListXml;
    }


    public function constructWorkpathElement(DOMDocument $dom, Workpath $workpath, Collection $desktopUsers, array $uriMap): DOMElement
    {
        $workpathXml = $dom->createElement('cheminementCalcules');

        $workpathXml->setAttribute("nom", $workpath->getUsername());

        $user = array_values(array_filter($desktopUsers->toArray(), function (DesktopUser $item) use ($workpath) {
            return $item->getLogin() === $workpath->getUsername();
        }))[0];

        $workpathXml->setAttribute("utilisateur", $uriMap[$user->getUri()]);
        /** @var ValueHint $valueHint */

        $explodedPath = explode('/', $workpath->getPath());
        array_shift($explodedPath);
        $workpathXml->setAttribute("objets", join(" ", array_map(function ($item) use ($uriMap) {
            return $uriMap["/" . $item];
        }, $explodedPath)));
        return $workpathXml;
    }

}
