<?php

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\Site;
use Shared\RightManagement\Annotation\AdvancedRight;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Webapp\Core\Validator\UniqueAttributeInParent;

/**
 * Class Factor
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER')"},
 *         "post"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER', object.getSite())",
 *              "denormalization_context"={"groups"={"factor_post"}}
 *          }
 *     },
 *     itemOperations={
 *         "get"={"security"="is_granted('ROLE_PLATFORM_MANAGER')"},
 *         "patch"={
 *              "security"="object.getProtocol() === null && is_granted('ROLE_PLATFORM_MANAGER', object.getSite())",
 *              "denormalization_context"={"groups"={"factor_post"}}
 *          },
 *         "delete"={"security"="object.getProtocol() === null && is_granted('ROLE_PLATFORM_MANAGER', object.getSite())"},
 *     }
 *
 * )
 * @ApiFilter(SearchFilter::class, properties={"protocol": "exact", "site": "exact", "name": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"design_explorer_view"}})
 *
 * @AdvancedRight(siteAttribute="site", parentFields={"protocol"})
 *
 * @Gedmo\SoftDeleteable()
 *
 * @ORM\Entity()
 * @ORM\Table(name="factor", schema="webapp")
 */
class Factor extends IdentifiedEntity
{
    use SoftDeleteableEntity;

    use OpenSilexEntity;

    /**
     * @var string
     * @Assert\NotBlank
     * @UniqueAttributeInParent(parentsAttributes={"site.factors", "protocol.factors"})
     *
     * @ORM\Column(type="string")
     * @Groups({"factor_post","design_explorer_view", "platform_full_view", "webapp_data_view", "protocol_synthesis", "protocol_full_view"})
     */
    private string $name;

    /**
     * @var Collection|array
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Modality", mappedBy="factor", cascade={"persist", "remove"}, orphanRemoval=true)
     * @Groups({"factor_post","design_explorer_view", "platform_full_view", "protocol_synthesis", "protocol_full_view"})
     */
    private $modalities;

    /**
     * @var Site|null
     *
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\Site", inversedBy="factors")
     * @Groups({"factor_post"})
     */
    private ?Site $site;

    /**
     * @var Protocol|null
     *
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Protocol", inversedBy="factors")
     */
    private ?Protocol $protocol;

    /**
     * @var int|null
     *
     * @ORM\Column(name="factor_order", type="integer", nullable=true)
     * @Groups({"factor_post","design_explorer_view", "platform_full_view", "webapp_data_view", "protocol_synthesis", "protocol_full_view", "data_view_item"})
     */
    private ?int $order;

    /**
     * @var bool
     * @ORM\Column(type="boolean")
     * @Groups({"factor_post","design_explorer_view", "protocol_full_view"})
     */
    private bool $germplasm;

    public function __construct()
    {
        $this->modalities = new ArrayCollection();
        $this->site = null;
        $this->order = null;
        $this->protocol = null;
        $this->germplasm = false;
        $this->openSilexInstance = null;
        $this->openSilexUri = null;
    }

    /**
     * @Groups({"platform_full_view"})
     * @return mixed
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Factor
     */
    public function setName(string $name): Factor
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return array<Modality>|Collection<array-key,Modality>
     */
    public function getModalities()
    {
        return $this->modalities;
    }

    /**
     * @param array<Modality>|Collection<array-key,Modality> $modalities
     * @return Factor
     */
    public function setModalities($modalities): Factor
    {
        $this->modalities = $modalities;
        foreach ($modalities as $modality){
            $modality->setFactor($this);
        }
        return $this;
    }

    /**
     * @param Modality $modality
     * @return Factor
     */
    public function addModality($modality)
    {
        if(!$this->modalities->contains($modality)){
            $this->modalities->add($modality);
            $modality->setFactor($this);
        }
        return $this;
    }

    /**
     * @param Modality $modality
     * @return Factor
     */
    public function removeModality($modality)
    {
        if($this->modalities->contains($modality)){
            $this->modalities->removeElement($modality);
        }
        return $this;
    }

    /**
     * @return Site|null
     */
    public function getSite(): ?Site
    {
        return $this->site;
    }

    /**
     * @param Site|null $site
     * @return Factor
     */
    public function setSite(?Site $site): Factor
    {
        $this->site = $site;
        return $this;
    }

    /**
     * @return Protocol|null
     */
    public function getProtocol(): ?Protocol
    {
        return $this->protocol;
    }

    /**
     * @param Protocol|null $protocol
     * @return Factor
     */
    public function setProtocol(?Protocol $protocol): Factor
    {
        $this->site = null;
        $this->protocol = $protocol;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getOrder(): ?int
    {
        return $this->order;
    }

    /**
     * @param int|null $order
     * @return Factor
     */
    public function setOrder(?int $order): Factor
    {
        $this->order = $order;
        return $this;
    }

    public function setDeletedAt(DateTime $deletedAt = null)
    {
        $this->deletedAt = $deletedAt;
        if($deletedAt === null){
            foreach ($this->getModalities() as $child){
                $child->setDeletedAt($deletedAt);
            }
        }

        return $this;
    }

    /**
     * @return bool
     */
    public function isGermplasm(): bool
    {
        return $this->germplasm;
    }

    /**
     * @param bool $germplasm
     * @return Factor
     */
    public function setGermplasm(bool $germplasm): Factor
    {
        $this->germplasm = $germplasm;
        return $this;
    }

}
