<?php


namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\ExistsFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\SoftDeleteable\Traits\SoftDeleteableEntity;
use Shared\Authentication\CustomFilters\DeletedFilter;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\User;
use Shared\RightManagement\Annotation\AdvancedRight;
use Shared\RightManagement\Traits\HasOwnerEntity;
use Shared\TransferSync\Controller\TransferToMobileAction;
use Shared\TransferSync\Entity\StatusDataEntry;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Webapp\Core\ApiOperation\RestoreObjectOperation;
use Webapp\Core\Dto\Project\UserSelectionInput;
use Webapp\Core\Validator\Project\ProjectPostConstraint;
use Webapp\Core\Validator\UniqueAttributeInParent;

/**
 * Class Project
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={},
 *         "post"={
 *              "security_post_denormalize"="is_granted('ROLE_PLATFORM_MANAGER', object.getPlatform().getSite())",
 *          },
 *     },
 *     itemOperations={
 *          "get"={},
 *          "patch"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER', object.getPlatform().getSite())",
 *          },
 *          "post_transfer"={
 *              "method"="PATCH",
 *              "path"="/projects/{id}/transfer",
 *              "controller"=TransferToMobileAction::class,
 *              "input"=UserSelectionInput::class,
 *          },
 *          "delete"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER', object.getPlatform().getSite()) && object.getProjectDatas().count() === 0"
 *          },
 *          "restore"={
 *              "controller"=RestoreObjectOperation::class,
 *              "method"="PATCH",
 *              "path"="/projects/{id}/restore",
 *              "security"="is_granted('ROLE_SITE_ADMIN')",
 *              "read"=false,
 *              "validate"=false,
 *              "openapi_context"={
 *                  "summary": "Restore deleted protocol",
 *                  "description": "Remove the deleted state"
 *              },
 *          }
 *     }
 * )
 *
 * @ApiFilter(SearchFilter::class, properties={"platform.site": "exact", "name": "exact", "pathBase": "exact"})
 * @ApiFilter(ExistsFilter::class, properties={"transferDate", "deletedAt"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={
 *     "project_explorer_view",
 *     "transfer_explorer_view",
 *     "data_explorer_view",
 *     "admin_explorer_view",
 *     "project_synthesis"
 * }})
 * @ApiFilter(DeletedFilter::class)
 *
 * @AdvancedRight(classIdentifier="webapp_project", ownerField="owner", parentFields="platform")
 *
 * @Gedmo\SoftDeleteable()
 *
 * @ORM\Entity
 * @ORM\Table(name="project", schema="webapp")
 * @ProjectPostConstraint
 */
class Project extends IdentifiedEntity
{
    use HasOwnerEntity;

    use SoftDeleteableEntity;

    /**
     * @var string
     * @ORM\Column(type="string")
     * @Groups({"project_explorer_view", "data_explorer_view", "transfer_explorer_view", "status_project_webapp_view", "admin_explorer_view", "change_report", "data_entry_synthesis", "project_synthesis", "variable_synthesis"})
     * @Assert\NotBlank
     * @UniqueAttributeInParent(parentsAttributes={"site.projects"})
     */
    private string $name;

    /**
     * @var string|null
     * @ORM\Column(type="string", nullable=true)
     */
    private ?string $comment;

    /**
     * @var Platform|null
     * @ORM\ManyToOne(targetEntity="Webapp\Core\Entity\Platform", inversedBy="projects")
     * @Groups({"data_entry_synthesis", "project_synthesis", "change_report", "variable_synthesis", "status_project_mobile_view"})
     */
    private ?Platform $platform;

    /**
     * @var Collection<array-key, Experiment>|array<Experiment>
     * @ORM\ManyToMany(targetEntity="Webapp\Core\Entity\Experiment", inversedBy="projects")
     * @ORM\JoinTable(schema="webapp", name="rel_project_experiment")
     * @Groups({"project_explorer_view", "change_report", "data_entry_synthesis", "project_synthesis", "variable_synthesis"})
     */
    private $experiments;

    /**
     * @var DateTime
     * @Gedmo\Timestampable(on="create")
     * @ORM\Column(type="datetime")
     * @Groups({"project_synthesis"})
     */
    private DateTime $created;

    /**
     * @var DateTime|null
     * @ORM\Column(type="datetime", nullable=true)
     * @Groups({"project_explorer_view", "transfer_explorer_view"})
     */
    private ?DateTime $transferDate;

    /**
     * @var Collection | array<SimpleVariable>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\SimpleVariable", mappedBy="project", cascade={"remove", "persist"}, orphanRemoval=true)
     * @Groups({"project_explorer_view"})
     */
    private $simpleVariables;

    /**
     * @var Collection | array<GeneratorVariable>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\GeneratorVariable", mappedBy="project", cascade={"remove", "persist"}, orphanRemoval=true)
     * @Groups({"project_explorer_view"})
     */
    private $generatorVariables;

    /**
     * @var Collection | array<Device>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\Device", mappedBy="project", cascade={"remove", "persist"}, orphanRemoval=true)
     * @Groups({"project_explorer_view"})
     */
    private $devices;

    /**
     * @var Collection | array<StateCode>
     *
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\StateCode", mappedBy="project", cascade={"remove", "persist"}, orphanRemoval=true)
     * @Groups({"project_explorer_view", "variable_synthesis"})
     */
    private $stateCodes;

    /**
     * @var Collection<array-key, ProjectData>|ProjectData[]
     * @Groups({"data_explorer_view", "project_explorer_view", "platform_synthesis", "project_synthesis"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\ProjectData", mappedBy="project", cascade={"remove", "persist"}, orphanRemoval=true)
     */
    private $projectDatas;

    /**
     * @var PathBase|null
     * @ORM\OneToOne(targetEntity="Webapp\Core\Entity\PathBase", mappedBy="project", cascade={"remove", "persist"}, orphanRemoval=true)
     * @Groups({"project_explorer_view", "project_synthesis", "variable_synthesis"})
     */
    private ?PathBase $pathBase;

    /**
     * @var Collection<array-key, RequiredAnnotation>|RequiredAnnotation[]
     * @Groups({"project_explorer_view", "data_entry_synthesis"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\RequiredAnnotation", mappedBy="project", cascade={"remove", "persist"}, orphanRemoval=true)
     */
    private $requiredAnnotations;

    /**
     * @var Collection<array-key, StatusDataEntry> | StatusDataEntry[]
     * @ORM\OneToMany(targetEntity="Shared\TransferSync\Entity\StatusDataEntry", mappedBy="webappProject", cascade={"remove"})
     */
    private Collection $statusDataEntries;

    /**
     * Project constructor.
     */
    public function __construct()
    {
        $this->comment = null;
        $this->owner = null;
        $this->transferDate = null;
        $this->pathBase = null;
        $this->experiments = new ArrayCollection();
        $this->simpleVariables = new ArrayCollection();
        $this->generatorVariables = new ArrayCollection();
        $this->devices = new ArrayCollection();
        $this->stateCodes = new ArrayCollection();
        $this->projectDatas = new ArrayCollection();
        $this->requiredAnnotations = new ArrayCollection();
        $this->userPaths = new ArrayCollection();
        $this->statusDataEntries = new ArrayCollection();
    }

    /**
     * @Groups({"project_synthesis"})
     * @return array
     */
    public function getVariables(): array
    {
        return array_merge(
            $this->getSimpleVariables()->toArray(),
            array_reduce($this->getDevices()->toArray(), function ($acc, Device $device) {
                return [...$acc, ...$device->getManagedVariables()];
            }, []),
            $this->getGeneratorVariables()->toArray());
    }

    /**
     * @return User | null
     * @Groups({"project_explorer_view", "status_project_webapp_view", "project_synthesis", "variable_synthesis"})
     */
    public function getOwner(): ?User
    {
        return $this->owner;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Project
     */
    public function setName(string $name): Project
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return Project
     */
    public function setComment(?string $comment): Project
    {
        $this->comment = $comment;
        return $this;
    }

    /**
     * @return Platform|null
     */
    public function getPlatform(): ?Platform
    {
        return $this->platform;
    }

    /**
     * @param Platform|null $platform
     * @return Project
     */
    public function setPlatform(?Platform $platform): Project
    {
        $this->platform = $platform;
        return $this;
    }

    /**
     * @return Collection|Experiment[]
     */
    public function getExperiments()
    {
        return $this->experiments;
    }

    /**
     * @param Collection|Experiment[] $experiments
     * @return Project
     */
    public function setExperiments($experiments): Project
    {
        $this->experiments = $experiments;
        return $this;
    }

    /**
     * @param Experiment $experiment
     * @return Project
     */
    public function addExperiment(Experiment $experiment): Project
    {
        if (!$this->experiments->contains($experiment)) {
            $this->experiments->add($experiment);
        }
        return $this;
    }

    /**
     * @param Experiment $experiment
     * @return Project
     */
    public function removeExperiment(Experiment $experiment): Project
    {
        if ($this->experiments->contains($experiment)) {
            $this->experiments->removeElement($experiment);
        }
        return $this;
    }

    /**
     * @return DateTime
     */
    public function getCreated(): DateTime
    {
        return $this->created;
    }

    /**
     * @param DateTime $created
     * @return Project
     */
    public function setCreated(DateTime $created): Project
    {
        $this->created = $created;
        return $this;
    }

    /**
     * @return DateTime|null
     */
    public function getTransferDate(): ?DateTime
    {
        return $this->transferDate;
    }

    /**
     * @param DateTime|null $transferDate
     * @return Project
     */
    public function setTransferDate(?DateTime $transferDate): Project
    {
        $this->transferDate = $transferDate;
        return $this;
    }

    /**
     * @return Collection|SimpleVariable[]
     */
    public function getSimpleVariables()
    {
        return $this->simpleVariables;
    }

    /**
     * @param Collection|SimpleVariable[] $simpleVariables
     * @return Project
     */
    public function setSimpleVariables($simpleVariables): Project
    {
        $this->simpleVariables = $simpleVariables;
        return $this;
    }

    /**
     * @param SimpleVariable $simpleVariable
     * @return Project
     */
    public function addSimpleVariable(SimpleVariable $simpleVariable): Project
    {
        if (!$this->simpleVariables->contains($simpleVariable)) {
            $this->simpleVariables->add($simpleVariable);
            $simpleVariable->setProject($this);
        }
        return $this;
    }

    /**
     * @param SimpleVariable $simpleVariable
     * @return Project
     */
    public function removeSimpleVariable(SimpleVariable $simpleVariable): Project
    {
        if ($this->simpleVariables->contains($simpleVariable)) {
            $this->simpleVariables->removeElement($simpleVariable);
            $simpleVariable->setProject(null);
        }
        return $this;
    }

    /**
     * @return Collection|GeneratorVariable[]
     */
    public function getGeneratorVariables()
    {
        return $this->generatorVariables;
    }

    /**
     * @param Collection|GeneratorVariable[] $generatorVariables
     * @return Project
     */
    public function setGeneratorVariables($generatorVariables): Project
    {
        $this->generatorVariables = $generatorVariables;
        return $this;
    }

    /**
     * @param GeneratorVariable $generatorVariable
     * @return Project
     */
    public function addGeneratorVariable(GeneratorVariable $generatorVariable): Project
    {
        if (!$this->generatorVariables->contains($generatorVariable)) {
            $this->generatorVariables->add($generatorVariable);
            $generatorVariable->setProject($this);
        }
        return $this;
    }

    /**
     * @param GeneratorVariable $generatorVariable
     * @return Project
     */
    public function removeGeneratorVariable(GeneratorVariable $generatorVariable): Project
    {
        if ($this->generatorVariables->contains($generatorVariable)) {
            $this->generatorVariables->removeElement($generatorVariable);
            $generatorVariable->setProject(null);
        }
        return $this;
    }

    /**
     * @return Collection|Device[]
     */
    public function getDevices()
    {
        return $this->devices;
    }

    /**
     * @param Collection|Device[] $devices
     * @return Project
     */
    public function setDevices($devices): Project
    {
        $this->devices = $devices;
        return $this;
    }

    /**
     * @param Device $device
     * @return Project
     */
    public function addDevice(Device $device): Project
    {
        if (!$this->devices->contains($device)) {
            $this->devices->add($device);
            $device->setProject($this);
        }
        return $this;
    }

    /**
     * @param Device $device
     * @return Project
     */
    public function removeDevice(Device $device): Project
    {
        if ($this->devices->contains($device)) {
            $this->devices->removeElement($device);
            $device->setProject(null);
        }
        return $this;
    }

    /**
     * @return Collection|StateCode[]
     */
    public function getStateCodes()
    {
        return $this->stateCodes;
    }

    /**
     * @param Collection|StateCode[] $stateCodes
     * @return Project
     */
    public function setStateCodes($stateCodes): Project
    {
        $this->stateCodes = $stateCodes;
        return $this;
    }

    /**
     * @param StateCode $stateCode
     * @return Project
     */
    public function addStateCode(StateCode $stateCode): Project
    {
        if (!$this->stateCodes->contains($stateCode)) {
            $this->stateCodes->add($stateCode);
            $stateCode->setProject($this);
        }
        return $this;
    }

    /**
     * @param StateCode $stateCode
     * @return Project
     */
    public function removeStateCode(StateCode $stateCode): Project
    {
        if ($this->stateCodes->contains($stateCode)) {
            $this->stateCodes->removeElement($stateCode);
            $stateCode->setProject(null);
        }
        return $this;
    }

    /**
     * @return Collection|ProjectData[]
     */
    public function getProjectDatas()
    {
        return $this->projectDatas;
    }

    /**
     * @param ProjectData $projectData
     * @return Project
     */
    public function addProjectData(ProjectData $projectData): Project
    {
        if (!$this->projectDatas->contains($projectData)) {
            $this->projectDatas->add($projectData);
            $projectData->setProject($this);
        }
        return $this;
    }

    /**
     * @param ProjectData $projectData
     * @return Project
     */
    public function removeProjectData(ProjectData $projectData): Project
    {
        if ($this->projectDatas->contains($projectData)) {
            $this->projectDatas->removeElement($projectData);
            $projectData->setProject(null);
        }
        return $this;
    }

    /**
     * @return PathBase|null
     *
     * @psalm-mutation-free
     */
    public function getPathBase(): ?PathBase
    {
        return $this->pathBase;
    }

    /**
     * @param PathBase|null $pathBase
     * @return Project
     */
    public function setPathBase(?PathBase $pathBase): Project
    {
        $this->pathBase = $pathBase;
        if ($pathBase !== null) {
            $pathBase->setProject($this);
        }
        return $this;
    }

    /**
     * @return Collection|RequiredAnnotation[]
     */
    public function getRequiredAnnotations()
    {
        return $this->requiredAnnotations;
    }

    /**
     * @param RequiredAnnotation $requiredAnnotation
     * @return Project
     */
    public function addRequiredAnnotation($requiredAnnotation)
    {
        if (!$this->requiredAnnotations->contains($requiredAnnotation)) {
            $this->requiredAnnotations->add($requiredAnnotation);
            $requiredAnnotation->setProject($this);
        }
        return $this;
    }

    /**
     * @param RequiredAnnotation $requiredAnnotation
     * @return Project
     */
    public function removeRequiredAnnotation($requiredAnnotation)
    {
        if ($this->requiredAnnotations->contains($requiredAnnotation)) {
            $this->requiredAnnotations->removeElement($requiredAnnotation);
            $requiredAnnotation->setProject(null);
        }
        return $this;
    }

    /**
     * @return Collection|StatusDataEntry[]
     */
    public function getStatusDataEntries()
    {
        return $this->statusDataEntries;
    }
}
