<?php
/**
 * @author      Aurélien BERNARD - TRYDEA - 2021
 */

declare(strict_types=1);

namespace Webapp\Core\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use ApiPlatform\Core\Serializer\Filter\GroupFilter;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Shared\Authentication\Entity\IdentifiedEntity;
use Shared\Authentication\Entity\User;
use Shared\RightManagement\Annotation\AdvancedRight;
use Symfony\Component\Serializer\Annotation\Groups;
use Webapp\Core\ApiOperation\DeleteProjectDataOperation;
use Webapp\Core\ApiOperation\ProjectWeightOperation;
use Webapp\Core\Dto\ProjectData\ExportOpenSilex\ProjectDataExportOpenSilex;

/**
 * Class ProjectData
 * @package Webapp\Core\Entity
 *
 * @ApiResource(
 *     collectionOperations={
 *         "get"={},
 *         "getWeight"={
 *               "controller"=ProjectWeightOperation::class,
 *               "method"="GET",
 *               "path"="/project_datas/weight",
 *               "read"=false,
 *               "validate"=false,
 *               "openapi_context"={
 *                   "summary": "Restore deleted protocol",
 *                   "description": "Remove the deleted state"
 *               },
 *          },
 *     },
 *     itemOperations={
 *         "get"={},
 *         "delete"={
 *              "security"="is_granted('ROLE_PLATFORM_MANAGER', object.getProject().getPlatform().getSite())",
 *              "controller"=DeleteProjectDataOperation::class,
 *          },
 *         "patch"={"denormalization_context"={"groups"={"edit"}}},
 *         "exportOpenSilex"={
 *               "method"="GET",
 *               "path"="/project_datas/{id}/openSilexExport",
 *               "output"=ProjectDataExportOpenSilex::class
 *         },
 *           "updateOpenSilexUri"={
 *               "method"="patch",
 *               "path"="/project_datas/{id}/opensilexUris",
 *               "security"="is_granted('ROLE_PLATFORM_MANAGER')",
 *               "input"=ProjectDataExportOpenSilex::class,
 *               "validate"=false
 *           },
 *     },
 * )
 *
 * @AdvancedRight(parentFields={"project"})
 *
 * @ApiFilter(SearchFilter::class, properties={"project.platform.projects": "exact", "project": "exact", "id": "exact", "sessions": "exact"})
 * @ApiFilter(GroupFilter::class, arguments={"whitelist"={"webapp_data_view", "id_read", "data_explorer_view", "change_report", "data_entry_synthesis", "webapp_data_path", "webapp_data_fusion"}})
 * @ORM\Entity
 * @ORM\Table(name="project_data", schema="webapp")
 *
 * @psalm-type VariableType = SimpleVariable | GeneratorVariable | SemiAutomaticVariable
 */
class ProjectData extends IdentifiedEntity
{
    use OpenSilexEntity;

    /**
     * @var Project
     * @ORM\ManyToOne(targetEntity="\Webapp\Core\Entity\Project", inversedBy="projectDatas")
     * @Groups({"webapp_data_view", "data_explorer_view", "change_report", "data_entry_synthesis", "variable_synthesis", "webapp_data_fusion"})
     */
    private Project $project;

    /**
     * @var Collection<int, Session>
     * @Groups({"webapp_data_view", "data_explorer_view", "data_entry_synthesis", "variable_synthesis", "webapp_data_path", "webapp_data_fusion"})
     * @ORM\OneToMany(targetEntity="\Webapp\Core\Entity\Session", mappedBy="projectData", cascade={"persist", "remove", "detach"})
     */
    private $sessions;

    /**
     * @var Collection<int, ChangeReport>
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\ChangeReport", mappedBy="projectData", cascade={"persist", "remove", "detach"})
     * @Groups({"change_report", "data_entry_synthesis"})
     */
    private $changeReports;

    /**
     * @var Collection<int, SimpleVariable>
     *
     * @Groups({"webapp_data_view", "data_explorer_view", "webapp_data_path", "webapp_data_fusion"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\SimpleVariable", mappedBy="projectData", cascade={"remove", "persist"}, orphanRemoval=true)
     */
    private $simpleVariables;

    /**
     * @var Collection<int, GeneratorVariable>
     *
     * @Groups({"webapp_data_view", "data_explorer_view", "webapp_data_path", "webapp_data_fusion"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\GeneratorVariable", mappedBy="projectData", cascade={"remove", "persist"}, orphanRemoval=true)
     */
    private $generatorVariables;

    /**
     * @var Collection<int, SemiAutomaticVariable>
     *
     * @Groups({"webapp_data_view", "data_explorer_view", "webapp_data_path", "webapp_data_fusion"})
     * @ORM\OneToMany(targetEntity="Webapp\Core\Entity\SemiAutomaticVariable", mappedBy="projectData", cascade={"remove", "persist"}, orphanRemoval=true)
     */
    private $semiAutomaticVariables;

    /**
     * @var bool
     *
     * @Groups({"data_explorer_view", "webapp_data_fusion", "webapp_data_path"})
     * @ORM\Column(type="boolean")
     */
    private bool $fusion;

    /**
     * @var User
     *
     * @Groups({"data_explorer_view", "data_entry_synthesis", "variable_synthesis"})
     * @ORM\ManyToOne(targetEntity="Shared\Authentication\Entity\User")
     */
    private User $user;

    /**
     * @var string|null
     * @Groups({"edit", "data_entry_synthesis", "variable_synthesis"})
     * @ORM\Column(type="string", nullable=true)
     */
    private ?string $comment;

    /**
     * @var string
     * @Groups({"data_explorer_view", "webapp_data_view", "edit", "data_entry_synthesis", "webapp_data_path", "webapp_data_fusion"})
     * @ORM\Column(type="string")
     */
    private string $name;

    /**
     * ProjectData constructor.
     */
    public function __construct()
    {
        $this->sessions = new ArrayCollection();
        $this->changeReports = new ArrayCollection();
        $this->simpleVariables = new ArrayCollection();
        $this->generatorVariables = new ArrayCollection();
        $this->semiAutomaticVariables = new ArrayCollection();
        $this->fusion = false;
        $this->comment = null;
    }

    /**
     * @Groups({"data_explorer_view", "webapp_data_view", "change_report", "data_entry_synthesis", "project_synthesis", "variable_synthesis", "webapp_data_path", "webapp_data_fusion"})
     * @return DateTime
     */
    public function getStart(): DateTime
    {
        $iterator = $this->sessions->getIterator();
        $iterator->uasort(function ($a, $b) {
            return ($a->getStartedAt() < $b->getStartedAt()) ? -1 : 1;
        });
        return $this->sessions->count() === 0 ? new DateTime() : $iterator->current()->getStartedAt();
    }

    /**
     * @Groups({"webapp_data_view", "data_entry_synthesis", "variable_synthesis", "webapp_data_fusion", "data_explorer_view"})
     * @return DateTime
     */
    public function getEnd(): DateTime
    {
        $iterator = $this->sessions->getIterator();
        $iterator->uasort(function ($a, $b) {
            return ($a->getEndedAt() > $b->getEndedAt()) ? -1 : 1;
        });
        return $this->sessions->count() === 0 ? new DateTime() : $iterator->current()->getEndedAt();
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return ProjectData
     */
    public function setName(string $name): ProjectData
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return Project
     */
    public function getProject(): Project
    {
        return $this->project;
    }

    /**
     * @param Project $project
     * @return ProjectData
     */
    public function setProject(Project $project): ProjectData
    {
        $this->project = $project;
        return $this;
    }

    /**
     * @return Collection<int, Session>
     */
    public function getSessions()
    {
        return $this->sessions;
    }

    /**
     * @param Session $session
     * @return ProjectData
     */
    public function addSession(Session $session): ProjectData
    {
        if (!$this->sessions->contains($session)) {
            $this->sessions->add($session);
            $session->setProjectData($this);
        }
        return $this;
    }

    /**
     * @param Session $session
     * @return ProjectData
     */
    public function removeSession(Session $session): ProjectData
    {
        if ($this->sessions->contains($session)) {
            $this->sessions->removeElement($session);
        }
        return $this;
    }

    /**
     * @return Collection<int, ChangeReport>
     */
    public function getChangeReports()
    {
        return $this->changeReports;
    }

    /**
     * @param Collection<int, ChangeReport> $changeReports
     * @return ProjectData
     */
    public function setChangeReports($changeReports)
    {
        $this->changeReports = $changeReports;
        return $this;
    }

    /**
     * @param ChangeReport $changeReport
     * @return ProjectData
     */
    public function addChangeReport($changeReport)
    {
        if (!$this->changeReports->contains($changeReport)) {
            $this->changeReports->add($changeReport);
            $changeReport->setProjectData($this);
        }
        return $this;
    }

    /**
     * @param ChangeReport $changeReport
     * @return ProjectData
     */
    public function removeChangeReport($changeReport)
    {
        if ($this->changeReports->contains($changeReport)) {
            $this->changeReports->removeElement($changeReport);
            $changeReport->setProjectData(null);
        }
        return $this;
    }

    /**
     * @return Collection<int, SimpleVariable>
     */
    public function getSimpleVariables()
    {
        return $this->simpleVariables;
    }

    /**
     * @param Collection<int, SimpleVariable> $simpleVariables
     * @return ProjectData
     */
    public function setSimpleVariables($simpleVariables)
    {
        $this->simpleVariables = $simpleVariables;
        return $this;
    }

    /**
     * @param SimpleVariable $simpleVariable
     * @return ProjectData
     */
    public function addSimpleVariable($simpleVariable)
    {
        if (!$this->simpleVariables->contains($simpleVariable)) {
            $this->simpleVariables->add($simpleVariable);
            $simpleVariable->setProjectData($this);
        }
        return $this;
    }

    /**
     * @param SimpleVariable $simpleVariable
     * @return ProjectData
     */
    public function removeSimpleVariable($simpleVariable)
    {
        if ($this->simpleVariables->contains($simpleVariable)) {
            $this->simpleVariables->removeElement($simpleVariable);
            $simpleVariable->setProjectData(null);
        }
        return $this;
    }

    /**
     * @return Collection<int, GeneratorVariable>
     */
    public function getGeneratorVariables()
    {
        return $this->generatorVariables;
    }

    /**
     * @param Collection|GeneratorVariable[] $generatorVariables
     * @return ProjectData
     */
    public function setGeneratorVariables($generatorVariables)
    {
        $this->generatorVariables = $generatorVariables;
        return $this;
    }

    /**
     * @param GeneratorVariable $generatorVariable
     * @return ProjectData
     */
    public function addGeneratorVariable($generatorVariable)
    {
        if (!$this->generatorVariables->contains($generatorVariable)) {
            $this->generatorVariables->add($generatorVariable);
            $generatorVariable->setProjectData($this);
        }
        return $this;
    }

    /**
     * @param GeneratorVariable $generatorVariable
     * @return ProjectData
     */
    public function removeGeneratorVariable($generatorVariable)
    {
        if ($this->generatorVariables->contains($generatorVariable)) {
            $this->generatorVariables->removeElement($generatorVariable);
            $generatorVariable->setProjectData(null);
        }
        return $this;
    }

    /**
     * @return Collection<int, SemiAutomaticVariable>
     */
    public function getSemiAutomaticVariables()
    {
        return $this->semiAutomaticVariables;
    }

    /**
     * @param Collection<int, SemiAutomaticVariable> $semiAutomaticVariables
     * @return ProjectData
     */
    public function setSemiAutomaticVariables($semiAutomaticVariables)
    {
        $this->semiAutomaticVariables = $semiAutomaticVariables;
        return $this;
    }

    /**
     * @param SemiAutomaticVariable $semiAutomaticVariable
     * @return ProjectData
     */
    public function addSemiAutomaticVariable($semiAutomaticVariable)
    {
        if (!$this->semiAutomaticVariables->contains($semiAutomaticVariable)) {
            $this->semiAutomaticVariables->add($semiAutomaticVariable);
            $semiAutomaticVariable->setProjectData($this);
        }
        return $this;
    }

    /**
     * @param SemiAutomaticVariable $semiAutomaticVariable
     * @return ProjectData
     */
    public function removeSemiAutomaticVariable($semiAutomaticVariable)
    {
        if ($this->semiAutomaticVariables->contains($semiAutomaticVariable)) {
            $this->semiAutomaticVariables->removeElement($semiAutomaticVariable);
            $semiAutomaticVariable->setProjectData(null);
        }
        return $this;
    }

    /**
     * @psalm-param VariableType $variable
     */
    public function addVariable(AbstractVariable $variable): self
    {
        if ($variable instanceof SimpleVariable) {
            $this->addSimpleVariable($variable);
        } elseif ($variable instanceof SemiAutomaticVariable) {
            $this->addSemiAutomaticVariable($variable);
        } elseif ($variable instanceof GeneratorVariable) {
            $this->addGeneratorVariable($variable);
        }
        return $this;
    }

    /**
     * @return bool
     */
    public function isFusion(): bool
    {
        return $this->fusion;
    }

    /**
     * @param bool $fusion
     * @return ProjectData
     */
    public function setFusion(bool $fusion): ProjectData
    {
        $this->fusion = $fusion;
        return $this;
    }

    /**
     * @return User
     */
    public function getUser(): User
    {
        return $this->user;
    }

    /**
     * @param User $user
     * @return ProjectData
     */
    public function setUser(User $user): ProjectData
    {
        $this->user = $user;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     * @return ProjectData
     */
    public function setComment(?string $comment): ProjectData
    {
        $this->comment = $comment;
        return $this;
    }

    /**
     * @Groups({"webapp_data_view", "webapp_data_fusion"})
     * @return string
     */
    public function getUserName(): string
    {
        return $this->user->getUsername();
    }

    /**
     * @Groups({"data_entry_synthesis", "platform_synthesis"})
     * @return (SimpleVariable | SemiAutomaticVariable | GeneratorVariable)[]
     */
    public function getVariables(): array
    {
        return array_merge($this->getSimpleVariables()->toArray(), $this->getSemiAutomaticVariables()->toArray(), $this->getGeneratorVariables()->toArray());
    }
}
