<?php

namespace Webapp\FileManagement\Service;


use ApiPlatform\Core\Api\IriConverterInterface;
use DateTime;
use Shared\Authentication\Entity\Site;
use Shared\Authentication\Entity\User;
use Shared\Authentication\Repository\UserRepository;
use SimpleXMLElement;
use Webapp\Core\Entity\AbstractVariable;
use Webapp\Core\Entity\Annotation;
use Webapp\Core\Entity\Block;
use Webapp\Core\Entity\Device;
use Webapp\Core\Entity\Experiment;
use Webapp\Core\Entity\Factor;
use Webapp\Core\Entity\FieldGeneration;
use Webapp\Core\Entity\FieldMeasure;
use Webapp\Core\Entity\GeneratorVariable;
use Webapp\Core\Entity\Individual;
use Webapp\Core\Entity\Measure;
use Webapp\Core\Entity\Modality;
use Webapp\Core\Entity\Note;
use Webapp\Core\Entity\OezNature;
use Webapp\Core\Entity\OutExperimentationZone;
use Webapp\Core\Entity\PathBase;
use Webapp\Core\Entity\PathLevelAlgorithm;
use Webapp\Core\Entity\Platform;
use Webapp\Core\Entity\Project;
use Webapp\Core\Entity\ProjectData;
use Webapp\Core\Entity\Protocol;
use Webapp\Core\Entity\RequiredAnnotation;
use Webapp\Core\Entity\SemiAutomaticVariable;
use Webapp\Core\Entity\Session;
use Webapp\Core\Entity\SimpleVariable;
use Webapp\Core\Entity\StateCode;
use Webapp\Core\Entity\SubBlock;
use Webapp\Core\Entity\SurfacicUnitPlot;
use Webapp\Core\Entity\Test;
use Webapp\Core\Entity\Treatment;
use Webapp\Core\Entity\UnitPlot;
use Webapp\Core\Entity\ValueList;
use Webapp\Core\Entity\ValueListItem;
use Webapp\Core\Entity\VariableScale;
use Webapp\Core\Entity\VariableScaleItem;
use Webapp\Core\Enumeration\AnnotationKindEnum;
use Webapp\Core\Enumeration\CombinationOperationEnum;
use Webapp\Core\Enumeration\CommunicationProtocolEnum;
use Webapp\Core\Enumeration\ComparisonOperationEnum;
use Webapp\Core\Enumeration\ExperimentStateEnum;
use Webapp\Core\Enumeration\GraphicalOriginEnum;
use Webapp\Core\Enumeration\PathLevelEnum;
use Webapp\Core\Enumeration\PossibleMoveEnum;
use Webapp\Core\Enumeration\PossibleStartPointEnum;
use Webapp\Core\Enumeration\SpreadingEnum;
use Webapp\Core\Enumeration\TestTypeEnum;
use Webapp\Core\Enumeration\VariableTypeEnum;
use Webapp\FileManagement\Exception\ParsingException;

/**
 * Class WebappParserHelper.
 */
class WebappParserHelper extends AbstractReaderHelper
{
    private IriConverterInterface $iriConverter;

    /**
     * WebappParserHelper constructor.
     * @param UserRepository $userRepository
     * @param string $projectDir
     */
    public function __construct(UserRepository $userRepository, IriConverterInterface $iriConverter, string $projectDir)
    {
        parent::__construct($userRepository, $projectDir);
        $this->iriConverter = $iriConverter;
    }

    /**
     * @param SimpleXMLElement $platformXml
     * @param array $uriMap
     * @param Site $site
     * @param User $user
     * @return Platform
     * @throws ParsingException
     */
    public function parsePlatform(SimpleXMLElement $platformXml, array &$uriMap, string $platformUri, Site $site, User $user): Platform
    {
        $this->testPlatform($platformXml, $uriMap);
        $platform = (new Platform())
            ->setSite($site)
            ->setOwner($user)
            ->setSiteName($platformXml->attributes()['nomSite'])
            ->setPlaceName($platformXml->attributes()['nomLieu'])
            ->setName($platformXml->attributes()['nom'])
            ->setCreated(date_create($platformXml->attributes()['dateCreation']));
        if (count($platformXml->children()->structureGraphique)) {
            $platform
                ->setXMesh(intval($platformXml->children()->structureGraphique->attributes()['largeurMaille']))
                ->setYMesh(intval($platformXml->children()->structureGraphique->attributes()['hauteurMaille']));
            switch ($platformXml->children()->structureGraphique->attributes()['origine']) {
                case "basGauche":
                    $platform->setOrigin(GraphicalOriginEnum::BOTTOM_LEFT);
                    break;
                case "hautGauche":
                    $platform->setOrigin(GraphicalOriginEnum::TOP_LEFT);
                    break;
                case "hautDroite":
                    $platform->setOrigin(GraphicalOriginEnum::TOP_RIGHT);
                    break;
                default:
                    $platform->setOrigin(GraphicalOriginEnum::BOTTOM_RIGHT);
                    break;
            }
        }
        $devCNT = 0;
        foreach ($platformXml->children()->dispositifs as $deviceXml) {
            $devURI = $platformUri . "/@dispositifs." . $devCNT;
            $device = $this->parseExperiment($deviceXml, $uriMap, $devURI, $site, $user);
            $platform->addExperiment($device);
            $uriMap[$devURI] = $device;
            $devCNT++;
        }
        return $platform;
    }

    /**
     * @param SimpleXMLElement $experimentXml
     * @param array $uriMap
     * @param string $experimentUri
     * @param Site $site
     * @param User $user
     * @return Experiment
     * @throws ParsingException
     */
    public function parseExperiment(SimpleXMLElement $experimentXml, array &$uriMap, string $experimentUri, Site $site, User $user): Experiment
    {
        $this->testExperiment($experimentXml, $uriMap, $experimentUri);
        $experiment = (new Experiment())
            ->setName(isset($experimentXml->attributes()['nom']) ? $experimentXml->attributes()['nom']->__toString() : '0')
            ->setValidated(isset($experimentXml->attributes()['dateValidation']) ? date_create($experimentXml->attributes()['dateValidation']) : null)
            ->setCreated(date_create($experimentXml->attributes()['dateCreation']))
            ->setSite($site)
            ->setOwner($user);
        $experiment->setIndividualUP(!(isset($experimentXml->attributes()['puSurfacique']) && $experimentXml->attributes()['puSurfacique']->__toString() === 'true'));
        switch ($experimentXml->attributes()['etat']) {
            case "valide":
                $experiment->setState(ExperimentStateEnum::VALIDATED);
                break;
            case "verrouille":
            case "nonDeverrouillable":
                $experiment->setState(ExperimentStateEnum::LOCKED);
                break;
            default:
                $experiment->setState(ExperimentStateEnum::CREATED);
                break;
        }

        $blocCNT = 0;

        $protocolUri = $experimentUri . "/@protocoles";
        $protocol = $this->parseProtocol($experimentXml->children()->protocoles, $uriMap, $protocolUri, $site, $user);
        $experiment->setProtocol($protocol);
        $uriMap[$protocolUri] = $protocol;

        foreach ($experimentXml->children()->blocs as $blocXml) {
            $blocURI = $experimentUri . "/@blocs." . $blocCNT;
            $block = $this->parseBlock($blocXml, $uriMap, $blocURI);
            $experiment->addBlocks($block);
            $uriMap[$blocURI] = $block;
            $blocCNT++;
        }
        foreach ($experimentXml->children()->zheAvecEpaisseurs as $outExperimentationZoneXml) {
            $outExperimentationZone = $this->parseOutExperimentationZone($outExperimentationZoneXml, $uriMap);
            $experiment->addOutExperimentationZone($outExperimentationZone);
        }
        foreach ($experimentXml->children()->notes as $noteXml) {
            $note = $this->parseEntryNote($noteXml);
            $experiment->addNote($note);
        }

        return $experiment;
    }

    /**
     * @param SimpleXMLElement $protocolXml
     * @param array $uriMap
     * @param string $protocolUri
     * @param Site $site
     * @param User $user
     * @return Protocol
     * @throws ParsingException
     */
    public function parseProtocol(SimpleXMLElement $protocolXml, array &$uriMap, string $protocolUri, Site $site, User $user): Protocol
    {
        $this->testProtocol($protocolXml, $uriMap, $protocolUri);
        $protocol = (new Protocol())
            ->setName($protocolXml->attributes()['nom'])
            ->setCreated(date_create($protocolXml->attributes()['dateCreation']))
            ->setAim(isset($protocolXml->attributes()['objectifs']) ? $protocolXml->attributes()['objectifs'] : '')
            ->setOwner($user)
            ->setSite($site);
        $factorCount = 0;
        foreach ($protocolXml->children()->facteurs as $factorXml) {
            $factorUri = $protocolUri . "/@facteurs." . $factorCount;
            $factor = $this->parseFactor($factorXml, $uriMap, $factorUri, $site);
            $factor->setOrder($factorCount);
            $protocol->addFactors($factor);
            $uriMap[$factorUri] = $factor;
            $factorCount++;
        }

        $treatmentCount = 0;
        foreach ($protocolXml->children()->traitements as $treatmentXml) {
            $treatmentUri = $protocolUri . "/@traitements." . $treatmentCount;
            $treatment = $this->parseTreatment($treatmentXml, $uriMap, $treatmentUri);
            $uriMap[$treatmentUri] = $treatment;
            $protocol->addTreatments($treatment);
            $treatmentCount++;
        }

        return $protocol;
    }

    /**
     * @param SimpleXMLElement $factorXml
     * @param array $uriMap
     * @param string $factorUri
     * @param Site $site
     * @return Factor
     * @throws ParsingException
     */
    public function parseFactor(SimpleXMLElement $factorXml, array &$uriMap, string $factorUri, Site $site): Factor
    {
        $this->testFactor($factorXml, $uriMap, $factorUri);
        $factor = (new Factor())
            ->setName($factorXml->attributes()['nom'])
            ->setSite($site);
        $modalityCount = 0;
        foreach ($factorXml->children()->modalites as $modalityXml) {
            $modalityUri = $factorUri . "/@modalites." . $modalityCount;
            $modality = $this->parseModality($modalityXml, $uriMap, $modalityUri);
            $factor->addModality($modality);
            $uriMap[$modalityUri] = $modality;
            $modalityCount++;
        }
        return $factor;
    }

    /**
     * @param SimpleXMLElement $modalityXml
     * @return Modality
     * @throws ParsingException
     */
    public function parseModality(SimpleXMLElement $modalityXml, array $uriMap, string $modalityUri): Modality
    {
        $this->testModality($modalityXml, $uriMap, $modalityUri);
        $modality = (new Modality())
            ->setIdentifier(isset($modalityXml->attributes()['identifiant'])
                ? $modalityXml->attributes()['identifiant']->__toString()
                : null)
            ->setShortName(isset($modalityXml->attributes()['nomCourt'])
                ? $modalityXml->attributes()['nomCourt']->__toString()
                : null)
            ->setValue($modalityXml->attributes()['valeur']);
        return $modality;
    }

    /**
     * @param SimpleXMLElement $treatmentXml
     * @param array $uriMap
     * @param string $treatmentUri
     * @return Treatment
     * @throws ParsingException
     */
    public function parseTreatment(SimpleXMLElement $treatmentXml, array $uriMap, string $treatmentUri): Treatment
    {
        $this->testTreatment($treatmentXml, $uriMap, $treatmentUri);
        $treatment = (new Treatment())
            ->setName($treatmentXml->attributes()['nom'])
            ->setShortName($treatmentXml->attributes()['nomCourt'])
            ->setRepetitions(isset($treatmentXml->attributes()['nbRepetitions'])
                ? intval($treatmentXml->attributes()['nbRepetitions'])
                : 1);
        foreach (explode(' ', trim($treatmentXml->attributes()['combinaisons'])) as $item) {
            $combinationItem = $uriMap[$item];
            $treatment->addModalities($combinationItem);
        }

        return $treatment;
    }

    /**
     * @param SimpleXMLElement $blocXml
     * @param array $uriMap
     * @param string $blocUri
     * @return Block
     * @throws ParsingException
     */
    public function parseBlock(SimpleXMLElement $blocXml, array &$uriMap, string $blocUri): Block
    {
        $this->testBlock($blocXml, $uriMap, $blocUri);
        $bloc = new Block();
        $bloc->setNumber(isset($blocXml->attributes()['numero'])
            ? $blocXml->attributes()['numero']->__toString()
            : '0');
        foreach ($blocXml->children()->zheAvecEpaisseurs as $outExperimentationZoneXml) {
            $outExperimentationZone = $this->parseOutExperimentationZone($outExperimentationZoneXml, $uriMap);
            $bloc->addOutExperimentationZone($outExperimentationZone);
        }
        if (isset($blocXml->children()->parcellesUnitaire)) {
            $this->parseBlockOrSubBlockChildren($blocXml, $uriMap, $blocUri, $bloc);
        } elseif (isset($blocXml->children()->sousBlocs)) {
            $subBlocCNT = 0;
            foreach ($blocXml->children()->sousBlocs as $subBlocXml) {
                $subBlocURI = $blocUri . "/@sousBlocs." . $subBlocCNT;
                $subBloc = $this->parseSubBloc($subBlocXml, $uriMap, $subBlocURI);
                $bloc->addSubBlocks($subBloc);
                $uriMap[$subBlocURI] = $subBloc;
                $subBlocCNT++;
            }
        }

        foreach ($blocXml->children()->notes as $noteXml) {
            $note = $this->parseEntryNote($noteXml);
            $bloc->addNote($note);
        }
        return $bloc;
    }

    /**
     * @param SimpleXMLElement $outExperimentationZoneXml
     * @param array $uriMap
     * @return OutExperimentationZone
     * @throws ParsingException
     */
    public function parseOutExperimentationZone(SimpleXMLElement $outExperimentationZoneXml, array $uriMap): OutExperimentationZone
    {
        $this->testOutExperimentationZone($outExperimentationZoneXml, $uriMap);
        return (new OutExperimentationZone())
            ->setX(isset($outExperimentationZoneXml->attributes()['x']) ? intval($outExperimentationZoneXml->attributes()['x']->__toString()) : 0)
            ->setY(isset($outExperimentationZoneXml->attributes()['y']) ? intval($outExperimentationZoneXml->attributes()['y']->__toString()) : 0)
            ->setNumber($outExperimentationZoneXml->attributes()['numero'])
            ->setNature($uriMap[$outExperimentationZoneXml->attributes()['natureZhe']->__toString()]);
    }

    /**
     * @param SimpleXMLElement $blocXml
     * @param array $uriMap
     * @param string $blocUri
     * @param Block|SubBlock $bloc
     * @return void
     * @throws ParsingException
     */
    private function parseBlockOrSubBlockChildren(SimpleXMLElement $blocXml, array &$uriMap, string $blocUri, $bloc): void
    {
        $puCNT = 0;
        foreach ($blocXml->children()->parcellesUnitaire as $puXml) {
            if (isset($puXml->attributes()['traitement'])) {
                $puUri = $blocUri . "/@parcellesUnitaire." . $puCNT;
                if (isset($puXml->attributes("xsi", true)["type"]) &&
                    $puXml->attributes("xsi", true)["type"]->__toString() === "adonis.modeleMetier.plateforme:PuSurfacique") {
                    $unitPlot = $this->parseSurfacicUnitPlot($puXml, $uriMap, $puUri);
                    $bloc->addSurfacicUnitPlots($unitPlot);
                } else {
                    $unitPlot = $this->parseUnitPlot($puXml, $uriMap, $puUri);
                    $bloc->addUnitPlots($unitPlot);
                }

                $uriMap[$puUri] = $unitPlot;
                $puCNT++;
            }
        }
    }

    /**
     * @param SimpleXMLElement $puXml
     * @param array $uriMap
     * @param string $puUri
     * @return UnitPlot
     * @throws ParsingException
     */
    public function parseUnitPlot(SimpleXMLElement $puXml, array &$uriMap, string $puUri): UnitPlot
    {
        $this->testUnitPlot($uriMap, $puUri);
        $unitParcel = (new UnitPlot())
            ->setNumber(isset($puXml->attributes()['numero']) ?
                $puXml->attributes()['numero']->__toString() :
                '0');
        if (isset($uriMap[$puXml->attributes()['traitement']->__toString()])) {
            $unitParcel->setTreatment($uriMap[$puXml->attributes()['traitement']->__toString()]);
        }

        $individuCNT = 0;
        foreach ($puXml->children()->individus as $individualXml) {
            $individualURI = $puUri . "/@individus." . $individuCNT;
            $individual = $this->parseIndividual($individualXml, $uriMap, $individualURI);
            $individuCNT++;
            $uriMap[$individualURI] = $individual;
            $unitParcel->addIndividual($individual);
        }
        foreach ($puXml->children()->zheAvecEpaisseurs as $outExperimentationZoneXml) {
            $outExperimentationZone = $this->parseOutExperimentationZone($outExperimentationZoneXml, $uriMap);
            $unitParcel->addOutExperimentationZone($outExperimentationZone);
        }
        foreach ($puXml->children()->notes as $noteXml) {
            $note = $this->parseEntryNote($noteXml);
            $unitParcel->addNote($note);
        }
        return $unitParcel;
    }

    /**
     * @param SimpleXMLElement $puXml
     * @param array $uriMap
     * @param string $puUri
     * @return SurfacicUnitPlot
     * @throws ParsingException
     */
    public function parseSurfacicUnitPlot(SimpleXMLElement $puXml, array &$uriMap, string $puUri): SurfacicUnitPlot
    {
        $this->testUnitPlot($uriMap, $puUri);
        $unitPlot = (new SurfacicUnitPlot())
            ->setNumber(isset($puXml->attributes()['numero']) ?
                $puXml->attributes()['numero']->__toString() :
                '0')
            ->setDead(isset($puXml->attributes()["mort"]) && $puXml->attributes()["mort"]->__toString() === "true")
            ->setDisappeared(isset($puXml->attributes()["dateDisparition"]) ? date_create($puXml->attributes()['dateDisparition']) : null)
            ->setAppeared(date_create($puXml->attributes()['dateApparition']))
            ->setX(isset($puXml->attributes()['x']) ? intval($puXml->attributes()['x']->__toString()) : 0)
            ->setY(isset($puXml->attributes()['y']) ? intval($puXml->attributes()['y']->__toString()) : 0);
        if ($unitPlot->isDead()) {
            $unitPlot->setDisappeared(new DateTime());
        }
        if (isset($puXml->attributes()['idRfidCodeBarre'])) {
            $unitPlot->setIdentifier($puXml->attributes()['idRfidCodeBarre']);
        }
        if (isset($uriMap[$puXml->attributes()['traitement']->__toString()])) {
            $unitPlot->setTreatment($uriMap[$puXml->attributes()['traitement']->__toString()]);
        }

        foreach ($puXml->children()->notes as $noteXml) {
            $note = $this->parseEntryNote($noteXml);
            $unitPlot->addNote($note);
        }
        return $unitPlot;
    }

    /**
     * @param SimpleXMLElement $individualXml
     * @param array $uriMap
     * @param string $individualUri
     * @return Individual
     * @throws ParsingException
     */
    public function parseIndividual(SimpleXMLElement $individualXml, array $uriMap, string $individualUri): Individual
    {
        $this->testIndividual($uriMap, $individualUri);
        $individual = (new Individual())
            ->setX(intval($individualXml->attributes()['x']))
            ->setY(intval($individualXml->attributes()['y']))
            ->setNumber(isset($individualXml->attributes()['numero'])
                ? $individualXml->attributes()['numero']->__toString()
                : '0')
            ->setDead(isset($individualXml->attributes()["mort"]) && $individualXml->attributes()["mort"]->__toString() === "true")
            ->setDisappeared(isset($individualXml->attributes()["dateDisparition"]) ? date_create($individualXml->attributes()['dateDisparition']) : null)
            ->setAppeared(date_create($individualXml->attributes()['dateApparition']));
        if ($individual->isDead()) {
            $individual->setDisappeared(new DateTime());
        }
        if (isset($individualXml->attributes()['idRfidCodeBarre'])) {
            $individual->setIdentifier($individualXml->attributes()['idRfidCodeBarre']);
        }

        foreach ($individualXml->children()->notes as $noteXml) {
            $note = $this->parseEntryNote($noteXml);
            $individual->addNote($note);
        }
        return $individual;
    }

    /**
     * @param SimpleXMLElement $stateCodeXml
     *
     * @return StateCode
     *
     * @throws ParsingException
     */
    public function parseStateCode(SimpleXMLElement $stateCodeXml): StateCode
    {
        $this->testStateCode($stateCodeXml);
        $stateCode = (new StateCode())
            ->setCode(intval($stateCodeXml->attributes()['code']))
            ->setTitle($stateCodeXml->attributes()['intitule']);
        switch ($stateCode->getTitle()) {
            case "Mort":
            case "Donnée Manquante":
                $stateCode->setPermanent(true);
                break;
            default:
                $stateCode->setPermanent(false);

        }
        if (isset($stateCodeXml->attributes()['signification'])) {
            $stateCode->setMeaning($stateCodeXml->attributes()['signification']);
        }
        if (isset($stateCodeXml->attributes()['propagation'])) {
            switch ($stateCodeXml->attributes()['propagation']) {
                case "individu":
                    $stateCode->setSpreading(SpreadingEnum::INDIVIDUAL);
                    break;
                case "parcelle":
                    $stateCode->setSpreading(SpreadingEnum::UNIT_PLOT);
                    break;
                default:
                    $stateCode->setSpreading(null);
                    break;
            }
        }
        if (isset($stateCodeXml->attributes()['couleur'])) {
            $stateCode->setColor($this->transformColor($stateCodeXml->attributes()['couleur']->__toString()));
        }
        return $stateCode;
    }

    /**
     * @param SimpleXMLElement $subBlocXml
     * @param array $uriMap
     * @param string $subBlocURI
     * @return SubBlock
     * @throws ParsingException
     */
    public function parseSubBloc(SimpleXMLElement $subBlocXml, array &$uriMap, string $subBlocURI): SubBlock
    {
        $this->testSubBlock($uriMap, $subBlocURI);
        $subBloc = new SubBlock();
        $subBloc->setNumber(isset($subBlocXml->attributes()['numero'])
            ? $subBlocXml->attributes()['numero']->__toString()
            : '0');
        foreach ($subBlocXml->children()->zheAvecEpaisseurs as $outExperimentationZoneXml) {
            $outExperimentationZone = $this->parseOutExperimentationZone($outExperimentationZoneXml, $uriMap);
            $subBloc->addOutExperimentationZone($outExperimentationZone);
        }
        $this->parseBlockOrSubBlockChildren($subBlocXml, $uriMap, $subBlocURI, $subBloc);

        foreach ($subBlocXml->children()->notes as $noteXml) {
            $note = $this->parseEntryNote($noteXml);
            $subBloc->addNote($note);
        }
        return $subBloc;
    }

    /**
     * @param SimpleXMLElement $natureZHEXml
     * @param Site $site
     * @return OezNature
     */
    public function readNatureZheItem(SimpleXMLElement $natureZHEXml, Site $site): OezNature
    {
        $existingOEZ = array_values(array_filter(
            $site->getOezNatures()->toArray(),
            fn(OezNature $existing) => $existing->getNature() === $natureZHEXml->attributes()["nom"]->__toString()
        ));
        return $existingOEZ[0] ?? (new OezNature())
            ->setSite($site)
            ->setNature($natureZHEXml->attributes()["nom"]->__toString())
            ->setColor($this->transformColor($natureZHEXml->attributes()["couleur"]))
            ->setTexture($this->transformTexture($natureZHEXml->attributes()["texture"]));
    }

    /**
     * @param SimpleXMLElement $noteXml
     * @return Note
     * @throws ParsingException
     */
    private function parseEntryNote(SimpleXMLElement $noteXml): Note
    {
        $this->testEntryNote($noteXml);
        $creator = $this->userRepository->findOneBy(["username" => $noteXml->attributes()["loginCreateur"]->__toString()]);
        return (new Note())
            ->setCreationDate(date_create($noteXml->attributes()["dateCreation"]))
            ->setDeleted(isset($noteXml->attributes()["supprime"]) && $noteXml->attributes()["supprime"]->__toString() === "true")
            ->setText($noteXml->attributes()["texte"]->__toString())
            ->setCreator($creator);
    }

    // -----------------PROJECT PART-----------------

    /**
     * @param SimpleXMLElement $projectXml
     * @param $uriMap
     * @return Project
     * @throws ParsingException
     */
    public function parseDataEntryProject(SimpleXMLElement $projectXml, &$uriMap, string $projectUri, User $user): Project
    {
        $this->testDataEntryProject($projectXml);
        $this->testProject($projectXml);
        $dataEntryProject = new Project();
        $dataEntryProject->setName($projectXml->attributes()['nom'])
            ->setCreated(date_create($projectXml->attributes()['dateCreation']));
        $dataEntryProject->setOwner($user);
        foreach (explode(' ', $projectXml->attributes()['dispositifs']) as $item) {
            if (!empty($item)) {
                $dataEntryProject->addExperiment($uriMap[$item]);
            }
        }

        if (isset($projectXml->children()->codesEtats)) {
            $stateCodeCnt = 0;
            foreach ($projectXml->children()->codesEtats as $codesEtatXml) {
                $stateCodeUri = $projectUri . "/@codesEtats." . $stateCodeCnt;
                $stateCode = $this->parseStateCode($codesEtatXml);
                $dataEntryProject->addStateCode($stateCode);
                $uriMap[$stateCodeUri] = $stateCode;
                $stateCodeCnt++;
            }
        }

        if (isset($projectXml->children()->materiel)) {
            $deviceCnt = 0;
            foreach ($projectXml->children()->materiel as $materialXml) {
                $deviceUri = $projectUri . "/@materiel." . $deviceCnt;
                $device = $this->parseDevice($materialXml, $uriMap, $deviceUri);
                $dataEntryProject->addDevice($device);
                $uriMap[$deviceUri] = $device;
                $deviceCnt++;
            }
        }

        if (isset($projectXml->children()->variables)) {
            $variableCnt = 0;
            foreach ($projectXml->children()->variables as $variableXml) {
                $variableUri = $projectUri . "/@variables." . $variableCnt;
                $variable = $this->parseVariable($variableXml, $uriMap, $variableUri);
                if ($variable instanceof SimpleVariable) {
                    $dataEntryProject->addSimpleVariable($variable);
                } else if ($variable instanceof GeneratorVariable) {
                    $dataEntryProject->addGeneratorVariable($variable);
                }
                $uriMap[$variableUri] = $variable;
                $variableCnt++;
            }

            // Test references variables, all variables must be handled before parsing tests
            $variableCnt = 0;
            foreach ($projectXml->children()->variables as $variableXml) {
                $variableUri = $projectUri . "/@variables." . $variableCnt;
                $variable = $uriMap[$variableUri];
                if (isset($variableXml->children()->tests)) {
                    foreach ($variableXml->children()->tests as $testXml) {
                        $test = $this->parseTest($testXml, $uriMap, $variableUri);
                        $variable->addTest($test);
                    }
                }
                $variableCnt++;
            }
        }

        foreach ($projectXml->children()->metadonneesASaisir as $mandatoryAnnotationXml) {
            $requiredAnnotation = $this->parseRequiredAnnotation($mandatoryAnnotationXml, $uriMap);
            $dataEntryProject->addRequiredAnnotation($requiredAnnotation);
        }

        if (isset($projectXml->children()->sectionCheminements)) {
            $path = $this->parsePath($projectXml->children()->sectionCheminements, $uriMap);
            $dataEntryProject->setPathBase($path);
        }

        return $dataEntryProject;
    }

    /**
     * @param SimpleXMLElement $deviceXml
     * @return Device
     */
    public function parseDevice(SimpleXMLElement $deviceXml, array $uriMap, string $deviceUri): Device
    {
        $this->testDevice($deviceXml, $uriMap, $deviceUri);
        $device = new Device();
        $device->setName($deviceXml->attributes()["nom"])
            ->setType(isset($deviceXml->attributes()["type"]) && $deviceXml->attributes()["type"]->__toString() === '');
        if (isset($deviceXml->attributes()["fabriquant"])) {
            $device->setManufacturer($deviceXml->attributes()["fabriquant"]);
        }
        if (isset($deviceXml->attributes()["appareil"])) {
            $device->setAlias($deviceXml->attributes()["appareil"]);
        }
        $this->parseDriver($deviceXml->children()->driver, $device);
        return $device;
    }

    /**
     * @param SimpleXMLElement $driverXml
     */
    private function parseDriver(SimpleXMLElement $driverXml, Device $device)
    {
        if (isset($driverXml->attributes("xsi", true)["type"]) && $driverXml->attributes("xsi", true)["type"]->__toString() === "adonis.modeleMetier.projetDeSaisie.variables:DriverCom") {
            $device->setCommunicationProtocol(CommunicationProtocolEnum::RS232)
                ->setBaudrate(intval($driverXml->attributes()["baudrate"]))
                ->setStopbit(floatval($driverXml->attributes()["stopbit"]))
                ->setParity($driverXml->attributes()["parity"]);
            if (isset($driverXml->attributes()["flowcontrol"])) {
                $device->setFlowControl($driverXml->attributes()["flowcontrol"]);
            } else {
                $device->setFlowControl("none");
            }
            if (isset($driverXml->attributes()["push"])) {
                $device->setRemoteControl($driverXml->attributes()["push"]->__toString() === "true");
            } else {
                $device->setRemoteControl(false);
            }

            if (isset($driverXml->attributes()["request"])) {
                // TODO
            }
            if (isset($driverXml->attributes()["databitsFormat"])) {
                $device->setBitFormat(intval($driverXml->attributes()["databitsFormat"]));
            }
            if (isset($driverXml->attributes()["port"])) {
                // TODO
            }


        } else if ($driverXml->attributes()["type"]->__toString() === 'bluetooth') {
            $device->setCommunicationProtocol(CommunicationProtocolEnum::BLUETOOTH);
        } else if ($driverXml->attributes()["type"]->__toString() === 'usb') {
            $device->setCommunicationProtocol(CommunicationProtocolEnum::USB);
        }
        $device->setFrameLength(isset($driverXml->attributes()['tailleTrame']) ? intval($driverXml->attributes()['tailleTrame']->__toString()) : 0);
        if (isset($driverXml->attributes()["debutTrame"])) {
            $device->setFrameStart($driverXml->attributes()["debutTrame"]);
        }
        if (isset($driverXml->attributes()["finTrame"])) {
            $device->setFrameEnd($driverXml->attributes()["finTrame"]);
        }
        if (isset($driverXml->attributes()["separateurCsv"])) {
            $device->setFrameCsv($driverXml->attributes()["separateurCsv"]);
        }
    }

    /**
     * @param SimpleXMLElement $variableXml
     * @param array $uriMap
     * @param string $variableUri
     * @return AbstractVariable
     * @throws ParsingException
     */
    public function parseVariable(SimpleXMLElement $variableXml, array &$uriMap, string $variableUri): AbstractVariable
    {
        $this->testVariable($variableXml, $uriMap, $variableUri);
        if (isset($variableXml->attributes()['nomGenere'])) {
            $variable = (new GeneratorVariable("", "", 0, "", false, "", false, false, false))
                ->setGeneratedPrefix($variableXml->attributes()['nomGenere'])
                ->setNumeralIncrement(isset($variableXml->attributes()['extension']) ? false : true);
            $variableCNT = 0;
            foreach ($variableXml->children()->variablesGenerees as $generatedVariableXML) {
                $generatedVariableURI = $variableUri . "/@variablesGenerees." . $variableCNT;
                $generatedVariable = $this->parseVariable($generatedVariableXML, $uriMap, $generatedVariableURI);
                if ($generatedVariable instanceof SimpleVariable) {
                    $variable->addGeneratedSimpleVariable($generatedVariable);
                } else {
                    $variable->addGeneratedGeneratorVariable($generatedVariable);
                }
                $uriMap[$generatedVariableURI] = $generatedVariable;
                $variableCNT++;
            }
        } else if (isset($variableXml->attributes("xsi", true)['type']) && $variableXml->attributes("xsi", true)['type']->__toString() === "adonis.modeleMetier.projetDeSaisie.variables:VariableSemiAuto" || isset($variableXml->attributes()['materiel'])) {
            $variable = new SemiAutomaticVariable("", "", 0, "", false, "", 0, 0);
            switch (isset($variableXml->attributes()['typeVariable']) ? $variableXml->attributes()['typeVariable'] : "alphanumerique") {
                case "alphanumerique":
                    $variable->setType(VariableTypeEnum::ALPHANUMERIC);
                    break;
                case "reel":
                    $variable->setType(VariableTypeEnum::REAL);
                    break;
                case "booleen":
                    $variable->setType(VariableTypeEnum::BOOLEAN);
                    break;
                case "entiere":
                    $variable->setType(VariableTypeEnum::INTEGER);
                    break;
                case "date":
                    $variable->setType(VariableTypeEnum::DATE);
                    break;
                case "heure":
                    $variable->setType(VariableTypeEnum::HOUR);
                    break;
            }
            $variable->setStart(intval($variableXml->attributes()['debut']));
            $variable->setEnd(intval($variableXml->attributes()['fin']));
            $uriMap[$variableXml->attributes()['materiel']->__toString()]->addManagedVariable($variable);
        } else {
            $variable = new SimpleVariable("", "", 0, "", false, "");
            switch (isset($variableXml->attributes()['typeVariable']) ? $variableXml->attributes()['typeVariable'] : "alphanumerique") {
                case "alphanumerique":
                    $variable->setType(VariableTypeEnum::ALPHANUMERIC);
                    break;
                case "reel":
                    $variable->setType(VariableTypeEnum::REAL);
                    break;
                case "booleen":
                    $variable->setType(VariableTypeEnum::BOOLEAN);
                    break;
                case "entiere":
                    $variable->setType(VariableTypeEnum::INTEGER);
                    break;
                case "date":
                    $variable->setType(VariableTypeEnum::DATE);
                    break;
                case "heure":
                    $variable->setType(VariableTypeEnum::HOUR);
                    break;
            }
            if (isset($variableXml->children()->listevaleur)) {
                $valueList = $this->parseValueHintList($variableXml->children()->listevaleur);
                $variable->setValueList($valueList);
            }
        }
        $variable->setCreated(date_create($variableXml->attributes()['dateCreation']));
        $variable->setLastModified(date_create($variableXml->attributes()['dateDerniereModification']));
        $variable->setMandatory(isset($variableXml->attributes()['saisieObligatoire']) && $variableXml->attributes()['saisieObligatoire'] == "true");
        $variable->setName($variableXml->attributes()['nom']);
        $variable->setShortName($variableXml->attributes()['nomCourt']);
        $variable->setRepetitions(intval($variableXml->attributes()['nbRepetitionSaisies']->__toString()));
        $variable->setOrder(isset($variableXml->attributes()['ordre']) ? intval($variableXml->attributes()['ordre']->__toString()) : 0);
        switch ($variableXml->attributes()['uniteParcoursType'] === null ? "parcelle" : strtolower($variableXml->attributes()['uniteParcoursType']->__toString())) {
            case "parcelle":
                $variable->setPathLevel(PathLevelEnum::UNIT_PLOT);
                break;
            case "bloc":
                $variable->setPathLevel(PathLevelEnum::BLOCK);
                break;
            case "sousBloc":
                $variable->setPathLevel(PathLevelEnum::SUB_BLOCK);
                break;
            case "dispositif":
                $variable->setPathLevel(PathLevelEnum::EXPERIMENT);
                break;
            case "plateforme":
                $variable->setPathLevel(PathLevelEnum::PLATFORM);
                break;
            case "individu":
                $variable->setPathLevel(PathLevelEnum::INDIVIDUAL);
                break;
        }
        if (isset($variableXml->attributes()['commentaire'])) {
            $variable->setComment($variableXml->attributes()['commentaire']);
        }
        /*if ($variable->getType() === "reel") {
            $variable->setFormatLength(2);
        }
        if (isset($variableXml->attributes()['format']) && $variableXml->attributes()['format']->__toString() !== "") {
            $variable->setFormat($variableXml->attributes()['format']);
        }*/ // TODO gérer les formats de variable
        if (isset($variableXml->attributes()['valeurDefaut'])) {
            $variable->setDefaultTrueValue($variableXml->attributes()['valeurDefaut'] == "true");
        }
        if (isset($variableXml->attributes()['unite'])) {
            $variable->setUnit($variableXml->attributes()['unite']);
        }

        // TODO gérer les variables connectées
        /*if (isset($variableXml->children()->mesuresVariablesPrechargees)) {
            foreach ($variableXml->children()->mesuresVariablesPrechargees as $preloadedValueXml) {
                $preloadedValue = $this->parsePreviousValue($preloadedValueXml, $uriMap);
                $variable->addPreviousValue($preloadedValue);
            }
        }*/

        if (isset($variableXml->children()->echelle)) {
            $scale = $this->parseScale($variableXml->children()->echelle);
            $variable->setScale($scale);
        }

        return $variable;
    }

    /**
     * @param SimpleXMLElement $valueListXml
     * @return ValueList
     */
    private function parseValueHintList(SimpleXMLElement $valueListXml): ValueList
    {
        $this->testValueHintList($valueListXml);
        $valueHintList = new ValueList();
        $valueHintList->setName($valueListXml->attributes()['nom']);
        foreach ($valueListXml->children()->valeurs as $valueHintXml) {
            $valueHint = new ValueListItem("");
            $valueHint->setName($valueHintXml->__toString());
            $valueHintList->addValue($valueHint);
        }
        return $valueHintList;
    }

    /*
    public function parsePreviousValue(SimpleXMLElement $previousValueXml, array $uriMap): PreviousValue
    {
        $this->testPreviousValue($previousValueXml, $uriMap);
        $previousValue = new PreviousValue();
        $previousValue->setOriginVariable($uriMap[$previousValueXml->attributes()['variable']->__toString()]);
        if (isset($previousValueXml->attributes()['horodatage'])) {
            $previousValue->setDate(date_create($previousValueXml->attributes()['horodatage']));
        }
        if (
            isset($previousValueXml->attributes()['codeEtat']) &&
            isset($uriMap[(string)$previousValueXml->attributes()['codeEtat']])
        ) {
            $stateCode = $uriMap[(string)$previousValueXml->attributes()['codeEtat']];
            $previousValue->setState($this->stateCodes[$stateCode->getCode()]);
        }
        $previousValue->setValue(isset($previousValueXml->attributes()['valeur']) ? $previousValueXml->attributes()['valeur'] : null);
        $objectUri = $uriMap[$previousValueXml->attributes()['objetMetier']->__toString()]->getUri();
        $previousValue->setObjectUri($objectUri);
        return $previousValue;
    }*/

    /**
     * @param SimpleXMLElement $scaleXml
     * @return VariableScale
     * @throws ParsingException
     */
    public function parseScale(SimpleXMLElement $scaleXml): VariableScale
    {
        $this->testScale($scaleXml);
        $scale = new VariableScale();
        $scale->setName($scaleXml->attributes()['nom'])
            ->setOpen(!(isset($scaleXml->attributes()['exclusif']) && $scaleXml->attributes()['exclusif'] == "true"))
            ->setMinValue(isset($scaleXml->attributes()['valeurMin']) ? intval($scaleXml->attributes()['valeurMin']->__toString()) : 0)
            ->setMaxValue(isset($scaleXml->attributes()['valeurMax']) ? intval($scaleXml->attributes()['valeurMax']->__toString()) : 0);
        foreach ($scaleXml->children()->notations as $scaleItemXml) {
            $scaleItem = $this->parseScaleItem($scaleItemXml);
            $scale->addValue($scaleItem);
        }

        return $scale;
    }

    /**
     * @param SimpleXMLElement $scaleItemXml
     * @return VariableScaleItem
     * @throws ParsingException
     */
    public function parseScaleItem(SimpleXMLElement $scaleItemXml): VariableScaleItem
    {
        $this->testMark($scaleItemXml);
        $scaleItem = (new VariableScaleItem(0))
            ->setText($scaleItemXml->attributes()['texte'])
            ->setValue(isset($scaleItemXml->attributes()['valeur']) ? intval($scaleItemXml->attributes()['valeur']->__toString()) : 0);
        if ($scaleItemXml->attributes()['image'] != "") {
            // TODO vérifier que les images sont bien importées
            $scaleItem->setPic($this->projectDir . 'echelles/' . $scaleItemXml->attributes()['image']);
        }
        return $scaleItem;
    }

    /**
     * @param SimpleXMLElement $testXml
     * @param array $uriMap
     * @return Test
     * @throws ParsingException
     */
    private function parseTest(SimpleXMLElement $testXml, array $uriMap, string $variableUri): Test
    {
        $this->testTest($testXml, $uriMap, $variableUri);
        $test = new Test();
        switch ($testXml->attributes()['nom']->__toString()) {
            case "Test sur intervalle" :
                $test->setType(TestTypeEnum::INTERVAL)
                    ->setProbIntervalMin(
                        isset($testXml->children()->limiteOptionnel) &&
                        isset($testXml->children()->limiteOptionnel->attributes()['minDouble']) ?
                            floatval($testXml->children()->limiteOptionnel->attributes()['minDouble']->__toString()) :
                            0
                    )->setProbIntervalMax(
                        isset($testXml->children()->limiteOptionnel) &&
                        isset($testXml->children()->limiteOptionnel->attributes()['maxDouble']) ?
                            floatval($testXml->children()->limiteOptionnel->attributes()['maxDouble']->__toString()) :
                            0
                    )->setAuthIntervalMin(
                        isset($testXml->children()->limiteObligatoire) &&
                        isset($testXml->children()->limiteObligatoire->attributes()['minDouble']) ?
                            floatval($testXml->children()->limiteObligatoire->attributes()['minDouble']->__toString()) :
                            0
                    )->setAuthIntervalMax(
                        isset($testXml->children()->limiteObligatoire) &&
                        isset($testXml->children()->limiteObligatoire->attributes()['maxDouble']) ?
                            floatval($testXml->children()->limiteObligatoire->attributes()['maxDouble']->__toString()) :
                            0
                    );
                break;
            case "Test d'accroissement" :
                $test->setType(TestTypeEnum::GROWTH)
                    ->setMinGrowth(
                        isset($testXml->children()->ecart) &&
                        isset($testXml->children()->ecart->attributes()['minDouble']) ?
                            floatval($testXml->children()->ecart->attributes()['minDouble']->__toString()) :
                            0
                    )->setMaxGrowth(
                        isset($testXml->children()->ecart) &&
                        isset($testXml->children()->ecart->attributes()['maxDouble']) ?
                            floatval($testXml->children()->ecart->attributes()['maxDouble']->__toString()) :
                            0
                    )->setComparedVariable($uriMap[$testXml->attributes()['variableDeComparaison']->__toString()]);
                break;
            case "Test sur combinaison entre variables" :

                $test->setType(TestTypeEnum::COMBINATION)
                    ->setCombinedVariable1($uriMap[$testXml->attributes()['operande1']->__toString()])
                    ->setCombinedVariable2($uriMap[$testXml->attributes()['operande2']->__toString()])
                    ->setMinLimit(
                        isset($testXml->children()->limite) &&
                        isset($testXml->children()->limite->attributes()['maxDouble']) ?
                            floatval($testXml->children()->limite->attributes()['maxDouble']->__toString()) :
                            0
                    )->setMaxLimit(
                        isset($testXml->children()->limite) &&
                        isset($testXml->children()->limite->attributes()['minDouble']) ?
                            floatval($testXml->children()->limite->attributes()['minDouble']->__toString()) :
                            0
                    );
                switch ($testXml->attributes()['operateur']) {
                    case "soustraction":
                        $test->setCombinationOperation(CombinationOperationEnum::SUBSTRACTION);
                        break;
                    case "division":
                        $test->setCombinationOperation(CombinationOperationEnum::DIVISION);
                        break;
                    case "multiplication":
                        $test->setCombinationOperation(CombinationOperationEnum::MULTIPLICATION);
                        break;
                    default:
                        $test->setCombinationOperation(CombinationOperationEnum::ADIDITION);
                }
                break;
            case "Précalcul conditionnel" :
                $test->setType(TestTypeEnum::PRECONDITIONED)
                    ->setCompareWithVariable(isset($testXml->attributes()['variableAComparer']));

                $test->setComparedVariable1($uriMap[$testXml->attributes()['variableDeComparaison']->__toString()]);

                if (isset($testXml->attributes()['variableAComparer'])) {

                    $test->setComparedVariable2($uriMap[$testXml->attributes()['variableAComparer']->__toString()]);
                } else {
                    $test->setComparedValue(
                        isset($testXml->children()->valeurAComparer) &&
                        isset($testXml->children()->valeurAComparer->attributes()['valeur']) ?
                            floatval($testXml->children()->valeurAComparer->attributes()['valeur']->__toString()) :
                            0
                    );
                }
                $test->setAssignedValue(
                    isset($testXml->children()->valeurAAffecter) &&
                    isset($testXml->children()->valeurAAffecter->attributes()['valeur']) ?
                        $testXml->children()->valeurAAffecter->attributes()['valeur']->__toString() :
                        '0'
                );
                switch ($testXml->attributes()['condition']) {
                    case "=":
                        $test->setComparisonOperation(ComparisonOperationEnum::EQUAL);
                        break;
                    case "<=":
                        $test->setCombinationOperation(ComparisonOperationEnum::INF_EQUAL);
                        break;
                    case ">=":
                        $test->setCombinationOperation(ComparisonOperationEnum::SUP_EQUAL);
                        break;
                    case "<":
                        $test->setCombinationOperation(ComparisonOperationEnum::INF);
                        break;
                    case ">":
                        $test->setCombinationOperation(ComparisonOperationEnum::SUP);
                        break;
                }
                break;
        }

        return $test;
    }

    /**
     * @param SimpleXMLElement $mandatoryAnnotationXml
     * @param $uriMap
     * @return RequiredAnnotation
     * @throws ParsingException
     */
    private function parseRequiredAnnotation(SimpleXMLElement $mandatoryAnnotationXml, $uriMap): RequiredAnnotation
    {
        $this->testRequiredAnnotation($mandatoryAnnotationXml, $uriMap);
        $requiredAnnotation = (new RequiredAnnotation())
            ->setAskWhenEntering(isset($mandatoryAnnotationXml->attributes()['avantSaisie']) && $mandatoryAnnotationXml->attributes()['avantSaisie']->__toString() === 'true')
            ->setComment($mandatoryAnnotationXml->attributes()['commentaire']);

        switch ($mandatoryAnnotationXml->attributes()['nature'] === null ? "son" : $mandatoryAnnotationXml->attributes()['nature']->__toString()) {
            case "texte":
                $requiredAnnotation->setType(AnnotationKindEnum::TEXT);
                break;
            case "photo":
                $requiredAnnotation->setType(AnnotationKindEnum::PICTURE);
                break;
            case "son":
                $requiredAnnotation->setType(AnnotationKindEnum::SOUND);
                break;
        }

        switch ($mandatoryAnnotationXml->attributes()['typeUniteParcours'] === null ? "parcelle" : strtolower($mandatoryAnnotationXml->attributes()['typeUniteParcours']->__toString())) {
            case "parcelle":
                $requiredAnnotation->setLevel(PathLevelEnum::UNIT_PLOT);
                break;
            case "bloc":
                $requiredAnnotation->setLevel(PathLevelEnum::BLOCK);
                break;
            case "sousBloc":
                $requiredAnnotation->setLevel(PathLevelEnum::SUB_BLOCK);
                break;
            case "dispositif":
                $requiredAnnotation->setLevel(PathLevelEnum::EXPERIMENT);
                break;
            case "plateforme":
                $requiredAnnotation->setLevel(PathLevelEnum::PLATFORM);
                break;
            case "individu":
                $requiredAnnotation->setLevel(PathLevelEnum::INDIVIDUAL);
                break;
        }
        return $requiredAnnotation;
    }

    /**
     * @param $pathXml
     * @param $uriMap
     * @param User $user
     * @return PathBase
     *
     */
    public function parsePath($pathXml, $uriMap): PathBase
    {
        $path = (new PathBase())
            ->setName($pathXml->attributes()['name'])
            ->setAskWhenEntering(
                isset($pathXml->children()->cheminementCalcules[0]) &&
                isset($pathXml->children()->cheminementCalcules[0]->attributes()["declenchement"]) &&
                $pathXml->children()->cheminementCalcules[0]->attributes()["declenchement"]->__toString() === "sortie"
            )
            ->setPathFilterNodes([]);
        $orderedIris = [];
        foreach (explode(' ', $pathXml->children()->cheminementCalcules[0]->attributes()['objets']) as $item) {
            if (!empty($item)) {
                $orderedIris[] = $this->iriConverter->getIriFromItem($uriMap[$item]);
            }
        }
        $path->setOrderedIris($orderedIris)
            ->setSelectedIris($orderedIris);
        foreach ($pathXml->children()->cheminements as $cheminementXml) {
            $pathLevelAlgorithm = (new PathLevelAlgorithm(PossibleMoveEnum::LIBRE, PathLevelEnum::UNIT_PLOT))
                ->setOrderedIris([]);
            switch ($cheminementXml->attributes()["niveau"] === null ? "parcelle" : strtolower($cheminementXml->attributes()["niveau"])) {
                case "parcelle":
                    $pathLevelAlgorithm->setPathLevel(PathLevelEnum::UNIT_PLOT);
                    break;
                case "bloc":
                    $pathLevelAlgorithm->setPathLevel(PathLevelEnum::BLOCK);
                    break;
                case "sousBloc":
                    $pathLevelAlgorithm->setPathLevel(PathLevelEnum::SUB_BLOCK);
                    break;
                case "dispositif":
                    $pathLevelAlgorithm->setPathLevel(PathLevelEnum::EXPERIMENT);
                    break;
                case "plateforme":
                    $pathLevelAlgorithm->setPathLevel(PathLevelEnum::PLATFORM);
                    break;
                case "individu":
                    $pathLevelAlgorithm->setPathLevel(PathLevelEnum::INDIVIDUAL);
                    break;
            }
            switch ($cheminementXml->attributes()["sensDeplacement"]) {
                case "basGauche":
                    switch ($cheminementXml->attributes()["sensDepart"]) {
                        case "gaucheDroite":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::BOT_LFT_DRGT);
                            break;
                        case "basHaut":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::BOT_LFT_DTOP);
                            break;
                    }
                    break;
                case "basDroite":
                    switch ($cheminementXml->attributes()["sensDepart"]) {
                        case "droiteGauche":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::BOT_RGT_DLFT);
                            break;
                        case "basHaut":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::BOT_RGT_DTOP);
                            break;
                    }
                    break;
                case "hautGauche":
                    switch ($cheminementXml->attributes()["sensDepart"]) {
                        case "gaucheDroite":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::TOP_LFT_DRGT);
                            break;
                        case "hautBas":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::TOP_LFT_DBOT);
                            break;
                    }
                    break;
                case "hautDroite":
                    switch ($cheminementXml->attributes()["sensDepart"]) {
                        case "droiteGauche":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::TOP_RGT_DLFT);
                            break;
                        case "hautBas":
                            $pathLevelAlgorithm->setStartPoint(PossibleStartPointEnum::TOP_RGT_DBOT);
                            break;
                    }
                    break;
                default:
                    $pathLevelAlgorithm->setStartPoint(null);
                    break;
            }
            switch ($cheminementXml->attributes()["sensDeplacement"]) {
                case "allerSimple":
                    $pathLevelAlgorithm->setMove(PossibleMoveEnum::ALLER_SIMPLE);
                    break;
                case "allerRetour":
                    $pathLevelAlgorithm->setMove(PossibleMoveEnum::ALLER_RETOUR);
                    break;
                case "serpentin":
                    $pathLevelAlgorithm->setMove(PossibleMoveEnum::SERPENTIN);
                    break;
                case "demiSerpentin":
                    $pathLevelAlgorithm->setMove(PossibleMoveEnum::DEMI_SERPENTIN);
                    break;
                case "graphique":
                    $pathLevelAlgorithm->setMove(PossibleMoveEnum::GRAPHIQUE);
                    break;
                case "ordonnanceur":
                    $pathLevelAlgorithm->setMove(PossibleMoveEnum::ORDONNANCEUR);
                    break;
                default:
                    $pathLevelAlgorithm->setMove(PossibleMoveEnum::LIBRE);
                    break;
            }
            $path->addPathLevelAlgorithm(
                $pathLevelAlgorithm
            );
        }
        return $path;
    }

    /**
     * @param $dataEntryXml
     * @param $uriMap
     * @param string $dataEntryUri
     * @param User $user
     * @return ProjectData
     * @throws ParsingException
     */
    public function parseDataEntry($dataEntryXml, $uriMap, string $dataEntryUri, User $user): ProjectData
    {
        $this->testDataEntry($dataEntryXml, $uriMap);
        $dataEntry = (new ProjectData())->setUser($user);
        $sessionCnt = 0;

        if (isset($dataEntryXml->children()->materiel)) {
            $deviceCnt = 0;
            foreach ($dataEntryXml->children()->materiel as $materialXml) {
                $deviceUri = $dataEntryUri . "/@materiel." . $deviceCnt;
                $device = $this->parseDevice($materialXml, $uriMap, $deviceUri);
                $uriMap[$deviceUri] = $device;
                $deviceCnt++;
            }
        }

        if (isset($dataEntryXml->children()->variables)) {
            $variableCnt = 0;
            foreach ($dataEntryXml->children()->variables as $variableXml) {
                $variableUri = $dataEntryUri . "/@variables." . $variableCnt;
                $variable = $this->parseVariable($variableXml, $uriMap, $variableUri);
                if ($variable instanceof SimpleVariable) {
                    $dataEntry->addSimpleVariable($variable);
                } else if ($variable instanceof GeneratorVariable) {
                    $dataEntry->addGeneratorVariable($variable);
                } else if ($variable instanceof SemiAutomaticVariable) {
                    $dataEntry->addSemiAutomaticVariable($variable);
                }
                $uriMap[$variableUri] = $variable;
                $variableCnt++;
            }

            // Test references variables, all variables must be handled before parsing tests
            $variableCnt = 0;
            foreach ($dataEntryXml->children()->variables as $variableXml) {
                if (isset($variableXml->children()->tests)) {
                    $variableUri = $dataEntryUri . "/@variables." . $variableCnt;
                    $variable = $uriMap[$variableUri];
                    foreach ($variableXml->children()->tests as $testXml) {
                        $test = $this->parseTest($testXml, $uriMap, $variableUri);
                        $variable->addTest($test);
                    }
                }
                $variableCnt++;
            }
        }
        foreach ($dataEntryXml->children()->sessions as $sessionXml) {
            $sessionUri = $dataEntryUri . '/@sessions.' . $sessionCnt++;
            $dataEntry->addSession($this->parseSession($sessionXml, $uriMap, $sessionUri, $user));
        }
        return $dataEntry;
    }

    /**
     * @param $sessionXml
     * @param $uriMap
     * @param string $sessionUri
     * @param User $user
     * @return Session
     * @throws ParsingException
     */
    private function parseSession($sessionXml, $uriMap, string $sessionUri, User $user): Session
    {
        $this->testSession($sessionXml);
        $session = (new Session())
            ->setStartedAt(date_create($sessionXml->attributes()['dateDebut']))
            ->setEndedAt(date_create($sessionXml->attributes()['dateFin']));

        $sessionUsername = isset($sessionXml->attributes()["experimentateur"]) ?
            $uriMap[$sessionXml->attributes()["experimentateur"]->__toString()] :
            "";
        if ($this->userRepository->findOneBy(["username" => $sessionUsername]) === null) {
            $session->setComment("Experimentateur non trouvé, expérimentateur d'origine : " . $sessionUsername);
        }
        $session->setUser($this->userRepository->findOneBy(["username" => $sessionUsername]) ?? $user);

        /** @var array<FieldGeneration> $generatedVariables generatedMeasureUri => FieldGeneration */
        $generatedVariables = [];
        /** @var array<FieldMeasure> $generatorVariables generatorMeasureUri => generatorField */
        $generatorVariables = [];
        /** @var array<FieldGeneration> $generatorFields generatorIndex . generatorMeasureUri => FieldGeneration */
        $generatorFields = [];
        /** @var array<FieldMeasure> $repetitionMap generatorIndex . generatorMeasure . variable . businessObject => FieldMeasure */
        $repetitionMap = [];

        $measureCnt = 0;
        foreach ($sessionXml->children()->mesureVariables as $measureXml) {
            $measureUri = $sessionUri . '/@mesureVariables.' . $measureCnt;
            $measure = (new Measure())
                ->setTimestamp(date_create($sessionXml->attributes()['horodatage']))
                ->setState(isset($measureXml->attributes()['codeEtat']) ? $uriMap[$measureXml->attributes()['codeEtat']->__toString()] : null)
                ->setValue($measureXml->attributes()['valeur']);
            $uriMap[$measureUri] = $measure;
            if (isset($measureXml->attributes("xsi", true)["type"]) &&
                $measureXml->attributes("xsi", true)["type"]->__toString() === "adonis.modeleMetier.saisieTerrain:MesureVariableBooleen") {
                $measure->setValue(isset($measureXml->attributes()['valeur']) ? 'true' : 'false');
            }

            // Handle repetitions (not available on generatorVariables)
            $repetitionUri = (isset($measureXml->attributes()['indiceGeneratrice']) ? $measureXml->attributes()['indiceGeneratrice']->__toString() : '') .
                (isset($measureXml->attributes()['mesureGeneratrice']) ? $measureXml->attributes()['mesureGeneratrice']->__toString() : '') .
                $measureXml->attributes()['variable']->__toString() .
                $measureXml->attributes()['objetMetier']->__toString();
            if (!isset($repetitionMap[$repetitionUri])) {
                $fieldMeasure = (new FieldMeasure())
                    ->setVariable($uriMap[$measureXml->attributes()['variable']->__toString()])
                    ->setTarget($uriMap[$measureXml->attributes()['objetMetier']->__toString()]);

                $repetitionMap[$repetitionUri] = $fieldMeasure;

                // Generator values
                if (isset($measureXml->attributes()['mesuresGenerees'])) {
                    foreach (explode(' ', $measureXml->attributes()['mesuresGenerees']) as $measureUri) {
                        $generatorVariables[$measureUri] = $fieldMeasure;
                        if (isset($generatedVariables[$measureUri])) {
                            $fieldMeasure->addFieldGeneration($generatedVariables[$measureUri]);
                        }
                    }
                }

                // Generated values
                if (isset($measureXml->attributes()['mesureGeneratrice'])) {
                    $generatorFieldUri = $measureXml->attributes()['indiceGeneratrice']->__toString() .
                        $measureXml->attributes()['mesureGeneratrice']->__toString();
                    if (!isset($generatorFields[$generatorFieldUri])) {
                        $generatorFields[$generatorFieldUri] = (new FieldGeneration())
                            ->setIndex(intval($measureXml->attributes()['indiceGeneratrice']))
                            ->setNumeralIncrement(false) // TODO prendre les vraies valeur ou retirer ces attributs de la classe (redondance avec la variable)
                            ->setPrefix($fieldMeasure->getVariable()->getGeneratorVariable()->getGeneratedPrefix());
                    }
                    $generatorFields[$generatorFieldUri]->addChild($fieldMeasure);

                    // If the generator measure was already found before
                    if (isset($generatorVariables[$measureXml->attributes()['mesureGeneratrice']->__toString()])) {
                        $generatorVariables[$measureXml->attributes()['mesureGeneratrice']->__toString()]->addFieldGeneration($generatorFields[$generatorFieldUri]);
                    } else {
                        $generatedVariables[$measureUri] = $generatorFields[$generatorFieldUri];
                    }
                } else {
                    $session->addFieldMeasure($fieldMeasure);
                }
            }
            $measure->setRepetition($repetitionMap[$repetitionUri]->getMeasures()->count());
            $repetitionMap[$repetitionUri]->addMeasure($measure);
            $measureCnt++;
        }
        foreach ($sessionXml->children()->metadonnees as $annotationXml) {
            $annotation = $this->parseAnnotation($annotationXml, $uriMap);
            $session->addAnnotation($annotation);
        }
        return $session;
    }

    private function parseAnnotation($annotationXml, $uriMap): Annotation
    {
        $annotation = (new Annotation())
            ->setValue($annotationXml->attributes()['donnee'])
            ->setCategories([])
            ->setKeywords([])
            ->setTimestamp(date_create($annotationXml->attributes()['date']));

        if (isset($annotationXml->attributes()['mesureVariable'])) {
            $annotation->setTarget($uriMap[$annotationXml->attributes()['mesureVariable']->__toString()]);
        } else {
            $annotation->setTarget($uriMap[$annotationXml->attributes()['objetMetier']->__toString()]);
        }

        switch ($annotationXml->attributes()['nature'] === null ? "son" : $annotationXml->attributes()['nature']->__toString()) {
            case "texte":
                $annotation->setType(AnnotationKindEnum::TEXT);
                break;
            case "photo":
                $annotation->setType(AnnotationKindEnum::PICTURE);
                break;
            case "son":
                $annotation->setType(AnnotationKindEnum::SOUND);
                break;
        }

        return $annotation;
    }

    public function parseDesktopUserName($desktopUserXml): string
    {
        return $desktopUserXml->attributes()['login']->__toString();
    }

    private function transformColor(string $color): int
    {
        preg_match('/.*{([0-9]+), *([0-9]+), *([0-9]+)}/', $color, $matches);
        return intval($matches[1]) * 256 * 256 + intval($matches[2]) * 256 + intval($matches[3]);
    }

    private function transformTexture(?string $texture): ?int
    {
        return null; // TODO gérer la texture
    }

}
