<?php

/*
 * @author TRYDEA - 2024
 */

namespace Webapp\FileManagement\Service\XmlUtils\Normalizer\Mobile;

use Mobile\Measure\Entity\Test\Base\Test;
use Mobile\Measure\Entity\Test\PreconditionedCalculation;
use Mobile\Measure\Entity\Variable\Base\Variable;
use Webapp\FileManagement\Dto\Common\ReferenceDto;
use Webapp\FileManagement\Entity\RequestFile;
use Webapp\FileManagement\Exception\ParsingException;
use Webapp\FileManagement\Service\XmlUtils\Normalizer\XmlImportConfig;

/**
 * @template-extends TestNormalizer<PreconditionedCalculation, PreconditionedCalculationXmlDto>
 *
 * @psalm-type PreconditionedCalculationXmlDto = array{
 *      "@active": ?bool,
 *      "@nom": ?string,
 *      "@condition": ?string,
 *      "@variableAComparer": ?Variable,
 *      "@variableDeComparaison": ?Variable,
 *      valeurAComparer: array,
 * }
 */
class PreconditionedCalculationNormalizer extends TestNormalizer
{
    protected const ATTR_VALEUR = '@valeur';
    protected const TAG_VALEUR_A_COMPARER = 'valeurAComparer';
    protected const TAG_VALEUR_A_AFFECTER = 'valeurAAffecter';
    protected const ATTR_VARIABLE_A_COMPARER = '@variableAComparer';
    protected const ATTR_VARIABLE_DE_COMPARAISON = '@variableDeComparaison';
    protected const ATTR_CONDITION = '@condition';
    protected const ATTR_VARIABLE_A_AFFECTER = '@variableAAffecter';
    protected const ATTR_VARIABLE = '@variable';

    protected function getClass(): string
    {
        return PreconditionedCalculation::class;
    }

    protected function extractData($object, array $context): array
    {
        return array_merge(parent::extractCommonData($object, $context), [
            self::ATTR_XSI_TYPE => 'adonis.modeleMetier.projetDeSaisie.variables:TestConditionnelEntreVariables',
            self::ATTR_NOM => 'Précalcul conditionnel',
            self::ATTR_CONDITION => $object->getCondition(),
            self::ATTR_VARIABLE_A_AFFECTER => new ReferenceDto($object->getVariable()),
            self::ATTR_VARIABLE_DE_COMPARAISON => new ReferenceDto($object->getComparisonVariable()),
            self::TAG_VALEUR_A_AFFECTER => [
                self::ATTR_XSI_TYPE => 'adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique',
                self::ATTR_VARIABLE => new ReferenceDto($object->getVariable()),
                self::ATTR_VALEUR => $object->getAffectationValue(),
            ],
            self::ATTR_VARIABLE_A_COMPARER => null === $object->getComparedValue() && null !== $object->getComparedVariable() ? new ReferenceDto($object->getComparedVariable()) : null,
            self::TAG_VALEUR_A_COMPARER => null !== $object->getComparedValue() ? [
                self::ATTR_XSI_TYPE => 'adonis.modeleMetier.saisieTerrain:MesureVariableAlphanumerique',
                self::ATTR_VARIABLE => new ReferenceDto($object->getComparisonVariable()),
                self::ATTR_VALEUR => $object->getComparedValue(),
            ] : null,
        ]);
    }

    protected function getImportDataConfig(array $context): array
    {
        return array_merge(parent::getCommonImportDataConfig($context), [
            self::ATTR_CONDITION => 'string',
            self::ATTR_VARIABLE_DE_COMPARAISON => XmlImportConfig::value(Variable::class),
            self::ATTR_VARIABLE_A_COMPARER => XmlImportConfig::value(Variable::class),
            self::TAG_VALEUR_A_COMPARER => 'array',
        ]);
    }

    protected function generateImportedObject($data, array $context): PreconditionedCalculation
    {
        return new PreconditionedCalculation();
    }

    protected function validateImportData($data, array $context): void
    {
        parent::validateImportData($data, $context);

        $readerHelper = self::getReaderHelper($context);
        if (!$readerHelper->exists($data[self::ATTR_VARIABLE_DE_COMPARAISON])) {
            throw new ParsingException('', RequestFile::ERROR_UNKNOWN_TEST_LINKED_VARIABLE);
        }
        if (isset($data[self::ATTR_VARIABLE_A_COMPARER]) && !$readerHelper->exists($data[self::ATTR_VARIABLE_A_COMPARER])) {
            throw new ParsingException('', RequestFile::ERROR_UNKNOWN_TEST_LINKED_VARIABLE);
        }
    }

    protected function completeImportedObject($object, $data, ?string $format, array $context): Test
    {
        $object = parent::completeImportedObject($object, $data, $format, $context); // TODO: Change the autogenerated stub
        $object
            ->setCondition($data[self::ATTR_CONDITION] ?? '')
            ->setComparisonVariable($data[self::ATTR_VARIABLE_DE_COMPARAISON])
            ->setAffectationValue($data[self::TAG_VALEUR_A_AFFECTER][self::ATTR_VALEUR] ?? '0');

        if (null !== $data[self::ATTR_VARIABLE_A_COMPARER]) {
            $object->setComparedVariable($data[self::ATTR_VARIABLE_A_COMPARER]);
        } else {
            $object->setComparedValue((float) ($data[self::TAG_VALEUR_A_COMPARER][self::ATTR_VALEUR] ?? 0));
        }

        return $object;
    }
}
