# cPanel recipe

### Installing

Install with composer

```bash
composer require deployer/recipes --dev
```

Add to your _deploy.php_

```php
require 'recipe/cpanel.php';
```

### Description
This is a recipe that uses the [cPanel 2 API](https://documentation.cPanel.net/display/DD/Guide+to+cPanel+API+2). 

Unfortunately the [UAPI](https://documentation.cPanel.net/display/DD/Guide+to+UAPI) that is recommended does not have support for creating addon domains.
The main idea behind is for staging purposes but I guess you can use it for other interesting concepts.

The idea is, every branch possibly has its own staging domain/subdomain (staging-neat-feature.project.com) and database db_neat-feature_project so it can be tested.
This recipe can make the domain/subdomain and database creation part of the deployment process so you don't have to manually create them through an interface.


### Configuration
The example uses a .env file and Dotenv for configuration, but you can set the parameters as you wish
```
set('cpanel', [
    'host' => getenv('CPANEL_HOST'),
    'port' => getenv('CPANEL_PORT'),
    'username' => getenv('CPANEL_USERNAME'),
    'auth_type' => getenv('CPANEL_AUTH_TYPE'),
    'token' => getenv('CPANEL_TOKEN'),
    'user' => getenv('CPANEL_USER'),
    'db_user' => getenv('CPANEL_DB_USER'),
    'db_user_privileges' => getenv('CPANEL_DB_PRIVILEGES'),
    'timeout' => 500,

    'allowInStage' => ['staging', 'beta', 'alpha'],

    'create_domain_format' => '%s-%s-%s',
    'create_domain_values' => ['staging', 'master', get('application')],
    'subdomain_prefix' => substr(md5(get('application')), 0,4) . '-',
    'subdomain_suffix' => getenv('SUDOMAIN_SUFFIX'),


    'create_db_format' => '%s_%s-%s-%s',
    'create_db_values' => ['apps', 'staging','master', get('application')],

]);
```

- `cpanel` – array with configuration for cPanel
    - `username` – WHM account
    - `user` – cPanel account that you want in charge of the domain
    - `token` – WHM API token
    - `create_domain_format` – Format for name creation of domain
    - `create_domain_values` – The actual value reference for naming
    - `subdomain_prefix` – cPanel has a weird way of dealing with addons and subdomains, you cannot create 2 addons with the same subdomain, so you need to change it in some way, example uses first 4 chars of md5(app_name)
    - `subdomain_suffix` – cPanel has a weird way of dealing with addons and subdomains, so the suffix needs to be your main domain for that account for deletion purposes
    - `addondir` – addon dir is different from the deploy path because cPanel "injects" /home/user/ into the path, so tilde cannot be used
    - `allowInStage` – Define the stages that cPanel recipe actions are allowed in


#### .env file example
```
CPANEL_HOST=xxx.xxx.xxx.xxx
CPANEL_PORT=2087
CPANEL_USERNAME=root
CPANEL_TOKEN=xxxx
CPANEL_USER=xxx
CPANEL_AUTH_TYPE=hash
CPANEL_DB_USER=db_user
CPANEL_DB_PRIVILEGES="ALL PRIVILEGES"
SUDOMAIN_SUFFIX=.mymaindomain.com

```

### Tasks

- `cpanel:createaddondomain` Creates an addon domain
- `cpanel:deleteaddondomain` Removes an addon domain
- `cpanel:createdb` Creates a new database

### Usage

A complete example with configs, staging and deployment

```
<?php

namespace Deployer;
use Dotenv\Dotenv;

require 'vendor/autoload.php';
(Dotenv::create(__DIR__))->load(); // this is used just so an .env file can be used for credentials

require 'cpanel.php';


// Project name
set('application', 'myproject.com');
// Project repository
set('repository', 'git@github.com:myorg/myproject.com');





set('cpanel', [
    'host' => getenv('CPANEL_HOST'),
    'port' => getenv('CPANEL_PORT'),
    'username' => getenv('CPANEL_USERNAME'),
    'auth_type' => getenv('CPANEL_AUTH_TYPE'),
    'token' => getenv('CPANEL_TOKEN'),
    'user' => getenv('CPANEL_USER'),
    'db_user' => getenv('CPANEL_DB_USER'),
    'db_user_privileges' => getenv('CPANEL_DB_PRIVILEGES'),
    'timeout' => 500,
    'allowInStage' => ['staging', 'beta', 'alpha'],

    'create_domain_format' => '%s-%s-%s',
    'create_domain_values' => ['staging', 'master', get('application')],
    'subdomain_prefix' => substr(md5(get('application')), 0,4) . '-',
    'subdomain_suffix' => getenv('SUDOMAIN_SUFFIX'),


    'create_db_format' => '%s_%s-%s-%s',
    'create_db_values' => ['apps', 'staging','master', get('application')],

]);

host('myproject.com')
    ->stage('staging')
    ->set('cpanel_createdb', vsprintf(get('cpanel')['create_db_format'], get('cpanel')['create_db_values']))
    ->set('branch', 'dev-branch')
    ->set('deploy_path',  '~/staging/' . vsprintf(get('cpanel')['create_domain_format'], get('cpanel')['create_domain_values']))
    ->set('addondir',  'staging/' . vsprintf(get('cpanel')['create_domain_format'], get('cpanel')['create_domain_values']));
// Tasks
task('build', function () {
    run('cd {{release_path}} && build');
});

after('deploy:prepare', 'cpanel:createaddondomain');
after('deploy:prepare', 'cpanel:createdb');
```
