<?php

declare(strict_types=1);

/*
 * @author TRYDEA - 2024
 */

namespace Mobile\Device\Repository;

use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NativeQuery;
use Doctrine\ORM\Query\ResultSetMapping;
use Doctrine\Persistence\ManagerRegistry;
use Mobile\Device\Entity\Anomaly;
use Mobile\Device\Entity\Individual;
use Mobile\Device\Entity\UnitParcel;
use Mobile\Measure\Entity\Variable\StateCode;
use Mobile\Project\Entity\DataEntryProject;

/**
 * Class IndividualRepository.
 *
 * @method Individual|null find($id, $lockMode = null, $lockVersion = null)
 * @method Individual|null findOneBy(array $criteria, array $orderBy = null)
 * @method Individual[]    findAll()
 * @method Individual[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class IndividualRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Individual::class);
    }

    public function findHasDeadStateCodeByProject(DataEntryProject $project): array
    {
        $results = $this
            ->createNativeQueryForProject(
                $project,
                ['s10_.type = ?'],
                ['INNER JOIN adonis.state_code s10_ ON s10_.id = i1_.state_code_id']
            )
            ->setParameter(2, StateCode::DEAD_STATE_CODE)
            ->getResult();

        return array_map(fn (array $result): Individual => $result['i1_'], $results);
    }

    public function findByNameByProject(DataEntryProject $project, array $names): array
    {
        $results = $this
            ->createNativeQueryForProject($project, ['b2_.name IN (?)'])
            ->setParameter(2, $names)
            ->getResult();

        return array_map(fn (array $result): Individual => $result['i1_'], $results);
    }

    private function hasSublock(DataEntryProject $project): bool
    {
        $sqlQueryHasSubBlock = <<<EOQ
            SELECT COUNT(i1_.id) AS sclr_0
            FROM adonis.individual i1_
                     INNER JOIN adonis.unit_plot u3_ ON i1_.unit_parcel_id = u3_.id
                     INNER JOIN adonis.sub_block s6_ ON u3_.sub_block_id = s6_.id
                     INNER JOIN adonis.block b9_ ON s6_.block_id = b9_.id
                     INNER JOIN adonis.experiment e12_ ON b9_.device_id = e12_.id
                     INNER JOIN adonis.platform p15_ ON e12_.platform_id = p15_.id
                     INNER JOIN adonis.project p16_ ON p15_.project_id = p16_.id
            WHERE p16_.id = ?
            EOQ;

        $rsmSubBlockExists = new ResultSetMapping();
        $rsmSubBlockExists->addScalarResult('sclr_0', 'sclr');
        $result = $this->getEntityManager()
            ->createNativeQuery($sqlQueryHasSubBlock, $rsmSubBlockExists)
            ->setParameter(1, $project->getId())
            ->getSingleScalarResult();

        return $result > 0;
    }

    private function createNativeQueryForProject(DataEntryProject $project, array $where = [], array $joins = []): NativeQuery
    {
        $rsm = new ResultSetMapping();
        $rsm
            ->addEntityResult(Individual::class, 'i1_', 'i1_')
            ->addMetaResult('i1_', 'discr', 'discr')
            ->setDiscriminatorColumn('i1_', 'discr')
            ->addFieldResult('i1_', 'id', 'id')
            ->addFieldResult('i1_', 'name', 'name')
            ->addFieldResult('i1_', 'x', 'x')
            ->addFieldResult('i1_', 'y', 'y')
            ->addFieldResult('i1_', 'apparition_date', 'apparitionDate')
            ->addFieldResult('i1_', 'demise_date', 'demiseDate')
            ->addFieldResult('i1_', 'ident', 'ident')
            ->addJoinedEntityResult(Anomaly::class, 'a1_', 'i1_', 'anomaly')
            ->addFieldResult('a1_', 'anomaly_id', 'id')
            ->addJoinedEntityResult(StateCode::class, 'sc1_', 'i1_', 'stateCode')
            ->addFieldResult('sc1_', 'state_code_id', 'id')
            ->addJoinedEntityResult(UnitParcel::class, 'up1_', 'i1_', 'unitParcel')
            ->addFieldResult('up1_', 'unit_parcel_id', 'id');

        $selectClause = '
               p0_.id              AS id,
               b2_.name            AS name,
               i1_.x               AS x,
               i1_.y               AS y,
               i1_.apparition_date AS apparition_date,
               i1_.demise_date     AS demise_date,
               i1_.ident           AS ident,
               p0_.discr           AS discr,
               b2_.anomaly_id      AS anomaly_id,
               i1_.state_code_id   AS state_code_id,
               i1_.unit_parcel_id  AS unit_parcel_id
        ';
        $joinsClause = implode(' ', $joins);
        $whereClause = implode(' AND ', ['p16_.id = ?', ...$where]);
        if ($this->hasSublock($project)) {
            $sql = <<<EOQ
                SELECT $selectClause, b2_.name,
                       FROM adonis.individual i1_
                         INNER JOIN adonis.business_object b2_ ON i1_.id = b2_.id
                         INNER JOIN adonis.project_object p0_ ON i1_.id = p0_.id
                         INNER JOIN adonis.unit_plot u3_ ON i1_.unit_parcel_id = u3_.id
                         INNER JOIN adonis.sub_block s6_ ON u3_.sub_block_id = s6_.id
                         INNER JOIN adonis.block b9_ ON s6_.block_id = b9_.id
                         INNER JOIN adonis.experiment e12_ ON b9_.device_id = e12_.id
                         INNER JOIN adonis.platform p15_ ON e12_.platform_id = p15_.id
                         INNER JOIN adonis.project p16_ ON p15_.project_id = p16_.id
                         $joinsClause
                         WHERE $whereClause
                EOQ;
        } else {
            $sql = <<<EOQ
                SELECT $selectClause, b2_.name
                FROM adonis.individual i1_
                         INNER JOIN adonis.business_object b2_ ON i1_.id = b2_.id
                         INNER JOIN adonis.project_object p0_ ON i1_.id = p0_.id
                         INNER JOIN adonis.unit_plot u3_ ON i1_.unit_parcel_id = u3_.id
                         INNER JOIN adonis.block b9_ ON u3_.block_id = b9_.id
                         INNER JOIN adonis.experiment e12_ ON b9_.device_id = e12_.id
                         INNER JOIN adonis.platform p15_ ON e12_.platform_id = p15_.id
                         INNER JOIN adonis.project p16_ ON p15_.project_id = p16_.id
                         $joinsClause
                         WHERE $whereClause
                EOQ;
        }

        return $this->getEntityManager()
            ->createNativeQuery($sql, $rsm)
            ->setParameter(1, $project->getId());
    }
}
