<?php

declare(strict_types=1);

/*
 * @author TRYDEA - 2024
 */

namespace Mobile\Device\Repository;

use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NativeQuery;
use Doctrine\ORM\Query\ResultSetMapping;
use Doctrine\Persistence\ManagerRegistry;
use Mobile\Device\Entity\Anomaly;
use Mobile\Device\Entity\Block;
use Mobile\Device\Entity\SubBlock;
use Mobile\Device\Entity\Treatment;
use Mobile\Device\Entity\UnitParcel;
use Mobile\Measure\Entity\Variable\StateCode;
use Mobile\Project\Entity\DataEntryProject;

/**
 * Class UnitParcelRepository.
 *
 * @method UnitParcel|null find($id, $lockMode = null, $lockVersion = null)
 * @method UnitParcel|null findOneBy(array $criteria, array $orderBy = null)
 * @method UnitParcel[]    findAll()
 * @method UnitParcel[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class UnitParcelRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, UnitParcel::class);
    }

    public function findHasDeadStateCodeByProject(DataEntryProject $project): array
    {
        $results = $this
            ->createNativeQueryForProject(
                $project,
                ['s10_.type = ?'],
                ['INNER JOIN adonis.state_code s10_ ON s10_.id = u3_.state_code_id']
            )
            ->setParameter(2, StateCode::DEAD_STATE_CODE)
            ->getResult();

        return array_map(fn (array $result): UnitParcel => $result['u3_'], $results);
    }

    public function findByNameByProject(DataEntryProject $project, array $names): array
    {
        $results = $this
            ->createNativeQueryForProject($project, ['b2_.name IN (?)'])
            ->setParameter(2, $names)
            ->getResult();

        return array_map(fn (array $result): UnitParcel => $result['u3_'], $results);
    }

    private function hasSublock(DataEntryProject $project): bool
    {
        $sqlQueryHasSubBlock = <<<EOQ
            SELECT COUNT(u3_.id) AS sclr_0
            FROM adonis.unit_plot u3_
                     INNER JOIN adonis.sub_block s6_ ON u3_.sub_block_id = s6_.id
                     INNER JOIN adonis.block b9_ ON s6_.block_id = b9_.id
                     INNER JOIN adonis.experiment e12_ ON b9_.device_id = e12_.id
                     INNER JOIN adonis.platform p15_ ON e12_.platform_id = p15_.id
                     INNER JOIN adonis.project p16_ ON p15_.project_id = p16_.id
            WHERE p16_.id = ?
            EOQ;

        $rsmSubBlockExists = new ResultSetMapping();
        $rsmSubBlockExists->addScalarResult('sclr_0', 'sclr');
        $result = $this->getEntityManager()
            ->createNativeQuery($sqlQueryHasSubBlock, $rsmSubBlockExists)
            ->setParameter(1, $project->getId())
            ->getSingleScalarResult();

        return $result > 0;
    }

    private function createNativeQueryForProject(DataEntryProject $project, array $where = [], array $joins = []): NativeQuery
    {
        $rsm = new ResultSetMapping();
        $rsm
            ->addEntityResult(UnitParcel::class, 'u3_', 'u3_')
            ->addMetaResult('u3_', 'discr', 'discr')
            ->setDiscriminatorColumn('u3_', 'discr')
            ->addFieldResult('u3_', 'id', 'id')
            ->addFieldResult('u3_', 'name', 'name')
            ->addFieldResult('u3_', 'x', 'x')
            ->addFieldResult('u3_', 'y', 'y')
            ->addFieldResult('u3_', 'apparition_date', 'apparitionDate')
            ->addFieldResult('u3_', 'demise_date', 'demiseDate')
            ->addFieldResult('u3_', 'ident', 'ident')
            ->addJoinedEntityResult(Anomaly::class, 'a1_', 'u3_', 'anomaly')
            ->addFieldResult('a1_', 'anomaly_id', 'id')
            ->addJoinedEntityResult(StateCode::class, 'sc1_', 'u3_', 'stateCode')
            ->addFieldResult('sc1_', 'state_code_id', 'id')
            ->addJoinedEntityResult(Block::class, 'b1_', 'u3_', 'block')
            ->addFieldResult('b1_', 'block_id', 'id')
            ->addJoinedEntityResult(SubBlock::class, 'sb1_', 'u3_', 'subBlock')
            ->addFieldResult('sb1_', 'sub_block_id', 'id')
            ->addJoinedEntityResult(Treatment::class, 't1_', 'u3_', 'treatment')
            ->addFieldResult('t1_', 'treatment_id', 'id');

        $selectClause = '
               p0_.id              AS id,
               b2_.name            AS name,
               u3_.x               AS x,
               u3_.y               AS y,
               u3_.apparition_date AS apparition_date,
               u3_.demise_date     AS demise_date,
               u3_.ident           AS ident,
               p0_.discr           AS discr,
               b2_.anomaly_id      AS anomaly_id,
               u3_.state_code_id   AS state_code_id,
               u3_.block_id        AS block_id,
               u3_.sub_block_id    AS sub_block_id,
               u3_.treatment_id    AS treatment_id
        ';
        $joinsClause = implode(' ', $joins);
        $whereClause = implode(' AND ', ['p16_.id = ?', ...$where]);
        if ($this->hasSublock($project)) {
            $sql = <<<EOQ
                SELECT $selectClause, b2_.name,
                       FROM adonis.unit_plot u3_
                         INNER JOIN adonis.business_object b2_ ON u3_.id = b2_.id
                         INNER JOIN adonis.project_object p0_ ON u3_.id = p0_.id
                         INNER JOIN adonis.sub_block s6_ ON u3_.sub_block_id = s6_.id
                         INNER JOIN adonis.block b9_ ON s6_.block_id = b9_.id
                         INNER JOIN adonis.experiment e12_ ON b9_.device_id = e12_.id
                         INNER JOIN adonis.platform p15_ ON e12_.platform_id = p15_.id
                         INNER JOIN adonis.project p16_ ON p15_.project_id = p16_.id
                         $joinsClause
                         WHERE $whereClause
                EOQ;
        } else {
            $sql = <<<EOQ
                SELECT $selectClause, b2_.name
                FROM adonis.unit_plot u3_
                         INNER JOIN adonis.business_object b2_ ON u3_.id = b2_.id
                         INNER JOIN adonis.project_object p0_ ON u3_.id = p0_.id
                         INNER JOIN adonis.block b9_ ON u3_.block_id = b9_.id
                         INNER JOIN adonis.experiment e12_ ON b9_.device_id = e12_.id
                         INNER JOIN adonis.platform p15_ ON e12_.platform_id = p15_.id
                         INNER JOIN adonis.project p16_ ON p15_.project_id = p16_.id
                         $joinsClause
                         WHERE $whereClause
                EOQ;
        }

        return $this->getEntityManager()
            ->createNativeQuery($sql, $rsm)
            ->setParameter(1, $project->getId());
    }
}
