<?php

/*
 * @author TRYDEA - 2024
 */

namespace Webapp\FileManagement\Controller;

use Shared\Authentication\Entity\Site;
use Shared\TransferSync\Entity\StatusDataEntry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\HeaderUtils;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\Routing\Annotation\Route;
use Vich\UploaderBundle\Handler\DownloadHandler;
use Webapp\Core\Entity\Experiment;
use Webapp\Core\Entity\Platform;
use Webapp\FileManagement\Entity\ResponseFile;
use Webapp\FileManagement\Service\MobileWriterService;
use Webapp\FileManagement\Service\WebappWriterService;
use Webapp\FileManagement\Voters\UserLinkedFileJobVoter;

/**
 * Class DownloadFileController.
 */
class DownloadFileController extends AbstractController
{
    /**
     * @Route("/api/download/response-files/{file}", name="downloadReturnFile")
     */
    public function executeDownload(ResponseFile $file, DownloadHandler $downloadHandler): Response
    {
        $this->denyAccessUnlessGranted(UserLinkedFileJobVoter::DOWNLOAD, $file);
        $response = $downloadHandler->downloadObject($file, 'file');
        $response->headers->set('Content-Type', 'application/zip');

        return $response;
    }

    /**
     * @Route("/api/download/project/{project}", name="downloadProjectFile")
     */
    public function exportProjectFile(StatusDataEntry $project, MobileWriterService $writerService): Response
    {
        $response = new StreamedResponse(function () use ($writerService, $project) {
            $outputStream = fopen('php://output', 'w');
            $zipStream = tmpfile();
            $writerService->constructProjectZipFile($project->getRequest(), stream_get_meta_data($zipStream)['uri']);
            stream_copy_to_stream(fopen(stream_get_meta_data($zipStream)['uri'], 'r'), $outputStream);
            fclose($zipStream);
        });
        $response->headers->set('Content-Type', 'application/zip');
        $response->headers->set('Content-Disposition', HeaderUtils::makeDisposition(
            HeaderUtils::DISPOSITION_ATTACHMENT,
            $project->getRequest()->getName().'.zip',
            iconv('UTF-8', 'ASCII//TRANSLIT', $project->getRequest()->getName().'.zip')
        ));

        return $response;
    }

    /**
     * @Route("/api/download/experiment/{experiment}", name="downloadExperimentFile")
     */
    public function exportExperimentFile(Experiment $experiment, WebappWriterService $writerService): Response
    {
        $experimentFile = $writerService->createExperiment($experiment);

        return $this->createFileResponse($experimentFile, $experiment->getName());
    }

    /**
     * @Route("/api/download/platform/{platform}", name="downloadPlatformFile")
     */
    public function exportPlatformFile(Platform $platform, WebappWriterService $writerService): Response
    {
        $platformFile = $writerService->createPlatform($platform);

        return $this->createFileResponse($platformFile, $platform->getName());
    }

    /**
     * @Route("/api/download/platformWithData/{platform}", name="downloadPlatformWithDataFile")
     */
    public function exportPlatformWithDataFile(Platform $platform, WebappWriterService $writerService): Response
    {
        $platformFile = $writerService->createPlatformWithData($platform);

        return $this->createFileResponse($platformFile, $platform->getName());
    }

    /**
     * @Route("/api/download/variables/{site}", name="downloadVariablesFile")
     */
    public function exportVariablesFile(Site $site, Request $request, WebappWriterService $writerService): Response
    {
        $datas = $request->query->get('datas') ?? [];

        $variablesFile = $writerService->createVariableCollectionFile($site, $datas);

        return $this->createFileResponse($variablesFile, $site->getLabel().'-variables');
    }

    protected function createFileResponse(\SplFileObject $file, string $filename): BinaryFileResponse
    {
        $filenameWithExtension = $filename.'.'.$file->getExtension();
        $response = new BinaryFileResponse($file, 200, [], true, HeaderUtils::DISPOSITION_ATTACHMENT);
        $response->setContentDisposition(
            HeaderUtils::DISPOSITION_ATTACHMENT,
            $filenameWithExtension,
            iconv('UTF-8', 'ASCII//TRANSLIT', $filenameWithExtension)
        );
        $response->deleteFileAfterSend();

        return $response;
    }
}
